﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/migration/detail/migration_AsyncExecutionResource.h>

#include <mutex>

#include <nn/migration/detail/migration_Diagnosis.h>
#include <nn/os/os_MultipleWaitUtility.h>
#include <nn/os/os_ThreadApi.h>

namespace nn { namespace migration { namespace detail {

void AsyncExecutionResource::ThreadFunctionImpl() NN_NOEXCEPT
{
    while (!m_StopEvent.TryWait())
    {
        auto index = os::WaitAny(m_StopEvent.GetBase(), m_AttachEvent.GetBase());
        NN_UNUSED(index);

        if (m_AttachEvent.TryWait())
        {
            m_AttachEvent.Clear();

            std::lock_guard<os::Mutex> lock(m_Lock);
            if (m_Task._isAttached)
            {
                m_Task.Invoke();
                m_Task.Reset();
                m_DetachCondition.Signal();
            }
        }
    }
}

void AsyncExecutionResource::Attach(void(*f)(void*), void* data, os::SystemEventType* pEvent) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Lock);
    while (m_Task._isAttached)
    {
        m_DetachCondition.Wait(*m_Lock.GetBase());
    }
    m_Task.Attach(f, data, pEvent);
    m_AttachEvent.Signal();
}

bool AsyncExecutionResource::TryAttach(void(*f)(void*), void* data, os::SystemEventType* pEvent) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Lock);
    if (m_Task._isAttached)
    {
        return false;
    }
    m_Task.Attach(f, data, pEvent);
    m_AttachEvent.Signal();
    return true;
}

AsyncExecutionResource::AsyncExecutionResource(ThreadResource&& threadResource) NN_NOEXCEPT
    : m_StopEvent(os::EventClearMode_ManualClear)
    , m_AttachEvent(os::EventClearMode_ManualClear)
    , m_Lock(false)
    , m_Resource(std::move(threadResource))
{
    m_Task.Reset();
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(CreateThread(&m_Thread, ThreadFunction, this, m_Resource));
    os::StartThread(&m_Thread);
}

AsyncExecutionResource::~AsyncExecutionResource() NN_NOEXCEPT
{
    m_StopEvent.Signal();
    os::WaitThread(&m_Thread);
    os::DestroyThread(&m_Thread);
}

}}} // ~namespace nn::migration::detail
