﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <nn/nn_Common.h>
#include <nn/ldn/ldn_Ipv4Address.h>
#include <nn/ldn/detail/Debug/ldn_Assert.h>
#include <nn/ldn/detail/Debug/ldn_Log.h>
#include <nn/ldn/detail/TcpIp/ldn_TcpIpStackConfigurationImpl-os.horizon.h>
#include <nn/ldn/detail/Utility/ldn_ReverseByteOrder.h>

namespace nn { namespace ldn { namespace detail { namespace impl { namespace
{
    void AddEntryImpl(
        Ipv4Address* entries, int entryCount, Ipv4Address ip, MacAddress mac) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(entries);
        NN_SDK_ASSERT(0 < entryCount);
        NN_SDK_ASSERT_NOT_EQUAL(ip, ZeroIpv4Address);
        for (int i = 0; i < entryCount; ++i)
        {
            if (entries[i] == ZeroIpv4Address)
            {
                entries[i] = ip;
                nn::socket::InAddr ipAddr = { };
                ipAddr.S_addr = ConvertToNetworkByteOrder(ip.raw);
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::tsc::AddStaticArpEntry(
                    ipAddr, mac.raw, sizeof(mac)));
                return;
            }
        }
        NN_SDK_ASSERT(false, "Failed to add an arp entry");
    }

    void RemoveEntryImpl(Ipv4Address* entries, int entryCount, Ipv4Address ip) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(entries);
        NN_SDK_ASSERT(0 < entryCount);
        NN_SDK_ASSERT_NOT_EQUAL(ip, ZeroIpv4Address);
        for (int i = 0; i < entryCount; ++i)
        {
            if (entries[i] == ip)
            {
                entries[i] = ZeroIpv4Address;
                nn::socket::InAddr ipAddr = { };
                ipAddr.S_addr = ConvertToNetworkByteOrder(ip.raw);
                Result result = nn::tsc::RemoveArpEntry(ipAddr);
                if (result.IsFailure())
                {
                    NN_LDN_LOG_ERROR("failed to remove arp entry: tsc error\n");
                }
                return;
            }
        }
        NN_SDK_ASSERT(false, "failed to remove arp entry: not found\n");
    }

    void ClearEntryImpl(Ipv4Address* entries, int entryCount) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(entries);
        NN_SDK_ASSERT(0 < entryCount);
        for (int i = 0; i < entryCount; ++i)
        {
            if (entries[i] != ZeroIpv4Address)
            {
                nn::socket::InAddr ipAddr = { };
                ipAddr.S_addr = ConvertToNetworkByteOrder(entries[i].raw);
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::tsc::RemoveArpEntry(ipAddr));
                entries[i] = ZeroIpv4Address;
            }
        }
    }

}}}}} // namespace nn::ldn::detail::impl::<unnamed>

namespace nn { namespace ldn { namespace detail { namespace impl
{
    TcpIpStackConfigurationImpl::TcpIpStackConfigurationImpl() NN_NOEXCEPT
        : m_pTscContext(nullptr)
    {
        std::memset(m_ArpEntries, 0, sizeof(m_ArpEntries));
    }

    TcpIpStackConfigurationImpl::~TcpIpStackConfigurationImpl() NN_NOEXCEPT
    {
        if (m_pTscContext)
        {
            Down();
        }
    }

    void TcpIpStackConfigurationImpl::Startup(
        Ipv4Address address, Ipv4Address gateway, SubnetMask mask,
        const NetworkInterfaceProfile& profile) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_pTscContext == nullptr, "TCP/IP stack has already been initialized");
        NN_SDK_ASSERT_NOT_EQUAL(ZeroIpv4Address, address);
        NN_SDK_ASSERT_NOT_EQUAL(BroadcastIpv4Address, address);
        NN_SDK_ASSERT_NOT_EQUAL(ZeroIpv4Address, gateway);
        NN_SDK_ASSERT_NOT_EQUAL(BroadcastIpv4Address, gateway);

        // 成功するまで繰り返します。
        // TODO: ResultConfigurationInProgress が発生する原因の調査と繰り返しの除去
        Result result;
        for (;;)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::tsc::Initialize());
            nn::tsc::Ipv4Config config;
            std::memset(&config, 0, sizeof(config));
            config.method = nn::tsc::Ipv4ConfigMethod_Static;
            config.interfaceAddress.storage = ConvertToNetworkByteOrder(address.raw);
            config.subnetMask.storage = ConvertToNetworkByteOrder(mask.raw);
            config.defaultGateway.storage = ConvertToNetworkByteOrder(gateway.raw);
            m_pTscContext = new (&m_TscContextStorage) nn::tsc::Ipv4ConfigContext(
                profile.name, nn::os::EventClearMode_AutoClear);
            NN_SDK_ASSERT_NOT_NULL(m_pTscContext);
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_pTscContext->SetMtu(profile.mtu));
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_pTscContext->SetConfig(&config));
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_pTscContext->ApplyConfig(nn::tsc::IpApplyModeMask_None));
            m_pTscContext->GetEventPointer()->Wait();
            result = m_pTscContext->GetApplyResult();
            if (!nn::tsc::ResultConfigurationInProgress::Includes(result))
            {
                break;
            }
            m_pTscContext->NotifyInterfaceDown();
            m_pTscContext->CancelApplyConfig();
            m_pTscContext->~Ipv4ConfigContext();
            m_pTscContext = nullptr;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::tsc::Finalize());
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    void TcpIpStackConfigurationImpl::Down() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_pTscContext != nullptr, "TCP/IP stack has yet to be initialized");
        ClearEntry();
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_pTscContext->NotifyInterfaceDown());
        m_pTscContext->CancelApplyConfig();
        m_pTscContext->~Ipv4ConfigContext();
        m_pTscContext = nullptr;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::tsc::Finalize());
    }

    void TcpIpStackConfigurationImpl::AddEntry(Ipv4Address ip, MacAddress mac) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_pTscContext != nullptr, "TCP/IP stack has yet to be initialized");
        AddEntryImpl(m_ArpEntries, ArpEntryCountMax, ip, mac);
    }

    void TcpIpStackConfigurationImpl::RemoveEntry(Ipv4Address ip) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_pTscContext != nullptr, "TCP/IP stack has yet to be initialized");
        RemoveEntryImpl(m_ArpEntries, ArpEntryCountMax, ip);
    }

    void TcpIpStackConfigurationImpl::ClearEntry() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_pTscContext != nullptr, "TCP/IP stack has yet to be initialized");
        ClearEntryImpl(m_ArpEntries, ArpEntryCountMax);
    }

}}}} // namespace nn::ldn::detail
