﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/ldn/ldn_PrivateResult.h>
#include <nn/ldn/ldn_Result.h>
#include <nn/ldn/detail/Debug/ldn_Log.h>
#include <nn/ldn/detail/NetworkInterface/ldn_NintendoEthernet.h>
#include <nn/ldn/detail/NetworkInterface/ldn_FrameSender.h>
#include <nn/os/os_ThreadApi.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace ldn { namespace detail { namespace
{
    const size_t SendThreadStackSize = 4 * 1024;

}}}} // namespace nn::ldn::detail::<unnamed>

namespace nn { namespace ldn { namespace detail
{
    FrameSender::FrameSender() NN_NOEXCEPT
        : m_pNetworkInterface(nullptr),
          m_SendBuffer(nullptr),
          m_SendBufferSize(0),
          m_IsRunning(false),
          m_IsInitialized(false),
          m_EnqueueEvent(nn::os::EventClearMode_AutoClear),
          m_DequeueEvent(nn::os::EventClearMode_AutoClear),
          m_Mutex(false)
    {
    }

    FrameSender::~FrameSender() NN_NOEXCEPT
    {
        if (m_IsInitialized)
        {
            Finalize();
        }
    }

    size_t FrameSender::GetRequiredBufferSize(size_t frameSizeMax, int capacity) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(sizeof(NintendoEthernetHeader) <= frameSizeMax);
        NN_SDK_ASSERT(0 < capacity);
        return SendThreadStackSize +
               FrameQueue::GetRequiredBufferSize(frameSizeMax, capacity) + frameSizeMax;
    }

    void FrameSender::Initialize(
        INetworkInterface* pInterface, void* buffer, size_t bufferSize,
        size_t frameSizeMax, int capacity) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!m_IsInitialized, "Sender has already been initialized");
        NN_SDK_ASSERT_NOT_NULL(pInterface);
        NN_SDK_ASSERT_NOT_NULL(buffer);
        NN_SDK_ASSERT_ALIGNED(buffer, nn::os::ThreadStackAlignment);
        NN_SDK_ASSERT(GetRequiredBufferSize(frameSizeMax, capacity) <= bufferSize);
        std::memset(buffer, 0, bufferSize);
        m_pNetworkInterface = pInterface;
        m_SendBuffer = static_cast<Bit8*>(buffer) + SendThreadStackSize;
        m_SendBufferSize = frameSizeMax;
        m_Queue.Initialize(
            m_SendBuffer + frameSizeMax,
            bufferSize - SendThreadStackSize - frameSizeMax,
            frameSizeMax, capacity);
        m_IsInitialized = true;
        m_IsRunning = true;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
            &m_SendThread, SendThread, this,
            buffer, SendThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(ldn, Sender)));
        nn::os::SetThreadNamePointer(&m_SendThread, NN_SYSTEM_THREAD_NAME(ldn, Sender));
        nn::os::StartThread(&m_SendThread);
    }

    void FrameSender::Finalize() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsInitialized, "Sender has not been initialized yet");
        m_IsRunning = false;
        m_IsInitialized = false;
        m_EnqueueEvent.Signal();
        m_DequeueEvent.Signal();
        nn::os::WaitThread(&m_SendThread);
        nn::os::DestroyThread(&m_SendThread);
        m_Queue.Finalize();
        m_SendBufferSize = 0;
        m_SendBuffer = nullptr;
        m_pNetworkInterface = nullptr;
    }

    Result FrameSender::Send(
        const void* data, size_t dataSize, MacAddress dst, ProtocolId protocol) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsInitialized, "Sender has not been initialized yet");
        NN_SDK_ASSERT_NOT_NULL(data);
        NN_SDK_ASSERT(dataSize <= NintendoEthernetFramePayloadSizeMax);
        NN_SDK_ASSERT_NOT_EQUAL(dst, ZeroMacAddress);
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        size_t payloadSizeMax = m_SendBufferSize - NintendoEthernetFrameSizeMin;
        size_t payloadSize = std::min(dataSize, payloadSizeMax);
        size_t frameSize = NintendoEthernetFrameSizeMin + payloadSize;

        // MAC アドレスを取得します。
        NetworkInterfaceProfile profile;
        m_pNetworkInterface->GetNetworkInterfaceProfile(&profile);

        // 送信キューに空きがでるまで待機します。
        while (m_IsRunning)
        {
            // 送信データを生成します。
            auto& frame = *reinterpret_cast<NintendoEthernetFrame*>(m_SendBuffer);
            frame.ethernetHeader.dst = dst;
            frame.ethernetHeader.src = profile.physicalAddress;
            frame.ethernetHeader.ethertype = OuiExtendedEthertype;
            frame.ouiExtendedEthernetHeader.oui = NintendoOui;
            frame.nintendoEthernetHeader.protocol = protocol;
            frame.nintendoEthernetHeader._reserved = 0;
            std::memcpy(frame.data, data, payloadSize);

            // キューに登録します。
            if (m_Queue.TryEnqueue(&frame, frameSize))
            {
                m_EnqueueEvent.Signal();
                nn::os::YieldThread();
                NN_RESULT_SUCCESS;
            }

            // キューへの登録に失敗した場合は空きができるまで待機します。
            m_DequeueEvent.Wait();
        }

        return ResultInvalidState();
    }

    void FrameSender::SendThread(void* pArg) NN_NOEXCEPT
    {
        NN_LDN_LOG_DEBUG("SendThread: started\n");
        auto& sender = *static_cast<FrameSender*>(pArg);
        for (;;)
        {
            // 送信対象となるフレームが存在しない場合は待機します。
            auto& queue = sender.m_Queue;
            while (queue.IsEmpty() && sender.m_IsRunning)
            {
                sender.m_EnqueueEvent.Wait();
            }

            // Finalize が実行された場合はスレッドを終了します。
            if (!sender.m_IsRunning)
            {
                break;
            }

            // キューの先頭からフレームを取り出して送信します。
            size_t dataSize;
            const auto& data = queue.GetFront(&dataSize);
            auto result = sender.m_pNetworkInterface->Send(data, dataSize);
            queue.Dequeue();

            // キューに空きができたことを通知します。
            sender.m_DequeueEvent.Signal();

            // ネットワークが破棄されるか切断された場合には送信スレッドを終了します。
            if (ResultInvalidState::Includes(result))
            {
                sender.m_IsRunning = false;
                break;
            }
            else if (ResultSendFailed::Includes(result))
            {
                // 送信に失敗しても救済措置はありません。フレーム損失と同様です。
                // 原則として、送信失敗が発生しないように設計します。
            }
            else if (ResultTxQueueIsFull::Includes(result))
            {
                // 送信に失敗しても救済措置はありません。フレーム損失と同様です。
                // 原則として、送信失敗が発生しないように設計します。
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
        }
        NN_LDN_LOG_DEBUG("SendThread: finished\n");
    }

}}} // namespace nn::ldn::detail
