﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/ldn/detail/Authentication/ldn_AuthenticationDataBuilder.h>
#include <nn/ldn/detail/Debug/ldn_Log.h>
#include <nn/ldn/detail/Utility/ldn_ReverseByteOrder.h>

namespace nn { namespace ldn { namespace detail
{
    AuthenticationServerDataBuilder::AuthenticationServerDataBuilder() NN_NOEXCEPT
    {
        std::memset(&m_Data, 0, sizeof(m_Data));
    }

    void AuthenticationServerDataBuilder::SetLdnNetworkInfo(const LdnNetworkInfo& ldn) NN_NOEXCEPT
    {
        NN_UNUSED(ldn);
    }

    const AuthenticationServerData& AuthenticationServerDataBuilder::GetData() const NN_NOEXCEPT
    {
        return m_Data;
    }

    AuthenticationClientDataBuilder::AuthenticationClientDataBuilder() NN_NOEXCEPT
    {
        std::memset(&m_Data, 0, sizeof(m_Data));
    }

    void AuthenticationClientDataBuilder::SetUserConfig(const UserConfig& user) NN_NOEXCEPT
    {
        std::strncpy(m_Data.userName, user.userName, UserNameBytesMax);
    }

    void AuthenticationClientDataBuilder::SetLocalCommunicationVersion(int version) NN_NOEXCEPT
    {
        m_Data.localCommunicationVersion = ConvertToNetworkByteOrder(
            static_cast<int16_t>(version));
    }

    const AuthenticationClientData& AuthenticationClientDataBuilder::GetData() const NN_NOEXCEPT
    {
        return m_Data;
    }

    AuthenticationClientDataParser::AuthenticationClientDataParser(
        const void* data, size_t dataSize) NN_NOEXCEPT
        : m_IsValid(sizeof(AuthenticationClientData) <= dataSize)
    {
        NN_SDK_ASSERT_NOT_NULL(data);
        NN_SDK_ASSERT_ALIGNED(data, NN_ALIGNOF(AuthenticationClientData));
        auto& clientData = *static_cast<const AuthenticationClientData*>(data);
        std::memset(&m_UserConfig, 0, sizeof(UserConfig));
        if (m_IsValid)
        {
            std::memcpy(m_UserConfig.userName, clientData.userName, UserNameBytesMax);
            m_LocalCommunicationVersion = ConvertToHostByteOrder(
                clientData.localCommunicationVersion);
        }
    }

    const UserConfig& AuthenticationClientDataParser::GetUserConfig() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsValid);
        return m_UserConfig;
    }

    int AuthenticationClientDataParser::GetLocalCommunicationVersion() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsValid);
        return m_LocalCommunicationVersion;
    }

    bool AuthenticationClientDataParser::IsValid() const NN_NOEXCEPT
    {
        return m_IsValid;
    }

}}} // namespace nn::ldn::detail
