﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <algorithm>
#include <nn/nn_Common.h>
#include <nn/ldn/detail/Advertise/ldn_Advertise.h>
#include <nn/ldn/detail/Advertise/ldn_PlainAdvertiseImpl.h>
#include <nn/ldn/detail/Debug/ldn_Log.h>
#include <nn/ldn/detail/Utility/ldn_Random.h>
#include <nn/ldn/detail/Utility/ldn_ReverseByteOrder.h>

namespace nn { namespace ldn { namespace detail { namespace impl
{
    PlainAdvertiseBuilderImpl::PlainAdvertiseBuilderImpl(
       AdvertiseFormat format, CalculateDigest digest) NN_NOEXCEPT
       : m_CalculateDigest(digest),
         m_HasNetworkId(false),
         m_HasVersion(false),
         m_Format(static_cast<Bit8>(format))
    {
        NN_SDK_ASSERT_NOT_EQUAL(m_Format, AdvertiseFormat_None);
        NN_SDK_ASSERT_NOT_NULL(digest);
        m_Counter = Random<uint32_t>();
    }

    PlainAdvertiseBuilderImpl::~PlainAdvertiseBuilderImpl() NN_NOEXCEPT
    {
    }

    void PlainAdvertiseBuilderImpl::SetNetworkId(const NetworkId& networkId) NN_NOEXCEPT
    {
        m_NetworkId = networkId;
        m_HasNetworkId = true;
    }

    void PlainAdvertiseBuilderImpl::SetVersion(Version version) NN_NOEXCEPT
    {
        m_Version = version;
        m_HasVersion = true;
    }

    void PlainAdvertiseBuilderImpl::Build(
        void* pOutAdvertise, size_t* pOutSize, size_t bufferSize,
        const void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutAdvertise);
        NN_SDK_ASSERT_ALIGNED(pOutAdvertise, NN_ALIGNOF(Advertise));
        NN_SDK_ASSERT_NOT_NULL(pOutSize);
        NN_SDK_ASSERT(AdvertiseSizeMin + dataSize <= bufferSize);
        NN_SDK_ASSERT(data != nullptr || dataSize == 0);
        NN_SDK_ASSERT(dataSize <= AdvertiseBodySizeMax);
        NN_SDK_ASSERT(m_HasNetworkId);
        NN_SDK_ASSERT(m_HasVersion);
        NN_UNUSED(bufferSize);

        // データが変更されたことをクライアントに通知するためカウンタをインクリメントします。
        ++m_Counter;

        // ヘッダを生成します。
        auto& advertise = *static_cast<Advertise*>(pOutAdvertise);
        auto& header = advertise.header;
        header.networkId = ConvertToNetworkByteOrder(m_NetworkId);
        header.version = m_Version;
        header.format = m_Format;
        header.length = ConvertToNetworkByteOrder(static_cast<uint16_t>(dataSize));
        header.counter = ConvertToNetworkByteOrder(m_Counter);

        // データを書き込みます。
        if (0 < dataSize)
        {
            std::memcpy(advertise.body, data, std::min(dataSize, AdvertiseBodySizeMax));
        }

        // Advertise 全体のサイズを計算します。
        size_t advertiseSize = sizeof(AdvertiseHeader) + AdvertiseDigestSize + dataSize;
        *pOutSize = advertiseSize;

        // メッセージダイジェストを計算します。
        Bit8 digest[AdvertiseDigestSize];
        size_t digestSize;
        std::memset(advertise.digest, 0, AdvertiseDigestSize);
        m_CalculateDigest(digest, &digestSize, sizeof(digest), &advertise, advertiseSize);
        std::memcpy(advertise.digest, digest, digestSize);
        NN_SDK_ASSERT_EQUAL(digestSize, sizeof(digest));
    }

    size_t PlainAdvertiseParserImpl::GetRequiredBufferSize() NN_NOEXCEPT
    {
        return sizeof(Advertise) * 2;
    }

    PlainAdvertiseParserImpl::PlainAdvertiseParserImpl(
       void* buffer, size_t bufferSize,
       AdvertiseFormat format, CalculateDigest digest) NN_NOEXCEPT
       : m_CalculateDigest(digest),
         m_pAdvertise(static_cast<Advertise*>(buffer)),
         m_pPreviousAdvertise(m_pAdvertise + 1),
         m_Format(format),
         m_HasParsedData(false)
    {
        NN_SDK_ASSERT_NOT_NULL(buffer);
        NN_SDK_ASSERT_ALIGNED(buffer, NN_ALIGNOF(Advertise));
        NN_SDK_ASSERT(GetRequiredBufferSize() <= bufferSize);
        NN_SDK_ASSERT_NOT_EQUAL(m_Format, AdvertiseFormat_None);
        NN_SDK_ASSERT_NOT_NULL(digest);
        NN_UNUSED(bufferSize);
    }

    PlainAdvertiseParserImpl::~PlainAdvertiseParserImpl() NN_NOEXCEPT
    {
    }

    AdvertiseParserResult PlainAdvertiseParserImpl::VerifyHeader(
        const void* data, size_t dataSize) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(data);
        NN_SDK_ASSERT(0 < dataSize);

        // サイズが異常な Advertise は解析できません。
        if (dataSize < AdvertiseSizeMin || AdvertiseSizeMax < dataSize)
        {
            NN_LDN_LOG_WARN("failed to parse advertise: invalid size (%u Bytes)\n", dataSize);
            return AdvertiseParserResult_InvalidAdvertise;
        }

        // Advertise Header の有効性を検証します。
        auto&  advertise = *static_cast<const Advertise*>(data);
        auto&  header    = advertise.header;
        auto   bodySize  = ConvertToHostByteOrder(header.length);
        auto   totalSize = bodySize + AdvertiseSizeMin;
        if (header.format != m_Format)
        {
            // この Parser で解析できるフォーマットになっていません。
            NN_LDN_LOG_WARN("failed to parse advertise: invalid format %u\n",
                header.format);
            return AdvertiseParserResult_InvalidAdvertise;
        }
        else if (AdvertiseBodySizeMax < bodySize)
        {
            // Advertise のペイロードが最大長を超えています。
            NN_LDN_LOG_WARN("failed to parse advertise: too large body (%u Bytes)\n", bodySize);
            return AdvertiseParserResult_InvalidAdvertise;
        }
        #if defined(NN_LDN_BUILD_CONFIG_INCOMPLETE_ADVERTISE_FORBIDDEN)
        else if (totalSize != dataSize)
        {
            // 受信サイズとヘッダに格納された Advertise 長が一致していません。
            NN_LDN_LOG_WARN("failed to parse advertise: invalid body length\n");
            return AdvertiseParserResult_InvalidAdvertise;
        }
        #endif

        // プロトコル・バージョンを検証します。
        if (!IsCompatibleVersion(header.version))
        {
            // 互換性のないプロトコルバージョンです。
            NN_LDN_LOG_INFO("failed to parse advertise: incompatible version %d.%d\n",
                GetMajorVersion(header.version), GetMinorVersion(header.version));
            return AdvertiseParserResult_IncompatibleVersion;
        }

        // 前回受信した Advertise との間に矛盾がないことを確認します。
        if (m_HasParsedData)
        {
            NetworkId networkId = ConvertToHostByteOrder(header.networkId);
            if (m_pPreviousAdvertise->header.networkId != networkId ||
                m_pPreviousAdvertise->header.version != header.version)
            {
                // 前回の Advertise と不一致があります。
                NN_LDN_LOG_WARN("failed to parse advertise: inconsistent update\n");
                return AdvertiseParserResult_InconsistentUpdate;
            }

            const uint32_t counter = ConvertToHostByteOrder(header.counter);
            const uint32_t diff = counter - m_pPreviousAdvertise->header.counter;
            if (AdvertiseRejectCounterThreshold <= diff)
            {
                // 古い Advertise の再送、あるいは Counter 値の不一致です。
                NN_LDN_LOG_WARN("failed to parse advertise: invalid counter\n");
                return AdvertiseParserResult_InconsistentUpdate;
            }
            else if (diff == 0)
            {
                // 前回解析した Advertise と一致しています。
                return AdvertiseParserResult_NotUpdated;
            }
        }
        return AdvertiseParserResult_Success;
    }

    AdvertiseParserResult PlainAdvertiseParserImpl::VerifySign(
        const void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(data);
        NN_SDK_ASSERT_MINMAX(dataSize, AdvertiseSizeMin, AdvertiseSizeMax);

        // メッセージダイジェスト部分を 00 00 .. 00 に書き換える必要があるためコピーします。
        std::memcpy(m_pAdvertise, data, dataSize);
        auto&  advertise    = *m_pAdvertise;
        auto&  header       = advertise.header;
        size_t bodySize     = ConvertToHostByteOrder(header.length);
        size_t totalSize    = AdvertiseSizeMin + bodySize;
        NN_SDK_ASSERT_EQUAL(totalSize, dataSize);

        // メッセージダイジェストを計算します。
        #if defined(NN_LDN_BUILD_CONFIG_INCOMPLETE_ADVERTISE_FORBIDDEN)
        Bit8   digest[AdvertiseDigestSize];
        size_t digestSize;
        std::memcpy(digest, advertise.digest, sizeof(digest));
        std::memset(advertise.digest, 0, AdvertiseDigestSize);
        m_CalculateDigest(advertise.digest, &digestSize, sizeof(digest), &advertise, totalSize);
        if (std::memcmp(digest, advertise.digest, sizeof(digest)) != 0)
        {
            // メッセージダイジェストが一致しません。
            NN_LDN_LOG_WARN("failed to parse advertise: signature mismatch\n");
            return AdvertiseParserResult_SignVerificationError;
        }
        NN_SDK_ASSERT_EQUAL(digestSize, AdvertiseDigestSize);
        #elif defined(NN_LDN_BUILD_CONFIG_INCOMPLETE_ADVERTISE_PERMITTED)
        NN_UNUSED(m_CalculateDigest);
        #endif

        return AdvertiseParserResult_Success;
    }

    void PlainAdvertiseParserImpl::Parse(
        const void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(data);
        NN_SDK_ASSERT(0 < dataSize);

        // Advertise を保存します。
        std::memcpy(m_pPreviousAdvertise, data, dataSize);

        // バイトオーダを変換しておきます。
        m_pPreviousAdvertise->header.networkId = ConvertToHostByteOrder(
            m_pPreviousAdvertise->header.networkId);
        m_pPreviousAdvertise->header.counter = ConvertToHostByteOrder(
            m_pPreviousAdvertise->header.counter);
        m_pPreviousAdvertise->header.length = ConvertToHostByteOrder(
            m_pPreviousAdvertise->header.length);
        m_HasParsedData = true;
    }

    const NetworkId PlainAdvertiseParserImpl::GetNetworkId() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_HasParsedData);
        return m_pPreviousAdvertise->header.networkId;
    }

    const Version PlainAdvertiseParserImpl::GetVersion() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_HasParsedData);
        return m_pPreviousAdvertise->header.version;
    }

    size_t PlainAdvertiseParserImpl::GetDataSize() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_HasParsedData);
        return m_pPreviousAdvertise->header.length;
    }

    const void* PlainAdvertiseParserImpl::GetData() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_HasParsedData);
        return m_pPreviousAdvertise->body;
    }

}}}} // namespace nn::ldn::detail::impl
