﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <new>
#include <nn/fs/fs_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/lcs/lcs_Config.h>
#include <nn/lcs/lcs_DebugApi.h>
#include <nn/lcs/lcs_Result.h>
#include <nn/lcs/lcs_Result.private.h>
#include <nn/lcs/lcs_Types.h>
#include <nn/lcs/detail/lcs_ApplicationShareApi.h>
#include <nn/lcs/detail/lcs_Definition.h>
#include <nn/lcs/detail/lcs_Host.h>
#include <nn/lcs/detail/lcs_NetworkApi.h>
#include <nn/lcs/detail/lcs_Socket.h>
#include <nn/lcs/detail/lcs_SystemShareApi.h>
#include <nn/lcs/detail/lcs_Util.h>
#include <nn/lcs/detail/Advertise/lcs_AdvertiseUtility.h>
#include <nn/lcs/detail/Debug/lcs_Log.h>
#include <nn/lcs/detail/Packet/lcs_Packet.h>
#include <nn/lcs/detail/Packet/lcs_PacketManager.h>
#include <nn/lcs/detail/Packet/lcs_PacketReceiver.h>
#include <nn/lcs/detail/Packet/lcs_PacketUtilty.h>
#include <nn/ldn.h>
#include <nn/nim/nim_Result.h>
#include <nn/ns/ns_Result.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/socket.h>

namespace nn { namespace lcs { namespace detail
{

    namespace
    {
        nn::os::Mutex g_Mutex(false);
    }

    Host::Host() NN_NOEXCEPT
    {
        ClearMemory();
    };

    void Host::ClearMemory() NN_NOEXCEPT
    {
        m_pResource = nullptr;

        m_Time = 0;

        m_IsOpened = false;
        m_IsAccepted = false;
        m_IsShareCompleted = false;
        m_IsNeedReboot = false;
        m_IsContentShareSucceeded = false;

        m_Socket = 0;
        m_MyAddress = 0;
        m_Policy = 0;

        m_Role = TransferRole_None;
        m_NodeIndex = 0;

        m_RequiredSize = 0;

        m_IsSender = false;
        m_IsWaitReceiver = false;
        m_TotalSendSize = 0;
    }

    void Host::ChangeState(State state) NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("ChangeState / %d -> %d\n", m_pResource->state.GetState(), state);

        if (m_pResource->state.GetState() == State_Completed)
        {
            return;
        }

        // 同じ状態なら変更なし
        if (m_pResource->state.GetState() == state)
        {
            return;
        }

        // 失敗するなら Client を切断
        if (state == State_ContentsShareFailed && IsNetworkAvailable())
        {
            NN_LCS_LOG_DEBUG("SendRejectToAll\n");
            SendRejectToAll(RejectReason_NodeLeaved);
        }
        // 完了なら全員の切断を待つ
        else if (state == State_Completed)
        {
            m_Role = TransferRole_None;

            // NodeManager にない IP の端末は切断する
            DisconnectUnlistedClinet();

            // 500ms 毎に ldn のノードを見に行く
            int waitCount = WaitClientDisconnectTimeoutSeconds * 1000 / WaitDisconnectIntervalMilliSeconds;
            for (int i = 0; i < waitCount; ++i)
            {
                NN_LCS_LOG_DEBUG("Wait disconnect %d / %d\n", i, waitCount);

                // Host が終了されていたら Break
                if (!m_IsOpened)
                {
                    break;
                }

                // ldn での接続端末を確認
                int connectedNodesCount = 0;
                if (GetConnectedNodesCount(&connectedNodesCount).IsFailure() ||
                    connectedNodesCount == 1)
                {
                    // ldn で接続している端末の取得に失敗したか、自分しかいなくなっていたら Break
                    break;
                }
                // 端末が残っているならスリープで待つ
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(WaitDisconnectIntervalMilliSeconds));
            }

            if (m_pResource->contentsShareFailureReason != ContentsShareFailureReason_None)
            {
                state = State_ContentsShareFailed;
            }
        }

        if (m_IsOpened)
        {
            m_pResource->state.SetState(state);
            m_pResource->stateEvent.SignalEvent();
        }
    }

    Result Host::InitializeHost() NN_NOEXCEPT
    {
        m_IsOpened = true;

        nn::os::InitializeEvent(&m_StartTransferEvent, false, nn::os::EventClearMode_AutoClear);
        nn::os::InitializeEvent(&m_RecvResPacketEvent, false, nn::os::EventClearMode_ManualClear);
        nn::os::InitializeEvent(&m_ResumeEvent, false, nn::os::EventClearMode_AutoClear);
        nn::os::InitializeEvent(&m_ClientStateChangeEvent, false, nn::os::EventClearMode_AutoClear);
        nn::os::InitializeEvent(&m_CancelEvent, false, nn::os::EventClearMode_ManualClear);

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::os::CreateThread(&m_pResource->monitorThread, this->MonitorThreadEntry, this,
                m_pResource->monitorThreadStack, sizeof(m_pResource->monitorThreadStack),
                m_pResource->monitorThreadPriority));
        nn::os::StartThread(&m_pResource->monitorThread);

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::os::CreateThread(&m_pResource->comThread, this->ComThreadEntry, this,
                m_pResource->comThreadStack, sizeof(m_pResource->comThreadStack),
                m_pResource->comThreadPriority));
        nn::os::StartThread(&m_pResource->comThread);

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::os::CreateThread(&m_pResource->shareThread, this->ShareThreadEntry, this,
                m_pResource->shareThreadStack, sizeof(m_pResource->shareThreadStack),
                m_pResource->shareThreadPriority));
        nn::os::StartThread(&m_pResource->shareThread);

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::os::CreateThread(&m_pResource->timeoutThread, this->TimeoutThreadEntry, this,
                m_pResource->timeoutThreadStack, sizeof(m_pResource->timeoutThreadStack),
                m_pResource->timeoutThreadPriority));
        nn::os::StartThread(&m_pResource->timeoutThread);

        m_pResource->contentsShareFailureReason = ContentsShareFailureReason_None;
        m_pResource->suspendedReason = SuspendedReason_None;

        NN_RESULT_SUCCESS;
    }

    void Host::FinalizeHost() NN_NOEXCEPT
    {
        if (m_IsOpened)
        {
            m_IsOpened = false;

            nn::os::SignalEvent(&m_StartTransferEvent);
            nn::os::SignalEvent(&m_RecvResPacketEvent);
            nn::os::SignalEvent(&m_ResumeEvent);
            nn::os::SignalEvent(&m_ClientStateChangeEvent);
            nn::os::SignalEvent(&m_CancelEvent);

            nn::os::WaitThread(&m_pResource->monitorThread);
            nn::os::WaitThread(&m_pResource->comThread);
            nn::os::WaitThread(&m_pResource->shareThread);
            nn::os::WaitThread(&m_pResource->timeoutThread);

            nn::os::DestroyThread(&m_pResource->monitorThread);
            nn::os::DestroyThread(&m_pResource->comThread);
            nn::os::DestroyThread(&m_pResource->shareThread);
            nn::os::DestroyThread(&m_pResource->timeoutThread);

            nn::os::FinalizeEvent(&m_StartTransferEvent);
            nn::os::FinalizeEvent(&m_RecvResPacketEvent);
            nn::os::FinalizeEvent(&m_ResumeEvent);
            nn::os::FinalizeEvent(&m_ClientStateChangeEvent);
            nn::os::FinalizeEvent(&m_CancelEvent);

            m_NodeManager.Finalize();
        }

        ClearMemory();
    }

    Result Host::InitializeSocket(int retryCountMax) NN_NOEXCEPT
    {
        int retryCount = 0;
        while (retryCount < retryCountMax)
        {
            if (SocketOpen(&m_Socket) == 0)
            {
                break;
            }
            retryCount++;
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(SocketInitializeRetryIntervalMilliSecond));
        }
        if (retryCount == retryCountMax)
        {
            NN_LCS_LOG_DEBUG("InitializeSocket Error!\n");
            return ResultSocketOpenError();
        }
        if (nn::socket::Listen(m_Socket, ClientCountMax) != 0)
        {
            NN_LCS_LOG_WARN("Failed to start listen: %d\n", nn::socket::GetLastError());
            SocketClose(&m_Socket);
            return ResultCommunicationError();
        }
        if (nn::socket::Fcntl(m_Socket, nn::socket::FcntlCommand::F_SetFl, static_cast<int>(nn::socket::FcntlFlag::O_NonBlock)) != 0)
        {
            NN_LCS_LOG_WARN("Failed to set socket non-blocking: %d\n", nn::socket::GetLastError());
            SocketClose(&m_Socket);
            return ResultSocketFcntlError();
        }
        NN_RESULT_SUCCESS;
    }

    void Host::FinalizeSocket()  NN_NOEXCEPT
    {
        SocketClose(&m_Socket);
    }

    Result Host::Open(LcsResources* resource) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(resource);

        m_pResource = resource;
        m_IsAccepted = true;
        m_Policy = m_pResource->sessionSettings.contentsShareVersionPolicy;

        // 本体を持っているかと、パッチが適用中で配信できないとかがないか確認と、
        // 指定された ID の AppDeliveryInfo がとれるか確認
        for (int i = 0; i < m_pResource->sessionSettings.applicationCount; ++i)
        {
            NN_RESULT_DO(CanGetApplicationInfo(m_pResource->sessionSettings.applications[i].value));
        }

        // ns から情報取ってきてアドバタイズデータに設定
        NN_RESULT_DO(m_AdvertiseManager.MakeAdvertiseData(
            m_pResource->sessionSettings, m_pResource->version, m_pResource->userName,
            m_pResource->appControlDataBuffer, AppControlDataSize));
        NN_RESULT_DO(m_AdvertiseManager.UpdateAdvertiseData());

        // ネットワークの開始
        NN_RESULT_DO(CreateNetwork(
            m_pResource->localCommunicationId, m_pResource->sessionSettings.nodeCountMax, m_pResource->version));
        NN_RESULT_DO(GetAccessPointIpv4Address(&m_pResource->hostAddress));

        NN_RESULT_DO(GetMyIpv4Address(&m_MyAddress));
        NN_LCS_LOG_DEBUG("My IpAddress : %x\n\n", m_MyAddress);

        NN_RESULT_DO(InitializeNodeManager(m_pResource->sessionSettings));

        NN_RESULT_DO(InitializeSocket(1));

        InitializeHost();

        GetSessionInfoFromNetwork(&m_pResource->sessionInfo);

        UpdateNodesInfo();

        NN_RESULT_SUCCESS;
    }

    Result Host::Reopen(LcsResources* resource) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(resource);

        NN_LCS_LOG_DEBUG("Reopen\n");

        m_pResource = resource;
        m_IsAccepted = false;
        m_Policy = m_pResource->sessionSettings.contentsShareVersionPolicy;

        m_AdvertiseManager.SetAdvertiseData(m_pResource->advertiseData);
        m_AdvertiseManager.SetSessionSetting(SessionAttribute_Limited);
        NN_RESULT_DO(m_AdvertiseManager.UpdateAdvertiseData());

        AddressEntry entry[NodeCountMax] = {};
        for (int i = 0; i < m_pResource->migrationInfo.nodeCount; ++i)
        {
            ::std::memcpy(&entry[i],
               &m_pResource->migrationInfo.nodeEntryInfo[i].nodeAddressEntry, sizeof(AddressEntry));
        }

        NN_RESULT_DO(CreateNetworkPrivate(m_pResource->migrationInfo.newNetworkSetting,
            entry, m_pResource->migrationInfo.nodeCount));

        NN_RESULT_DO(GetMyIpv4Address(&m_MyAddress));
        NN_LCS_LOG_DEBUG("My IpAddress : %x\n\n", m_MyAddress);

        NodeDetailInfo detailInfo = {};
        m_NodeManager.Initialize(m_pResource->migrationInfo.nodeCount);

        // 前のセッションから引き継いだ端末情報を追加(IP/Mac/UserName)
        for (int i = 0; i < m_pResource->migrationInfo.nodeCount; ++i)
        {
            NN_ABORT_UNLESS(m_NodeManager.AddNode(m_pResource->migrationInfo.nodeEntryInfo[i].nodeAddressEntry.ipv4Address,
                m_pResource->migrationInfo.nodeEntryInfo[i].nodeAddressEntry.macAddress,
                m_pResource->migrationInfo.nodeEntryInfo[i].index) != 0);
            detailInfo.nodeInfo.index = m_pResource->migrationInfo.nodeEntryInfo[i].index;
            ::std::strncpy(detailInfo.nodeInfo.userName,
                m_pResource->migrationInfo.nodeEntryInfo[i].userName, UserNameBytesMax);
            detailInfo.nodeInfo.userName[UserNameBytesMax] = 0;
            detailInfo.state = NodeInnerState_Rejoining;
            NN_ABORT_UNLESS(m_NodeManager.AddNodeDetail(detailInfo));
        }

        // 自分の情報を追加
        detailInfo.nodeInfo.index = m_NodeManager.GetNodeIndex(m_MyAddress);
        ::std::strncpy(detailInfo.nodeInfo.userName, m_pResource->userName, UserNameBytesMax);
        detailInfo.nodeInfo.userName[UserNameBytesMax] = 0;
        detailInfo.version = m_pResource->version;
        GetSystemDeliveryInfo(&detailInfo.sysInfo);
        for (int i = 0; i < m_pResource->sessionInfo.contentsCount; ++i)
        {
            NN_RESULT_DO(GetApplicationDetailInfo(&detailInfo.shareApp[i].info.detailInfo,
                m_pResource->appControlDataBuffer, AppControlDataSize, m_pResource->sessionInfo.contents[i].applicationId.value));
            NN_RESULT_DO(GetApplicationDeliveryInfo(
                &detailInfo.shareApp[i].info.deliveryInfoCount, detailInfo.shareApp[i].info.deliveryInfo,
                ApplicationDeliveryInfoCountMax, m_pResource->sessionInfo.contents[i].applicationId.value));
        }
        detailInfo.state = NodeInnerState_Connected;
        NN_ABORT_UNLESS(m_NodeManager.AddNodeDetail(detailInfo));

        NN_RESULT_DO(InitializeSocket(SocketInitializeRetryCountMax));

        InitializeHost();

        UpdateSessionInfo();

        NN_RESULT_SUCCESS;
    }

    void Host::WaitMigrationClients() NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("WaitMigrationClients\n");

        int count = static_cast<int>(WaitRejoinTimeout.GetMilliSeconds() / RetryInterval.GetMilliSeconds());
        for (int i = 0; i < count; ++i)
        {
            {
                // Close されないようにロック
                ::std::lock_guard<nn::os::Mutex> guard(g_Mutex);

                if (!m_IsOpened)
                {
                    NN_LCS_LOG_DEBUG("Close WaitMigrationClients\n");
                    return;
                }

                if (m_NodeManager.IsInnerStateAll(NodeInnerState_Connected))
                {
                    NN_LCS_LOG_DEBUG("All Client is Rejoin\n");
                    break;
                }
#ifdef NN_LCS_BUILD_CONFIG_DEBUG_LOG_LEVEL_DEBUG
                else
                {
                    int nodeCount = 0;
                    NodeInfo info[NodeCountMax] = {};
                    m_NodeManager.GetNodeInfo(&nodeCount, info, NodeCountMax);
                    for (int i = 0; i < nodeCount; ++i)
                    {
                        if (!m_NodeManager.IsInnerState(info[i].index, NodeInnerState_Connected))
                        {
                            NN_LCS_LOG_DEBUG("Wait : %x\n", info[i].index);
                        }
                    }
                }
#endif // NN_LCS_BUILD_CONFIG_DEBUG_LOG_LEVEL_DEBUG
            }

            nn::os::SleepThread(RetryInterval);
        }

        // Close されないようにロック
        ::std::lock_guard<nn::os::Mutex> guard(g_Mutex);

        if (m_IsOpened)
        {
            UpdateNodesInfo();

            bool isDown = false;
            UpdateShareAppInfo(&isDown);

            Start();

            if (m_NodeManager.GetNodeCount() == 1 || isDown)
            {
                m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
                ChangeState(State_ContentsShareFailed);
            }
        }
    }


    void Host::Close() NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("Close\n");

        // Migration と衝突しないようにロック
        ::std::lock_guard<nn::os::Mutex> guard(g_Mutex);
        NN_LCS_LOG_DEBUG("Close - GetMutex\n");

        if (m_IsOpened)
        {
            m_Role = TransferRole_None;

            if (m_pResource->state.GetState() == State_Transferring &&
                !m_NodeManager.IsInnerState(m_NodeManager.GetNodeIndex(m_MyAddress), NodeInnerState_Suspending))
            {
                NN_LCS_LOG_DEBUG("SendRejectToAll\n");
                SendRejectToAll(RejectReason_NodeLeaved);
            }
            FinalizeHost();
            FinalizeSocket();
        }
    }

    bool Host::IsInitialized() NN_NOEXCEPT
    {
        return m_IsOpened;
    }

    void Host::UpdateSessionInfo() NN_NOEXCEPT
    {
        GetSessionInfoFromNetwork(&m_pResource->sessionInfo);
    }

    void Host::UpdateNodesInfo() NN_NOEXCEPT
    {
        int nodeCount = 0;
        NodeInfo nodeInfo[NodeCountMax] = {};
        m_NodeManager.GetConnectedNodeInfo(&nodeCount, nodeInfo, NodeCountMax);

        int count = 0;
        nn::os::LockMutex(&m_pResource->mutex);
        for (int i = 0; i < nodeCount; ++i)
        {
            ::std::memcpy(&(m_pResource->nodeInfo[count]), &nodeInfo[i], sizeof(NodeInfo));
            ++count;
            NN_ABORT_UNLESS_LESS_EQUAL(count, NodeCountMax);
        }
        m_pResource->nodeInfoCount = count;
        nn::os::UnlockMutex(&m_pResource->mutex);

        UpdateAdvertiseInfo();
        SendSessionInfoToAll();
    }

    void Host::UpdateRequiredStorageSize() NN_NOEXCEPT
    {
        size_t size = 0;
        ApplicationDeliveryInfo appDeliveryInfo[ApplicationDeliveryInfoCountMax] = {};
        int appDeliveryInfoCount = 0;

        for (int i = 0; i < m_pResource->sessionInfo.contentsCount; ++i)
        {
            // 自分の情報を取得
            if (GetApplicationDeliveryInfo(
                &appDeliveryInfoCount, appDeliveryInfo, ApplicationDeliveryInfoCountMax,
                m_pResource->sessionInfo.contents[i].applicationId.value).IsFailure())
            {
                size += static_cast<size_t>(m_pResource->shareAppInfo[i].detailInfo.requiredSize);
                continue;
            }
            // 自分がパッチを持っていなければ必要サイズをそのまま加える
            bool hasPatch = false;
            NN_ABORT_UNLESS_RESULT_SUCCESS(HasAllContentsToDeliver(&hasPatch, appDeliveryInfo, appDeliveryInfoCount));
            if (!hasPatch)
            {
                size += static_cast<size_t>(m_pResource->shareAppInfo[i].detailInfo.requiredSize);
                continue;
            }
            // 自分のパッチと配信しているパッチで比較する
            if (m_pResource->shareAppInfo[i].deliveryInfoCount != 0)
            {
                int value = 0;
                if (CompareApplicationDeliveryInfo(
                    &value,
                    appDeliveryInfo,
                    appDeliveryInfoCount,
                    m_pResource->shareAppInfo[i].deliveryInfo,
                    m_pResource->shareAppInfo[i].deliveryInfoCount).IsSuccess())
                {
                    if (value < 0)
                    {
                        size += static_cast<size_t>(m_pResource->shareAppInfo[i].detailInfo.requiredSize);
                        continue;
                    }
                }
            }
        }
        m_pResource->requiredStorageSize = size;
    }

    void Host::GetSessionState(SessionState* pOutState) NN_NOEXCEPT
    {
        *pOutState = m_SessionState;
    }

    void Host::GetProgress(Progress* pOutProgress) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutProgress);

        uint64_t downloadedSize = 0;
        uint64_t size = 0;
        uint8_t role = m_Role;
        uint32_t transferIndex = 0;

        if (m_Role == TransferRole_None)
        {
            // 送信完了後、相手を待っているときの処理
            if (m_IsSender && m_IsWaitReceiver)
            {
                downloadedSize = m_TotalSendSize + 1;
                size = m_TotalSendSize + 1;
                role = TransferRole_Sender;
                transferIndex = m_NodeIndex;
            }
            // 通常の進捗処理
            else
            {
                downloadedSize = 0;
                size = 0;
                transferIndex = 0;
            }
        }
        else
        {
            if (m_pResource->contentInfo[m_pResource->contentInfoCount].contentsFlag.Test(ContentsType::System::Index))
            {
                if (m_Role == TransferRole_Sender)
                {
                    GetSendSystemUpdateProgress(&downloadedSize, &size);
                }
                else if (m_Role == TransferRole_Receiver)
                {
                    GetReceiveSystemUpdateProgress(&downloadedSize, &size, &(m_pResource->systemUpdateControl));
                }
            }
            else
            {
                if (m_Role == TransferRole_Sender)
                {
                    GetSendApplicationProgress(&downloadedSize, &size, m_pResource->contentInfo[m_pResource->contentInfoCount].applicationId.value);
                }
                else if (m_Role == TransferRole_Receiver)
                {
                    GetReceiveApplicationProgress(&downloadedSize, &size, m_pResource->contentInfo[m_pResource->contentInfoCount].applicationId.value);
                }
            }
            transferIndex = m_NodeIndex;

            // 取得した進捗で、SystemUpdate のサイズが total に反映されないようなので対処
            if (downloadedSize >= size)
            {
                downloadedSize = size;
            }
            // 相手の commit 待ちを表す進捗として +1 しておく
            // なので、常に実際に送受信するサイズより size(=送受信するサイズ)は 1Byte 多いものになる。
            size += 1;
        }

        pOutProgress->downloadedSize = downloadedSize;
        pOutProgress->size = size;
        pOutProgress->transferRole = role;
        pOutProgress->transferIndex = transferIndex;
        ::std::memcpy(&(pOutProgress->info), &m_pResource->contentInfo[m_pResource->contentInfoCount], sizeof(ContentsInfo));
    }

    Result Host::GetNodeProgress(NodeProgress* pOutNodeProgress, uint32_t index) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutNodeProgress);

        AdvertiseData data = {};
        m_AdvertiseManager.GetAdvertiseData(&data);
        /*
        NN_LCS_LOG_DEBUG("-- GetNodeProgress --\n");
        for (int i = 0; i < NodeCountMax; ++i)
        {
            NN_LCS_LOG_DEBUG("%d Index : %x[%d/%d]\n",
                i, data.progress[i].id, data.progress[i].downloadedContentCount, data.progress[i].contentCount);
        }
        NN_LCS_LOG_DEBUG("---------------------\n");
        */
        for (int i = 0; i < NodeCountMax; ++i)
        {
            if (nn::socket::InetNtohl(data.progress[i].id) == index)
            {
                pOutNodeProgress->contentCount = data.progress[i].contentCount;
                pOutNodeProgress->downloadedContentCount = data.progress[i].downloadedContentCount;
                NN_RESULT_SUCCESS;
            }
        }
        return ResultNodeNotFound();
    }

    Result Host::GetEulaDataSize(size_t* pOutSize, const char* eulaDataPath) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutSize);
        NN_SDK_ASSERT_NOT_NULL(eulaDataPath);

        return GetDownloadedEulaDataSize(pOutSize, eulaDataPath, &(m_pResource->systemUpdateControl));
    }

    Result Host::GetEulaData(
        size_t* pOutSize, void* buffer, size_t bufferSize, const char* eulaDataPath) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutSize);
        NN_SDK_ASSERT_NOT_NULL(buffer);
        NN_SDK_ASSERT_NOT_NULL(eulaDataPath);

        return GetDownloadedEulaData(
            pOutSize, buffer, bufferSize, eulaDataPath, &(m_pResource->systemUpdateControl));
    }

    Result Host::GetResumeContext(SessionContext* pOutSessionContext) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutSessionContext);

        ::std::memcpy(pOutSessionContext->_context, &m_pResource->context, sizeof(ResumeContext));

        NN_RESULT_SUCCESS;
    }

    Result Host::ResumeShare() NN_NOEXCEPT
    {
        ChangeState(State_Transferring);

        if (m_pResource->suspendedReason == SuspendedReason_IncompatibleContentsInfo)
        {
            // 発生しないので、省略。
        }
        else if (m_pResource->suspendedReason == SuspendedReason_NeedTerminateApplication)
        {
            m_pResource->suspendedReason = SuspendedReason_None;
            // suspend したことを通知するために送信
            SendSessionInfoToAll();
            nn::os::SignalEvent(&m_ResumeEvent);
        }
        else if (m_pResource->suspendedReason == SuspendedReason_EulaRequired)
        {
            /* ホストはここには来ない！ */
            NN_ABORT("Invalid State & SuspendedReason : SuspendedReason_EulaRequired\n");
        }
        else if (m_pResource->suspendedReason == SuspendedReason_RebootRequired)
        {
            /* ホストはここには来ない！ */
            NN_ABORT("Invalid State & SuspendedReason : SuspendedReason_RebootRequired\n");
        }
        else if (m_pResource->suspendedReason == SuspendedReason_StorageSpaceNotEnoughOnRequredNode)
        {
            NN_LCS_LOG_DEBUG("m_RequiredSize : %lld\n", m_RequiredSize);
            if (HasFreeSpaceSize(m_RequiredSize))
            {
                m_pResource->suspendedReason = SuspendedReason_None;
                nn::os::SignalEvent(&m_ResumeEvent);
            }
            else
            {
                ChangeState(State_Suspended);
            }
        }

        NN_RESULT_SUCCESS;
    }

    void Host::Start() NN_NOEXCEPT
    {
        m_IsAccepted = false;

        // セッションに参加できないようにクローズに設定
        m_AdvertiseManager.SetSessionSetting(SessionAttribute_Close);
        m_AdvertiseManager.UpdateAdvertiseData();

        // セッションの状態を更新
        m_SessionState = SessionState_None;
        NN_LCS_LOG_DEBUG("Update SessionState - %d\n", m_SessionState);
        UpdateAdvertiseSessionState();

        // Temp 接続のクライアントがいたら削除
        int nodeCount = 0;
        NodeInfo info[NodeCountMax] = {};
        m_NodeManager.GetNodeInfo(&nodeCount, info, NodeCountMax);
        for (int i = 0; i < nodeCount; ++i)
        {
            if (m_NodeManager.IsInnerState(info[i].index, NodeInnerState_Temporary) ||
                m_NodeManager.IsInnerState(info[i].index, NodeInnerState_Rejoining))
            {
                NN_LCS_LOG_DEBUG("Start - Reject : %x[InnerState : %d]\n", info[i].index);
                Reject(info[i].index, ContentsShareFailureReason_CommunicationError);
            }
        }

        // 端末の進捗を取得するためにインデックス保存
        m_pResource->nodeDetailProgressCount = m_pResource->nodeInfoCount;
        for (int i = 0; i < m_pResource->nodeDetailProgressCount; ++i)
        {
            m_pResource->nodeDetailProgress[i].index = m_pResource->nodeInfo[i].index;
            m_pResource->nodeDetailProgress[i].progress.contentCount = 0;
            m_pResource->nodeDetailProgress[i].progress.downloadedContentCount = 0;
        }

        // 端末の進捗を最新のものにアップデート
        UpdateAdvertiseProgress();

        // セッション情報を最新のものにアップデート
        UpdateNodesInfo();

        // スタートパケットの送信
        SendStartToAll();

        // 自身のコンテンツの情報を更新
        NodeDetailInfo detailInfo = {};
        detailInfo.nodeInfo.index = m_NodeManager.GetNodeIndex(m_MyAddress);
        ::std::strncpy(detailInfo.nodeInfo.userName, m_pResource->userName, UserNameBytesMax);
        detailInfo.nodeInfo.userName[UserNameBytesMax] = 0;
        detailInfo.version = m_pResource->version;
        GetSystemDeliveryInfo(&detailInfo.sysInfo);
        for (int i = 0; i < m_pResource->sessionInfo.contentsCount; ++i)
        {
            GetApplicationDetailInfo(&detailInfo.shareApp[i].info.detailInfo,
                m_pResource->appControlDataBuffer, AppControlDataSize, m_pResource->sessionInfo.contents[i].applicationId.value);
            GetApplicationDeliveryInfo(
                &detailInfo.shareApp[i].info.deliveryInfoCount, detailInfo.shareApp[i].info.deliveryInfo,
                ApplicationDeliveryInfoCountMax, m_pResource->sessionInfo.contents[i].applicationId.value);
        }
        detailInfo.state = NodeInnerState_Connected;
        NN_ABORT_UNLESS(m_NodeManager.AddNodeDetail(detailInfo));
        for (int i = 0; i < m_pResource->ownAppInfoCount; ++i)
        {
            if (m_pResource->ownAppInfo[i].detailInfo.id == m_pResource->shareComponent.id)
            {
                detail::GetApplicationDetailInfo(&m_pResource->ownAppInfo[i].detailInfo,
                    m_pResource->appControlDataBuffer, detail::AppControlDataSize,
                    m_pResource->shareComponent.id);
                detail::GetApplicationDeliveryInfo(&m_pResource->ownAppInfo[i].deliveryInfoCount,
                    m_pResource->ownAppInfo[i].deliveryInfo, detail::ApplicationDeliveryInfoCountMax,
                    m_pResource->shareComponent.id);
            }
        }

        // Client からのコンテンツ情報更新を待つ
        nn::os::SleepThread(WaitStartSharing);

        nn::os::SignalEvent(&m_StartTransferEvent);
    }

    Result Host::InitializeNodeManager(const SessionSettings& settings) NN_NOEXCEPT
    {
        Bit64 appId[SharableContentsCountMax] = {};
        for (int i = 0; i < settings.applicationCount; ++i)
        {
            appId[i] = settings.applications[i].value;
        }

        m_NodeManager.Initialize(settings.nodeCountMax);

        Result result;
        NodeDetailInfo detail = {};

        MacAddress macAddress;

        NN_RESULT_DO(GetMyIpv4Address(&m_MyAddress));
        NN_RESULT_DO(GetMacAddress(&macAddress, m_MyAddress));

        m_pResource->myIndex = m_NodeManager.AddNode(m_MyAddress, macAddress, 0);
        if (m_pResource->myIndex == 0)
        {
            return ResultNodeCountLimitation();
        }
        NN_LCS_LOG_DEBUG("AddNode : %x\n", m_pResource->myIndex);

        detail.nodeInfo.index = m_pResource->myIndex;
        ::std::memcpy(detail.nodeInfo.userName, m_pResource->userName, UserNameBytesMax);
        detail.version = m_pResource->version;
        GetSystemDeliveryInfo(&detail.sysInfo);
        for (int i = 0; i < settings.applicationCount; ++i)
        {
            NN_RESULT_DO(GetApplicationDetailInfo(&detail.shareApp[i].info.detailInfo,
                m_pResource->appControlDataBuffer, AppControlDataSize, settings.applications[i].value));
            NN_RESULT_DO(GetApplicationDeliveryInfo(
                &detail.shareApp[i].info.deliveryInfoCount, detail.shareApp[i].info.deliveryInfo,
                ApplicationDeliveryInfoCountMax, settings.applications[i].value));
        }
        detail.state = NodeInnerState_Connected;
        if (!m_NodeManager.AddNodeDetail(detail))
        {
            NN_ABORT("NodeManagerInitialize Failed\n");
        }
        NN_RESULT_SUCCESS;
    }

    void Host::SetAcceptance(bool policy) NN_NOEXCEPT
    {
        m_IsAccepted = policy;
    }

    void Host::DisconnectUnlistedClinet() NN_NOEXCEPT
    {
        int connectedAddressCount = 0;
        Ipv4Address connectedAddress[NodeCountMax] = {};
        // ldn で接続されているノードの IP アドレスを取得
        GetConnectedNodeIpAddress(&connectedAddressCount, connectedAddress, NodeCountMax);

        for (int i = 0; i < connectedAddressCount; ++i)
        {
            NN_LCS_LOG_DEBUG("connectedAddress : %x\n", connectedAddress[i]);
            // Index を取得できない(index = 0) のノードは ldn レイヤーで切断する
            if (connectedAddress[i] != 0 &&
                !m_NodeManager.GetNodeIndex(connectedAddress[i]))
            {
                NN_LCS_LOG_DEBUG("Reject %x\n", connectedAddress[i]);
                RejectOnNetwork(connectedAddress[i]);
            }
        }
    }

    void Host::MonitorThread() NN_NOEXCEPT
    {
        Result result;
        nn::ldn::NetworkInfo oldNetworkInfo = {};
        nn::ldn::NetworkInfo newNetworkInfo = {};
        nn::ldn::NodeLatestUpdate updates[nn::ldn::NodeCountMax];
        nn::os::SystemEventType stateChangeEvent = {};

        nn::ldn::AttachStateChangeEvent(&stateChangeEvent);

        nn::os::MultiWaitType       multiWait;
        nn::os::MultiWaitHolderType stateChangeHolder;
        nn::os::MultiWaitHolderType cancelHolder;
        nn::os::InitializeMultiWait(&multiWait);
        nn::os::InitializeMultiWaitHolder(&stateChangeHolder, &stateChangeEvent);
        nn::os::InitializeMultiWaitHolder(&cancelHolder, &m_CancelEvent);
        nn::os::LinkMultiWaitHolder(&multiWait, &stateChangeHolder);
        nn::os::LinkMultiWaitHolder(&multiWait, &cancelHolder);

        while (nn::os::WaitAny(&multiWait) != &cancelHolder)
        {
            nn::os::TryWaitSystemEvent(&stateChangeEvent);

            nn::ldn::State ldnState = nn::ldn::GetState();
            if (ldnState != nn::ldn::State_AccessPointCreated)
            {
                // Ldn の切断理由を取得するために一旦 API を呼ぶ
                result = nn::ldn::GetNetworkInfo(&newNetworkInfo);
                if (nn::ldn::ResultWifiOff().Includes(result))
                {
                    m_pResource->contentsShareFailureReason = ContentsShareFailureReason_FlightMode;
                }
                else if (nn::ldn::ResultSleep().Includes(result))
                {
                    m_pResource->contentsShareFailureReason = ContentsShareFailureReason_Sleep;
                }
                ChangeState(State_ContentsShareFailed);
                nn::os::SignalEvent(&m_CancelEvent);
                break;
            }

            result = nn::ldn::GetNetworkInfo(&newNetworkInfo, updates, nn::ldn::NodeCountMax);
            if (nn::ldn::ResultInvalidState::Includes(result))
            {
                nn::os::SignalEvent(&m_CancelEvent);
                break;
            }
            else if (nn::ldn::ResultDeviceNotAvailable::Includes(result))
            {
                nn::os::SignalEvent(&m_CancelEvent);
                break;
            }
            NN_ABORT_UNLESS(result.IsSuccess());

            for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
            {
                const auto& oldNode = oldNetworkInfo.ldn.nodes[i];
                const auto& newNode = newNetworkInfo.ldn.nodes[i];
                const auto& update = updates[i];
                if (update.stateChange == nn::ldn::NodeStateChange_Disconnect ||
                    update.stateChange == nn::ldn::NodeStateChange_DisconnectAndConnect)
                {
                    NN_LCS_LOG_DEBUG("Disconnect Node ID[%d] : %x\n", oldNode.nodeId, oldNode.ipv4Address.raw);
                    uint32_t index = m_NodeManager.GetNodeIndex(oldNode.ipv4Address.raw);
                    NN_LCS_LOG_DEBUG("Disconnect Node Index : %x\n", index);
                    if (index > 0)
                    {
                        if (m_NodeManager.IsInnerState(index, NodeInnerState_Rebooting) ||
                            m_NodeManager.IsInnerState(index, NodeInnerState_Rejoining))
                        {
                            NN_LCS_LOG_DEBUG("Temp Disconnect\n");
                            int socket = m_NodeManager.GetNodeSocket(index);
                            SocketClose(&socket);
                        }
                        else
                        {
                            if (!m_IsShareCompleted)
                            {
                                NN_LCS_LOG_DEBUG("Remove : %x\n", index);
                                m_NodeManager.RemoveNode(index);
                                UpdateNodesInfo();
                                m_pResource->stateEvent.SignalEvent();
                            }
                        }
                        nn::os::SignalEvent(&m_ClientStateChangeEvent);
                    }
                }
                if (update.stateChange == nn::ldn::NodeStateChange_Connect ||
                    update.stateChange == nn::ldn::NodeStateChange_DisconnectAndConnect)
                {
                    NN_LCS_LOG_DEBUG("Connect Node[%d] : %x\n", newNode.nodeId, newNode.ipv4Address.raw);
                    if (m_pResource->state.GetState() == State_Opened)
                    {
                        MacAddress macAddress = {};
                        for (int j = 0; j < 6; ++j)
                        {
                            macAddress.raw[j] = newNode.macAddress.raw[j];
                        }
                        m_NodeManager.AddNode(newNode.ipv4Address.raw, macAddress, 0);
                    }
                }
            }
            std::memcpy(&oldNetworkInfo, &newNetworkInfo, sizeof(nn::ldn::NetworkInfo));
        }
        nn::os::UnlinkMultiWaitHolder(&stateChangeHolder);
        nn::os::UnlinkMultiWaitHolder(&cancelHolder);
        nn::os::FinalizeMultiWaitHolder(&stateChangeHolder);
        nn::os::FinalizeMultiWaitHolder(&cancelHolder);
        nn::os::FinalizeMultiWait(&multiWait);

        NN_LCS_LOG_DEBUG("End MonitorThread\n");
    }// NOLINT(impl/function_size)

    void Host::ComThread() NN_NOEXCEPT
    {
        while (m_IsOpened)
        {
            AcceptClient();

            RecvPacket();

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(ReceiveIntervalMilliSeconds));
        }
        NN_LCS_LOG_DEBUG("End ComThread\n");
    }

    void Host::ShareThread() NN_NOEXCEPT
    {
        bool isForceUpdate = true;

        nn::os::WaitEvent(&m_StartTransferEvent);

        NN_LCS_LOG_DEBUG("/-------- Start ShareThread --------/\n");

        // ダウンロードしたコンテンツ数をクリア
        m_NodeManager.ClearDownloadedCount();

        while (m_IsOpened)
        {
            NN_LCS_LOG_DEBUG("while ShareThread\n");

            uint32_t latestIndex = 0;

            NN_LCS_LOG_DEBUG("m_MyAddress : %x\n", m_MyAddress);
            m_NodeManager.FindLatestSystemNode(&latestIndex, m_MyAddress);
            if (latestIndex != m_NodeManager.GetNodeIndex(m_MyAddress))
            {
                // 親交代をする
                NN_LCS_LOG_DEBUG("Migration!  - LatestIndex : %x\n", latestIndex);
                MigrationFunction(latestIndex);
                break;
            }

            if (isForceUpdate)
            {
                // 強制システムのスケジュールアップデート
                NN_LCS_LOG_DEBUG("ForceUpdate Schedule\n");
                if (!ConfirmSystemUpdateSchedule())
                {
                    NN_LCS_LOG_DEBUG("ClientLeaved System\n");
                    // 必須クライアントがいないので全員ｵﾜﾀ
                    SendRejectToAll(RejectReason_NodeLeaved);
                    m_pResource->contentsShareFailureReason = ContentsShareFailureReason_NodeLeaved;
                    ChangeState(State_ContentsShareFailed);
                    break;
                }
                // 強制システム更新が必要か(終わったか)を確認
                if (!m_NodeManager.IsCompletedSystemDownloadAll())
                {
                    NN_LCS_LOG_DEBUG("SystemUpdate - IncompatibleContentsInfo\n");

                    // セッションの状態を強制 LUP に更新
                    m_SessionState = SessionState_ForceLup;
                    NN_LCS_LOG_DEBUG("Update SessionState - %d\n", m_SessionState);
                    UpdateAdvertiseSessionState();

                    uint32_t receiverIndex = 0;
                    if (!GetCanShareSystemNodeIndex(&receiverIndex))
                    {
                        NN_LCS_LOG_DEBUG("Wait to Change Clients State\n");
                        nn::os::WaitEvent(&m_ClientStateChangeEvent);
                    }
                    else
                    {
                        ShareSystem(receiverIndex, SystemUpdateReason_IncompatibleContentsInfo);
                    }
                    continue;
                }
                // 強制アップデートした端末が全員そろうまで待つ
                while (!m_NodeManager.IsInnerStateAll(NodeInnerState_Connected))
                {
                    nn::os::WaitEvent(&m_ClientStateChangeEvent);
                    if (!m_IsOpened)
                    {
                        return;
                    }
                }
                bool isDown = false;
                UpdateShareAppInfo(&isDown);
                SendSessionInfoToAll();
            }
            isForceUpdate = false;

            // セッションの状態をコンテンツ配信中に更新
            m_SessionState = SessionState_ShareContents;
            NN_LCS_LOG_DEBUG("Update SessionState - %d\n", m_SessionState);
            UpdateAdvertiseSessionState();

            // ブラックリストに載っているかを確認
            for (int i = 0; i < m_pResource->shareAppInfoCount; ++i)
            {
                if (m_pResource->shareAppInfo[i].deliveryInfoCount == 0)
                {
                    continue;
                }
                auto result = CanDeliverApplication(
                    m_pResource->shareAppInfo[i].deliveryInfo, m_pResource->shareAppInfo[i].deliveryInfoCount,
                    m_pResource->appDeliveryInfoDataBuffer, AppDeliveryInfoSize);
                if (result.IsFailure())
                {
                    NN_LCS_LOG_DEBUG("CanDeliverApplication Error : %08x\n", result.GetInnerValueForDebug());
                    //ブラックリストに載ってるなら失敗
                    if (nn::ns::ResultApplicationUpdateRequiredByBlackList().Includes(result))
                    {
                        SendRejectToAll(RejectReason_UnshareableContents);
                        m_pResource->contentsShareFailureReason = ContentsShareFailureReason_UnshareableContents;
                    }
                    // その他での失敗はここでは発生しない想定だが、発生した場合は解散させる
                    else
                    {
                        SendRejectToAll(RejectReason_Unknown);
                        m_pResource->contentsShareFailureReason = ContentsShareFailureReason_UnknownError;
                    }
                    // Packet がクライアントに届くのを少し待つ
                    nn::os::SleepThread(WaitSendingPacket);
                    ChangeState(State_ContentsShareFailed);
                    return;
                }
            }

            // コンテンツ更新のスケジュールアップデート
            auto result = ConfirmContentShareSchedule();
            if (ResultFailureNodeLeaved::Includes(result))
            {
                NN_LCS_LOG_DEBUG("ClientLeaved Content\n");
                // 必須クライアントがいないので全員ｵﾜﾀ
                SendRejectToAll(RejectReason_NodeLeaved);
                m_pResource->contentsShareFailureReason = ContentsShareFailureReason_NodeLeaved;
                ChangeState(State_ContentsShareFailed);
                break;
            }
            else if (ResultFailureStorageError::Includes(result))
            {
                NN_LCS_LOG_DEBUG("ClientLeaved Content\n");
                // 配信途中でゲームカードが差され、想定していたアプリバージョンより上がったとき
                SendRejectToAll(RejectReason_NodeLeaved);
                m_pResource->contentsShareFailureReason = ContentsShareFailureReason_StorageError;
                ChangeState(State_ContentsShareFailed);
                break;
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }

            // システム更新が必要か(終わったか)を確認
            if (!m_NodeManager.IsCompletedSystemDownloadAll())
            {
                NN_LCS_LOG_DEBUG("SystemUpdate - Needed\n");

                uint32_t receiverIndex = 0;
                if (!GetCanShareSystemNodeIndex(&receiverIndex))
                {
                    NN_LCS_LOG_DEBUG("Wait to Change Clients State\n");
                    nn::os::WaitEvent(&m_ClientStateChangeEvent);
                }
                else
                {
                    ShareSystem(receiverIndex, SystemUpdateReason_NeedToRunApplication);
                }
                continue;
            }

            // コンテンツの配信が終わったか確認
            if (!m_NodeManager.IsCompletedContentDownloadAll())
            {
                NN_LCS_LOG_DEBUG("Content Share\n");

                uint32_t receiverIndex = 0;
                int appIndex = -1;
                if (!GetCanShareContentsNodeIndex(&receiverIndex, &appIndex))
                {
                    NN_LCS_LOG_DEBUG("Wait to Change Clients State\n");
                    // セッションの状態をユーザ操作待ちに更新
                    m_SessionState = SessionState_Waiting;
                    NN_LCS_LOG_DEBUG("Update SessionState - %d\n", m_SessionState);
                    UpdateAdvertiseSessionState();
                    nn::os::WaitEvent(&m_ClientStateChangeEvent);
                }
                else
                {
                    if (!ShareContent(receiverIndex, appIndex))
                    {
                        return;
                    }
                }
                continue;
            }
            // 終わり
            NN_LCS_LOG_DEBUG("CotentShare Complete\n");
            m_IsShareCompleted = true;

            // AdvertiseData がクライアントに届くのを少し待つ
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(WaitSendCompletePacketIntervalMilliSeconds));

            SendEndContentsSareToAll();
            if (m_pResource->state.GetState() != State_Suspended)
            {
                ChangeState(State_Completed);
            }
            break;
        }
        NN_LCS_LOG_DEBUG("End ShareThread\n");
    } // NOLINT(impl/function_size)

    void Host::TimeoutThread() NN_NOEXCEPT
    {
        int nodeCount = 0;
        NodeInfo info[NodeCountMax] = {};

        while (m_IsOpened)
        {
            ++m_Time;

            //NN_LCS_LOG_DEBUG("Time : %lld\n", m_Time);

            m_NodeManager.GetNodeInfo(&nodeCount, info, NodeCountMax);
            for (int i = 0; i < nodeCount; ++i)
            {
                if (m_NodeManager.IsInnerState(info[i].index, NodeInnerState_Rebooting) &&
                    m_NodeManager.GetLeaveTime(info[i].index) == m_Time)
                {
                    m_NodeManager.RemoveNode(info[i].index);
                    if (m_NodeManager.GetNodeCount() == 1)
                    {
                        m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
                        ChangeState(State_ContentsShareFailed);
                        return;
                    }
                    UpdateNodesInfo();
                    UpdateAdvertiseProgress();
                    nn::os::SignalEvent(&m_ClientStateChangeEvent);
                }
            }
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
    }

    void Host::UpdateShareAppInfo(bool* pOutIsDown) NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("--- UpdateShareAppInfo ---\n");
        *pOutIsDown = false;
        m_pResource->shareAppInfoCount = m_pResource->sessionInfo.contentsCount;
        for (int i = 0; i < m_pResource->shareAppInfoCount; ++i)
        {
            ApplicationInfo latestAppInfo = {};
            int value = 0;
            m_NodeManager.FindLatestAppDetail(&latestAppInfo, i);

            if (m_pResource->shareAppInfo[i].deliveryInfoCount != 0)
            {
                if (CompareApplicationDeliveryInfo(&value,
                    m_pResource->shareAppInfo[i].deliveryInfo, m_pResource->shareAppInfo[i].deliveryInfoCount,
                    latestAppInfo.deliveryInfo, latestAppInfo.deliveryInfoCount).IsSuccess())
                {
                    if (value > 0)
                    {
                        *pOutIsDown = true;
                    }
                }
            }
            ::std::memcpy(&m_pResource->shareAppInfo[i], &latestAppInfo, sizeof(ApplicationInfo));
        }
    }

    void Host::UpdateAdvertiseInfo() NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("--- UpdataAdvertiseInfo ---\n");

        // 端末数の更新
        int nodeCount = m_NodeManager.GetConnectedNodeCount();
        m_AdvertiseManager.SetNodeCount(nodeCount);
        NN_LCS_LOG_DEBUG("NodeCount : %d\n", nodeCount);

        // 最新のアプリの情報に更新
        if (m_Policy == ContentsShareVersionPolicy_Latest)
        {
            NN_ABORT_UNLESS_LESS_EQUAL(m_pResource->sessionInfo.contentsCount, SharableContentsCountMax);
            m_pResource->shareAppInfoCount = m_pResource->sessionInfo.contentsCount;
            for (int i = 0; i < m_pResource->sessionInfo.contentsCount; ++i)
            {
                ApplicationInfo latestAppInfo = {};

                m_NodeManager.FindLatestAppDetail(&latestAppInfo, i);
                m_AdvertiseManager.SetContentInfo(
                    static_cast<size_t>(latestAppInfo.detailInfo.requiredSize), latestAppInfo.detailInfo.displayVersion, i);

                // 最新の ApplicationInfo に更新
                if (m_pResource->state.GetState() == State_Initialized ||
                    m_pResource->state.GetState() == State_Opened)
                {
                    ::std::memcpy(&m_pResource->shareAppInfo[i], &latestAppInfo, sizeof(ApplicationInfo));
                }

                NN_LCS_LOG_DEBUG("App%d Version : %s\n", i, latestAppInfo.detailInfo.displayVersion);
                NN_LCS_LOG_DEBUG("App%d ReqSize : %d\n", i, latestAppInfo.detailInfo.requiredSize);
            }
        }
        NN_LCS_LOG_DEBUG("---------------------------\n");

        m_AdvertiseManager.UpdateAdvertiseData();
        UpdateSessionInfo();
    }

    void Host::UpdateAdvertiseSessionState() NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("--- UpdateAdvertiseSessionState ---\n");

        // セッションの状態を設定
        m_AdvertiseManager.SetSessionState(m_SessionState);

        // アドバタイズデータを更新
        m_AdvertiseManager.UpdateAdvertiseData();
    }

    void Host::UpdateAdvertiseProgress() NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("--- UpdataAdvertiseProgress ---\n");

        // 端末進捗を取得して更新
        int nodeCount = 0;
        NodeDetailProgress nodeProgress[NodeCountMax] = {};
        AdvertiseNodeProgress advertisProgress[NodeCountMax] = {};

        // NodeManager から端末の進捗の最新版を取得
        m_NodeManager.GetNodeProgress(&nodeCount, nodeProgress, NodeCountMax);
        for (int i = 0; i < nodeCount; ++i)
        {
            for (int j = 0; j < m_pResource->nodeDetailProgressCount; ++j)
            {
                if (m_pResource->nodeDetailProgress[j].index == nodeProgress[i].index)
                {
                    m_pResource->nodeDetailProgress[j].progress.contentCount = nodeProgress[i].progress.contentCount;
                    m_pResource->nodeDetailProgress[j].progress.downloadedContentCount
                        = nodeProgress[i].progress.downloadedContentCount;
                }
            }
        }

        // Advertise データに端末の進捗を入れる
        for (int i = 0; i < m_pResource->nodeDetailProgressCount; ++i)
        {
            advertisProgress[i].id = nn::socket::InetHtonl(m_pResource->nodeDetailProgress[i].index);
            advertisProgress[i].contentCount =
                static_cast<uint8_t>(m_pResource->nodeDetailProgress[i].progress.contentCount);
            advertisProgress[i].downloadedContentCount =
                static_cast<uint8_t>(m_pResource->nodeDetailProgress[i].progress.downloadedContentCount);
            NN_LCS_LOG_DEBUG("[%d]%x : %d / %d\n", i, advertisProgress[i].id,
                advertisProgress[i].downloadedContentCount, advertisProgress[i].contentCount);
        }

        NN_LCS_LOG_DEBUG("-------------------------------\n");

        m_AdvertiseManager.SetNodeProgress(advertisProgress, m_pResource->nodeDetailProgressCount);

        // コンテンツの成功・失敗を入れる
        m_AdvertiseManager.SetConteShareSucceeded(m_IsContentShareSucceeded);

        m_AdvertiseManager.UpdateAdvertiseData();
    }

    void Host::AcceptClient() NN_NOEXCEPT
    {
        nn::socket::SockAddrIn sockAddr = {};
        nn::socket::SockLenT sockLen = sizeof(sockAddr);

        int socket = nn::socket::Accept(m_Socket,
            reinterpret_cast<nn::socket::SockAddr*>(&sockAddr), &sockLen);

        if (socket > 0)
        {
            Ipv4Address address = nn::socket::InetNtohl(sockAddr.sin_addr.S_addr);
            NN_LCS_LOG_DEBUG("Socket Accept : %x\n", address);

            if (SetSockLinger(&socket) < 0)
            {
                NN_LCS_LOG_DEBUG("set sock opt failed(SO_NN_LINGER): %d\n", nn::socket::GetLastError());
                SocketClose(&socket);
            }

            if (SetSockReuseAddr(&socket) < 0)
            {
                NN_LCS_LOG_DEBUG("set sock opt failed(SO_REUSEADDR): %d\n", nn::socket::GetLastError());
                SocketClose(&socket);
            }

            if (!m_NodeManager.AddNodeSocket(socket, address))
            {
                NN_LCS_LOG_DEBUG("AddNodeSocket Fail : %x\n", address);
                SocketClose(&socket);
            }
        }
    }

    void Host::Reject(uint32_t index, ContentsShareFailureReason reason) NN_NOEXCEPT
    {
        NodeDetailInfo detailInfo = {};
        if (m_NodeManager.GetNodeDetailInfo(&detailInfo, index).IsFailure())
        {
            NN_LCS_LOG_DEBUG("Node Not Found : %x\n", index);
            return;
        }

        // 切断パケット送信
        RejectReason rejectReason = RejectReason_None;
        switch (reason)
        {
        case ContentsShareFailureReason_CommunicationError:
            rejectReason = RejectReason_CommunicationError;
            break;
        case ContentsShareFailureReason_NodeLeaved:
            rejectReason = RejectReason_NodeLeaved;
            break;
        case ContentsShareFailureReason_CannotUpdateSystem:
            rejectReason = RejectReason_CannotUpdateSystem;
            break;
        case ContentsShareFailureReason_NotExistDownloadContents:
            rejectReason = RejectReason_NotExistDownloadContents;
            break;
        default:
            break;
        }

        NN_LCS_LOG_DEBUG("SendReject\n");
        SendRejectNode(detailInfo.socket, rejectReason);

        // lcs のクライアントから消去
        if (m_NodeManager.RemoveNode(index).IsFailure())
        {
            NN_LCS_LOG_DEBUG("RemoveNode Failed\n");
        }
        UpdateNodesInfo();

        // ldn での切断 <- Reject が届く前に STA が切断を検知してしまうのでお蔵入り
        //RejectOnNetwork(detailInfo.ipAddress);

        m_pResource->stateEvent.SignalEvent();
    }

    void Host::RecvPacket() NN_NOEXCEPT
    {
        int nodeCount = 0;
        NodeInfo info[NodeCountMax] = {};
        m_NodeManager.GetNodeInfo(&nodeCount, info, NodeCountMax);
        for (int i = 0; i < nodeCount; ++i)
        {
            int socket = m_NodeManager.GetNodeSocket(info[i].index);
            if (socket != -1 && info[i].index != m_NodeManager.GetNodeIndex(m_MyAddress))
            {
                CommonHeader header;
                ssize_t recvSize = RecvCommonHeaderWithPeek(socket, &header);
                if (recvSize < 0)
                {
                    nn::socket::Errno error = nn::socket::GetLastError();
                    if (error != nn::socket::Errno::EAgain)
                    {
                        NN_LCS_LOG_WARN("%s failed: errno = %d\n",
                            NN_CURRENT_FUNCTION_NAME, error);
                    }
                }
                else if (sizeof(CommonHeader) <= static_cast<size_t>(recvSize))
                {
                    switch (header.type)
                    {
                    case Type_ApplicationControlDataSizeReq:
                    {
                        RecvApplicationControlDataSize(socket);
                        break;
                    }
                    case Type_ApplicationControlDataReq:
                    {
                        RecvApplicationControlData(socket);
                        break;
                    }
                    case Type_Join:
                    {
                        RecvJoin(socket, info[i].index);
                        break;
                    }
                    case Type_Leave:
                    {
                        RecvLeave(socket, info[i].index);
                        break;
                    }
                    case Type_ShareSystemClearRes:
                    {
                        RecvShareSystemClearRes(socket, info[i].index);
                        break;
                    }
                    case Type_ShareSystemSendRes:
                    {
                        RecvShareSystemSendRes(socket);
                        break;
                    }
                    case Type_ShareSystemRecvRes:
                    {
                        RecvShareSystemRecvRes(socket, info[i].index);
                        break;
                    }
                    case Type_ShareContentClearRes:
                    {
                        RecvShareContentClearRes(socket, info[i].index);
                        break;
                    }
                    case Type_ShareContentSendRes:
                    {
                        RecvShareContentSendRes(socket, info[i].index);
                        break;
                    }
                    case Type_ShareContentRecvRes:
                    {
                        RecvShareContentRecvRes(socket, info[i].index);
                        break;
                    }
                    case Type_Update:
                    {
                        RecvUpdate(socket, info[i].index);
                        break;
                    }
                    case Type_Suspend:
                    {
                        RecvSuspend(socket, info[i].index);
                        break;
                    }
                    case Type_Resume:
                    {
                        RecvResume(socket, info[i].index);
                        break;
                    }
                    case Type_Rejoin:
                    {
                        RecvRejoin(socket, info[i].index);
                        break;
                    }
                    default:
                        break;
                    }
                }
                else if (0 < recvSize)
                {
                    NN_LCS_LOG_WARN("%s failed: small packet %d Bytes)\n",
                        NN_CURRENT_FUNCTION_NAME, static_cast<int>(recvSize));
                }
                else
                {
                    /* 再合流のためにクライアントが切断しようとしたとき、先に Socket で Reject してしまうのでお蔵入り。
                    Reject(index, ContentsShareFailureReason_CommunicationError);
                    */
                }
            }
        }
    } // NOLINT(impl/function_size)

    void Host::RecvApplicationControlDataSize(int socket) NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("Recv Type_ApplicationControlDataSizeReq\n");

        ssize_t recvSize = RecvData(
            socket, m_pResource->packetBuffer, PacketSize, sizeof(AppControlDataReqPacket));
        if (recvSize < 0)
        {
            NN_LCS_LOG_WARN("%s failed: errno = %d\n",
                NN_CURRENT_FUNCTION_NAME, nn::socket::GetLastError());
        }
        else if (static_cast<size_t>(recvSize) < sizeof(AppControlDataReqPacket))
        {
            NN_LCS_LOG_WARN("%s failed: small packet (%d Bytes)\n",
                NN_CURRENT_FUNCTION_NAME, static_cast<int>(recvSize));
        }
        else
        {
            AppControlDataReqPacket* packet =
                reinterpret_cast<AppControlDataReqPacket*>(m_pResource->packetBuffer);
            nn::ncm::ApplicationId id;
            id.value = GetMerged(nn::socket::InetNtohl(packet->appUpperId),
                nn::socket::InetNtohl(packet->appLowerId));
            SendAppControlDataRes(socket, id,
                m_pResource->appControlDataBuffer, AppControlDataSize, true);
        }
    }

    void Host::RecvApplicationControlData(int socket) NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("Recv Type_ApplicationControlDataReq\n");

        ssize_t recvSize = RecvData(
            socket, m_pResource->packetBuffer, PacketSize, sizeof(AppControlDataReqPacket));
        if (recvSize < 0)
        {
            NN_LCS_LOG_WARN("%s failed: errno = %d\n",
                NN_CURRENT_FUNCTION_NAME, nn::socket::GetLastError());
        }
        else if (static_cast<size_t>(recvSize) < sizeof(AppControlDataReqPacket))
        {
            NN_LCS_LOG_WARN("%s failed: small packet (%d Bytes)\n",
                NN_CURRENT_FUNCTION_NAME, static_cast<int>(recvSize));
        }
        else
        {
            AppControlDataReqPacket* packet =
                reinterpret_cast<AppControlDataReqPacket*>(m_pResource->packetBuffer);
            nn::ncm::ApplicationId id;
            id.value = GetMerged(nn::socket::InetNtohl(packet->appUpperId),
                nn::socket::InetNtohl(packet->appLowerId));
            SendAppControlDataRes(socket, id,
                m_pResource->appControlDataBuffer, AppControlDataSize, false);
        }
    }

    void Host::RecvJoin(int socket, uint32_t index) NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("Recv Type_JoinReq from : %x\n", index);

        if (!m_IsAccepted)
        {
            SendRejectNode(socket, RejectReason_ConnectionFailed);
            SocketClose(&socket);
            m_NodeManager.RemoveNode(index);
            return;
        }

        Result result = UpdateClient(socket, index);
        if (result.IsSuccess())
        {
            SendAcceptNode(socket, index);
            m_NodeManager.SetInnerState(NodeInnerState_Connected, index);
            UpdateNodesInfo();
            m_pResource->stateEvent.SignalEvent();
        }
        else
        {
            if (ResultConnectionFailed().Includes(result))
            {
                SendRejectNode(socket, RejectReason_ConnectionFailed);
            }
            else if (ResultCommunicationError().Includes(result))
            {
                SendRejectNode(socket, RejectReason_CommunicationError);
            }
            else if (ResultLowerSystemVersion().Includes(result))
            {
                SendRejectNode(socket, RejectReason_LowerSystemVersion);
            }
            else if (ResultHigherSystemVersion().Includes(result))
            {
                SendRejectNode(socket, RejectReason_HigherSystemVersion);
            }
            else if (ResultIncompatibleSystemVersion().Includes(result))
            {
                SendRejectNode(socket, RejectReason_IncompatibleSystemVersion);
            }
            else
            {
                NN_LCS_LOG_DEBUG("RecvJion Error : %08x\n", result.GetDescription());
            }
        }
    }

    void Host::RecvLeave(int socket, uint32_t index) NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("Recv Type_Leave from %x\n", index);

        ssize_t recvSize = RecvData(
            socket, m_pResource->packetBuffer, PacketSize, sizeof(LeavePacket));
        if (recvSize < 0)
        {
            NN_LCS_LOG_WARN("%s failed: errno = %d\n",
                NN_CURRENT_FUNCTION_NAME, nn::socket::GetLastError());
        }
        else if (static_cast<size_t>(recvSize) < sizeof(LeavePacket))
        {
            NN_LCS_LOG_WARN("%s failed: small packet (%d Bytes)\n",
                NN_CURRENT_FUNCTION_NAME, static_cast<int>(recvSize));
        }
        else
        {
            LeavePacket* packet = reinterpret_cast<LeavePacket*>(m_pResource->packetBuffer);

            if (packet->leaveReason == LeaveReason_Reboot)
            {
                NN_LCS_LOG_DEBUG("%x is rebooting now / time : %d\n", index, m_Time);
                m_NodeManager.SetInnerState(NodeInnerState_Rebooting, index);
                m_NodeManager.DeleteNodeSocket(index);
                m_NodeManager.SetLeaveTime(m_Time + WaitRebootTimeSeconds, index);
                return;
            }

            if (m_NodeManager.IsInnerState(index, NodeInnerState_Rejoining))
            {
                return;
            }

            NN_LCS_LOG_DEBUG("Removed %x\n", index);
            m_NodeManager.RemoveNode(index);
            UpdateNodesInfo();
            nn::os::SignalEvent(&m_ClientStateChangeEvent);
            m_pResource->stateEvent.SignalEvent();
        }
    }

    void Host::RecvShareSystemClearRes(int socket, uint32_t index) NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("Recv Type_ShareSystemClearRes\n");

        ssize_t recvSize = RecvData(
            socket, m_pResource->packetBuffer, PacketSize, sizeof(SystemShareResPacket));
        if (recvSize < 0)
        {
            NN_LCS_LOG_WARN("%s failed: errno = %d\n",
                NN_CURRENT_FUNCTION_NAME, nn::socket::GetLastError());
        }
        else if (static_cast<size_t>(recvSize) < sizeof(SystemShareResPacket))
        {
            NN_LCS_LOG_WARN("%s failed: small packet (%d Bytes)\n",
                NN_CURRENT_FUNCTION_NAME, static_cast<int>(recvSize));
        }
        else
        {
            SystemShareResPacket* packet =
                reinterpret_cast<SystemShareResPacket*>(m_pResource->packetBuffer);

            if (packet->response == Response_Reject)
            {
                NN_LCS_LOG_DEBUG("Response_Reject\n");
                Reject(index, ContentsShareFailureReason_NotExistDownloadContents);
            }
            else if (packet->response == Response_Accept)
            {
                NN_LCS_LOG_DEBUG("Response_Accept\n");
            }
            else if (packet->response == Response_Suspend)
            {
                NN_LCS_LOG_DEBUG("Response_Suspend\n");
                m_NodeManager.SetInnerState(NodeInnerState_Suspending, index);
            }
            nn::os::SignalEvent(&m_RecvResPacketEvent);
        }
    }

    void Host::RecvShareSystemSendRes(int socket) NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("Recv Type_ShareSystemSendRes\n");

        ssize_t recvSize = RecvData(
            socket, m_pResource->packetBuffer, PacketSize, sizeof(SystemShareResPacket));
        if (recvSize < 0)
        {
            NN_LCS_LOG_WARN("%s failed: errno = %d\n",
                NN_CURRENT_FUNCTION_NAME, nn::socket::GetLastError());
        }
        else if (static_cast<size_t>(recvSize) < sizeof(SystemShareResPacket))
        {
            NN_LCS_LOG_WARN("%s failed: small packet (%d Bytes)\n",
                NN_CURRENT_FUNCTION_NAME, static_cast<int>(recvSize));
        }
        else
        {
            SystemShareResPacket* packet =
                reinterpret_cast<SystemShareResPacket*>(m_pResource->packetBuffer);

            ::std::memcpy(m_pResource->shareComponent.systemDeliveryInfo.sysDeliveryInfo,
                packet->sysInfo, SystemDeliveryInfoSize);

            nn::os::SignalEvent(&m_RecvResPacketEvent);
        }
    }

    void Host::RecvShareSystemRecvRes(int socket, uint32_t index) NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("Recv Type_ShareSystemRecvRes\n");

        ssize_t recvSize = RecvData(
            socket, m_pResource->packetBuffer, PacketSize, sizeof(SystemShareResPacket));
        if (recvSize < 0)
        {
            NN_LCS_LOG_WARN("%s failed: errno = %d\n",
                NN_CURRENT_FUNCTION_NAME, nn::socket::GetLastError());
        }
        else if (static_cast<size_t>(recvSize) < sizeof(SystemShareResPacket))
        {
            NN_LCS_LOG_WARN("%s failed: small packet (%d Bytes)\n",
                NN_CURRENT_FUNCTION_NAME, static_cast<int>(recvSize));
        }
        else
        {
            SystemShareResPacket* packet =
                reinterpret_cast<SystemShareResPacket*>(m_pResource->packetBuffer);
            if (packet->response == Response_Suspend)
            {
                NN_LCS_LOG_DEBUG("Response_Suspend\n");
                m_NodeManager.SetInnerState(NodeInnerState_Suspending, index);
            }
            else if (packet->response == Response_Accept)
            {
                NN_LCS_LOG_DEBUG("Response_Accept\n");
            }
            nn::os::SignalEvent(&m_RecvResPacketEvent);
        }
    }

    void Host::RecvShareContentClearRes(int socket, uint32_t index) NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("Recv Type_ShareContentClearRes\n");

        ssize_t recvSize = RecvData(
            socket, m_pResource->packetBuffer, PacketSize, sizeof(ContentShareResPacket));
        if (recvSize < 0)
        {
            NN_LCS_LOG_WARN("%s failed: errno = %d\n",
                NN_CURRENT_FUNCTION_NAME, nn::socket::GetLastError());
        }
        else if (static_cast<size_t>(recvSize) < sizeof(ContentShareResPacket))
        {
            NN_LCS_LOG_WARN("%s failed: small packet (%d Bytes)\n",
                NN_CURRENT_FUNCTION_NAME, static_cast<int>(recvSize));
        }
        else
        {
            ContentShareResPacket* packet =
                reinterpret_cast<ContentShareResPacket*>(m_pResource->packetBuffer);
            if (packet->contentMetaKeyCount != 0)
            {
                NN_LCS_LOG_WARN("ContentShareResPacket.contentMetaKeyCount = %u\n",
                    packet->contentMetaKeyCount);
            }
            else
            {
                if (packet->response == Response_Suspend)
                {
                    NN_LCS_LOG_DEBUG("Response_Suspend\n");
                    m_NodeManager.SetInnerState(NodeInnerState_Suspending, index);
                }
                else if (packet->response == Response_Accept)
                {
                    NN_LCS_LOG_DEBUG("Response_Accept\n");
                }
                nn::os::SignalEvent(&m_RecvResPacketEvent);
            }
        }
    }

    void Host::RecvShareContentSendRes(int socket, uint32_t index) NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("Recv Type_ShareContentSendRes\n");

        ssize_t recvSize = RecvData(
            socket, m_pResource->packetBuffer, PacketSize, sizeof(ContentShareResPacket));
        if (recvSize < 0)
        {
            NN_LCS_LOG_WARN("%s failed: errno = %d\n",
                NN_CURRENT_FUNCTION_NAME, nn::socket::GetLastError());
        }
        else if (static_cast<size_t>(recvSize) < sizeof(ContentShareResPacket))
        {
            NN_LCS_LOG_WARN("%s failed: small packet (%d Bytes)\n",
                NN_CURRENT_FUNCTION_NAME, static_cast<int>(recvSize));
        }
        else
        {
            ContentShareResPacket* packet =
                reinterpret_cast<ContentShareResPacket*>(m_pResource->packetBuffer);

            if (packet->response == Response_Suspend)
            {
                NN_LCS_LOG_DEBUG("Response_Suspend\n");
                m_NodeManager.SetInnerState(NodeInnerState_Suspending, index);
            }
            else if (packet->response == Response_Accept)
            {
                NN_LCS_LOG_DEBUG("Response_Accept\n");

                if (packet->contentMetaKeyCount != 0)
                {
                    m_pResource->shareComponent.contentMetaKeyCount = packet->contentMetaKeyCount;
                    NN_LCS_LOG_DEBUG("contentMetaKeyCount : %d\n", m_pResource->shareComponent.contentMetaKeyCount);
                    bool isRecv = false;
                    int count = 0;
                    while (!isRecv)
                    {
                        if (RecvData(socket, m_pResource->packetBuffer, PacketSize,
                            sizeof(ContentMetaKeyPacket)) == sizeof(ContentMetaKeyPacket))
                        {
                            ContentMetaKeyPacket *contentMetaKeyPacket =
                                reinterpret_cast<ContentMetaKeyPacket*>(m_pResource->packetBuffer);
                            ::std::memcpy(
                                &m_pResource->shareComponent.contentMetaKey[count],
                                contentMetaKeyPacket->contentMetaKey, ContentMetaKeySize);
                            ++count;
                            if (count == m_pResource->shareComponent.contentMetaKeyCount)
                            {
                                isRecv = true;
                            }
                        }
                    }
                }
            }
            nn::os::SignalEvent(&m_RecvResPacketEvent);
        }
    }

    void Host::RecvShareContentRecvRes(int socket, uint32_t index) NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("Recv Type_ShareContentRecvRes\n");

        ssize_t recvSize = RecvData(
            socket, m_pResource->packetBuffer, PacketSize, sizeof(ContentShareResPacket));
        if (recvSize < 0)
        {
            NN_LCS_LOG_WARN("%s failed: errno = %d\n",
                NN_CURRENT_FUNCTION_NAME, nn::socket::GetLastError());
        }
        else if (static_cast<size_t>(recvSize) < sizeof(ContentShareResPacket))
        {
            NN_LCS_LOG_WARN("%s failed: small packet (%d Bytes)\n",
                NN_CURRENT_FUNCTION_NAME, static_cast<int>(recvSize));
        }
        else
        {
            ContentShareResPacket* packet =
                reinterpret_cast<ContentShareResPacket*>(m_pResource->packetBuffer);
            if (packet->contentMetaKeyCount != 0)
            {
                NN_LCS_LOG_WARN("ContentShareResPacket.contentMetaKeyCount = %u\n",
                    packet->contentMetaKeyCount);
            }
            else
            {
                if (packet->response == Response_Suspend)
                {
                    NN_LCS_LOG_DEBUG("Response_Suspend\n");
                    m_NodeManager.SetInnerState(NodeInnerState_Suspending, index);
                }
                else if (packet->response == Response_Accept)
                {
                    NN_LCS_LOG_DEBUG("Response_Accept\n");
                }
                nn::os::SignalEvent(&m_RecvResPacketEvent);
            }
        }
    }

    void Host::RecvUpdate(int socket, uint32_t index) NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("Recv Type_Update from %x\n", index);

        if (UpdateClient(socket, index).IsSuccess())
        {
            if (m_NodeManager.IsInnerState(index, NodeInnerState_Transferring) ||
                m_NodeManager.IsInnerState(index, NodeInnerState_Rejoining))
            {
                // Client -> Client の配信終わった時と、再接続からの Update だったら状態を変更する
                m_NodeManager.SetInnerState(NodeInnerState_Connected, index);
            }
            UpdateNodesInfo();
            UpdateAdvertiseProgress();

            nn::os::SignalEvent(&m_ClientStateChangeEvent);
        }
    }

    void Host::RecvSuspend(int socket, uint32_t index) NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("Recv Type_Suspend from %x\n", index);

        ssize_t recvSize = RecvData(
            socket, m_pResource->packetBuffer, PacketSize, sizeof(SuspendPacket));
        if (recvSize < 0)
        {
            NN_LCS_LOG_WARN("%s failed: errno = %d\n",
                NN_CURRENT_FUNCTION_NAME, nn::socket::GetLastError());
        }
        else if (static_cast<size_t>(recvSize) < sizeof(SuspendPacket))
        {
            NN_LCS_LOG_WARN("%s failed: small packet (%d Bytes)\n",
                NN_CURRENT_FUNCTION_NAME, static_cast<int>(recvSize));
        }
        else
        {
            m_NodeManager.SetInnerState(NodeInnerState_Suspending, index);
            nn::os::SignalEvent(&m_ClientStateChangeEvent);
        }
    }

    void Host::RecvResume(int socket, uint32_t index) NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("Recv Type_Resume form %x\n", index);

        ssize_t recvSize = RecvData(
            socket, m_pResource->packetBuffer, PacketSize, sizeof(ResumePacket));
        if (recvSize < 0)
        {
            NN_LCS_LOG_WARN("%s failed: errno = %d\n",
                NN_CURRENT_FUNCTION_NAME, nn::socket::GetLastError());
        }
        else if (static_cast<size_t>(recvSize) < sizeof(ResumePacket))
        {
            NN_LCS_LOG_WARN("%s failed: small packet (%d Bytes)\n",
                NN_CURRENT_FUNCTION_NAME, static_cast<int>(recvSize));
        }
        else
        {
            // 受信者から SuspendedReason_NeedTerminateApplication の Resume を受けたときの処理
            if (m_NodeIndex == index && m_IsWaitReceiver)
            {
                m_IsWaitReceiver = false;
            }
            // 通常の Resume 処理
            else
            {
                m_NodeManager.SetInnerState(NodeInnerState_Connected, index);
                SendSessionInfoToAll();
                nn::os::SignalEvent(&m_ClientStateChangeEvent);
            }
        }
    }

    void Host::RecvRejoin(int socket, uint32_t index) NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("Recv Type_Rejoin from index\n", index);

        UpdateClient(socket, index);
        m_NodeManager.SetInnerState(NodeInnerState_Connected, index);
    }

    Result Host::UpdateClient(int socket, uint32_t index) NN_NOEXCEPT
    {
        NodeDetailInfo detail = {};
        NodeDetailInfoPacket packet;

        if (m_NodeManager.IsNodeCountLimited())
        {
            return ResultConnectionFailed();
        }

        // クライアントの情報があるならロードされる
        m_NodeManager.GetNodeDetailInfo(&detail, index);

        NN_RESULT_DO(WaitRecvData(socket, &packet, sizeof(NodeDetailInfoPacket)));
        detail.nodeInfo.index = index;
        ::std::memcpy(detail.nodeInfo.userName, packet.userName, UserNameBytesMax);
        detail.version._raw = packet.lcsVersion;
        int appCount = packet.appCount;
        detail.downloadedCount = packet.downloadedContentCount;
        ::std::memcpy(&detail.sysInfo, packet.sysInfo, SystemDeliveryInfoSize);
        NN_LCS_LOG_DEBUG("Name : %s\n", detail.nodeInfo.userName);
        NN_LCS_LOG_DEBUG("appCount : %d\n", appCount);
        NN_LCS_LOG_DEBUG("DownloadedContentCount : %d\n", detail.downloadedCount);
        NN_ABORT_UNLESS_LESS_EQUAL(appCount, SharableContentsCountMax);
        for (int i = 0; i < appCount; ++i)
        {
            AppInfoPacket fromAppInfo;
            ApplicationDetailInfo toAppInfo;
            NN_RESULT_DO(WaitRecvData(socket, &fromAppInfo, sizeof(AppInfoPacket)));
            ::std::memcpy(&toAppInfo, fromAppInfo.detailInfo, ApplicationDetailInfoSize);

            NN_LCS_LOG_DEBUG("id : %llx\n", toAppInfo.id);
            NN_LCS_LOG_DEBUG("displayVersion : %s\n", toAppInfo.displayVersion);
            NN_LCS_LOG_DEBUG("requiredSize : %lld\n", toAppInfo.requiredSize);

            int appIndex = 0;
            NN_ABORT_UNLESS_LESS_EQUAL(m_pResource->sessionInfo.contentsCount, SharableContentsCountMax);
            while (appIndex < m_pResource->sessionInfo.contentsCount)
            {
                NN_LCS_LOG_DEBUG("toAppInfo.id : %llx\n", toAppInfo.id);
                NN_LCS_LOG_DEBUG("m_SessionSettings.value : %llx\n", m_pResource->sessionInfo.contents[appIndex].applicationId.value);
                if (toAppInfo.id == m_pResource->sessionInfo.contents[appIndex].applicationId.value)
                {
                    break;
                }
                ++appIndex;
            }
            if(appIndex == m_pResource->sessionInfo.contentsCount)
            {
                NN_ABORT("AppInfo id Error : %s\n", __FUNCTION__);
            }

            ::std::memcpy(&detail.shareApp[appIndex].info.detailInfo, &toAppInfo, ApplicationDetailInfoSize);
            NN_ABORT_UNLESS_LESS_EQUAL(fromAppInfo.appDeliveryInfoCount, ApplicationDeliveryInfoCountMax);
            detail.shareApp[appIndex].info.deliveryInfoCount = fromAppInfo.appDeliveryInfoCount;
            NN_LCS_LOG_DEBUG("deliveryInfoCount : %d\n", detail.shareApp[appIndex].info.deliveryInfoCount);
            for (int j = 0; j < detail.shareApp[appIndex].info.deliveryInfoCount; ++j)
            {
                NN_RESULT_DO(WaitRecvData(
                    socket, &detail.shareApp[appIndex].info.deliveryInfo[j], ApplicationDeliveryInfoSize));
            }
            detail.shareApp[appIndex].hasApp = false;
        }

        // 互換性がある GetSystemDeliveryInfo() かを確認
        auto result = VerifyDeliveryProtocolVersion(detail.sysInfo);
        if (nn::ns::ResultInvalidSystemDeliveryProtocolVersion().Includes(result))
        {
            // プロトコルバージョンが 1.2 より小さい場合は親の Result を返す
            if (GetMajorVersion(detail.version) == 1 && GetMinorVersion(detail.version) < 2)
            {
                return ResultIncompatibleSystemVersion();
            }
            else
            {
                if (nn::ns::ResultOldSystemDeliveryProtocolVersion().Includes(result))
                {
                    return ResultLowerSystemVersion();
                }
                else if (nn::ns::ResultUnknownSystemDeliveryProtocolVersion().Includes(result))
                {
                    return ResultHigherSystemVersion();
                }
                else
                {
                    NN_LCS_LOG_DEBUG("VerifyDeliveryProtocolVersion Error\n");
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                }
            }
        }
        else if (nn::ns::ResultInvalidApplicationDeliveryProtocolVersion().Includes(result) ||
                 nn::ns::ResultSystemUpdateRequiredForLocalContentDelivery().Includes(result))
        {
            // AppDeliveryInfo の互換切れはあとで LUP するので、何もせず接続させる。
            // LcsRequiredSystemVersion 切れも必要に応じて LUP するので、何もせずに接続させる。
        }
        else
        {
            NN_LCS_LOG_DEBUG("VerifyDeliveryProtocolVersion Error\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }

        // 端末の登録
        if (!m_NodeManager.AddNodeDetail(detail))
        {
            NN_LCS_LOG_DEBUG("AddNodeDetail Error\n");
            return ResultConnectionFailed();
        }
        NN_RESULT_SUCCESS;
    }

    void Host::SendRejectToAll(RejectReason reason) NN_NOEXCEPT
    {
        int nodeCount = 0;
        NodeInfo info[NodeCountMax] = {};
        m_NodeManager.GetNodeInfo(&nodeCount, info, NodeCountMax);
        for (int i = 0; i < nodeCount; ++i)
        {
            RejectReason sendReason = reason;
            NodeDetailInfo detailInfo = {};
            if (m_NodeManager.GetNodeDetailInfo(&detailInfo, info[i].index).IsSuccess())
            {
                // プロトコルバージョンが 1.1 より小さい場合は RejectReason を変換して送信
                if (GetMajorVersion(detailInfo.version) <= 1 && GetMinorVersion(detailInfo.version) < 1)
                {
                    if (sendReason == RejectReason_UnshareableContents)
                    {
                        sendReason = RejectReason_NotExistDownloadContents;
                    }
                    else if (sendReason == RejectReason_Unknown)
                    {
                        sendReason = RejectReason_NodeLeaved;
                    }
                }
                if (detailInfo.ipAddress != m_MyAddress)
                {
                    NN_LCS_LOG_DEBUG("SendReject to %s[%x] reason : %d\n",
                        detailInfo.nodeInfo.userName, detailInfo.nodeInfo.index, sendReason);
                    if (detailInfo.socket > 0)
                    {
                        SendRejectNode(detailInfo.socket, sendReason);
                    }
                }
            }
        }
    }

    void Host::SendSessionInfoToAll() NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("SendSessionInfoToAll\n");
        int nodeCount = 0;
        NodeInfo info[NodeCountMax] = {};
        m_NodeManager.GetConnectedNodeInfo(&nodeCount, info, NodeCountMax);
        NN_LCS_LOG_DEBUG("NodeCount : %d\n", nodeCount);
        for (int i = 0; i < nodeCount; ++i)
        {
            NodeDetailInfo detailInfo = {};
            if (m_NodeManager.GetNodeDetailInfo(&detailInfo, info[i].index).IsSuccess())
            {
                if (detailInfo.ipAddress != m_MyAddress && detailInfo.state != NodeInnerState_Temporary)
                {
                    NN_LCS_LOG_DEBUG("SendSessionInfo to %s[%x]\n",
                        detailInfo.nodeInfo.userName, detailInfo.nodeInfo.index);
                    if (detailInfo.socket > 0)
                    {
                        SendSessionInfo(detailInfo.socket,
                            m_pResource->shareAppInfoCount, m_pResource->shareAppInfo, nodeCount, info);
                    }
                }
            }
        }
    }

    void Host::SendStartToAll() NN_NOEXCEPT
    {
        int nodeCount = 0;
        NodeInfo info[NodeCountMax] = {};
        m_NodeManager.GetNodeInfo(&nodeCount, info, NodeCountMax);
        for (int i = 0; i < nodeCount; ++i)
        {
            NodeDetailInfo detailInfo = {};
            if (m_NodeManager.GetNodeDetailInfo(&detailInfo, info[i].index).IsSuccess())
            {
                if (detailInfo.ipAddress != m_MyAddress)
                {
                    NN_LCS_LOG_DEBUG("SendStart\n");
                    SendShareStart(detailInfo.socket);
                }
            }
        }
    }

    void Host::SendEndContentsSareToAll() NN_NOEXCEPT
    {
        int nodeCount = 0;
        NodeInfo info[NodeCountMax] = {};
        m_NodeManager.GetNodeInfo(&nodeCount, info, NodeCountMax);
        for (int i = 0; i < nodeCount; ++i)
        {
            NodeDetailInfo detailInfo = {};
            if (m_NodeManager.GetNodeDetailInfo(&detailInfo, info[i].index).IsSuccess())
            {
                if (detailInfo.ipAddress != m_MyAddress)
                {
                    // プロトコルバージョンが 1.2 より小さい場合は EndContentsSarePacket を送信
                    if (GetMajorVersion(detailInfo.version) == 1 && GetMinorVersion(detailInfo.version) < 2)
                    {
                        NN_LCS_LOG_DEBUG("SendEnd to %x\n", detailInfo.ipAddress);
                        SendShareEnd(detailInfo.socket);
                    }
                    else
                    {
                        NN_LCS_LOG_DEBUG("SendEndWithProgress to %x\n\n", detailInfo.ipAddress);
                        AdvertiseData data = {};
                        m_AdvertiseManager.GetAdvertiseData(&data);
                        SendShareEndWithProgress(detailInfo.socket, data);
                    }
                }
            }
        }
    }

    void Host::MigrationFunction(uint32_t newHostIndex) NN_NOEXCEPT
    {
        // データを準備
        int nodeCount = 0;
        NodeInfo nodeInfo[NodeCountMax] = {};
        NodeDetailInfo detailNodeInfo = {};
        int count = 0;

        // 持ってるデータを更新保存
        UpdateSessionInfo();
        UpdateNodesInfo();
        UpdateRequiredStorageSize();

        ::std::memcpy(&(m_pResource->migrationInfo.newNetworkSetting),
            m_pResource->sessionInfo._internal, NetworkSettingSize);
        // パラメータをちょっといじる
        char random;
        nn::os::GenerateRandomBytes(&random, sizeof(random));
        m_pResource->migrationInfo.newNetworkSetting.securityParam[0] = random;
        nn::os::GenerateRandomBytes(&random, sizeof(random));
        m_pResource->migrationInfo.newNetworkSetting.securityParam[16] = random;

        ::std::memcpy(&(m_pResource->currentNetworkSetting),
            &(m_pResource->migrationInfo.newNetworkSetting), NetworkSettingSize);

        m_NodeManager.GetNodeInfo(&nodeCount, nodeInfo, NodeCountMax);
        for (int i = 0; i < nodeCount; ++i)
        {
            if (m_NodeManager.GetNodeDetailInfo(&detailNodeInfo, nodeInfo[i].index).IsSuccess())
            {
                m_pResource->migrationInfo.nodeEntryInfo[i].index = detailNodeInfo.nodeInfo.index;
                m_pResource->migrationInfo.nodeEntryInfo[i].nodeAddressEntry.ipv4Address = detailNodeInfo.ipAddress;
                ::std::memcpy(&(m_pResource->migrationInfo.nodeEntryInfo[i].nodeAddressEntry.macAddress),
                    &detailNodeInfo.macAddress, sizeof(MacAddress));
                ::std::strncpy(m_pResource->migrationInfo.nodeEntryInfo[i].userName,
                    detailNodeInfo.nodeInfo.userName, UserNameBytesMax);
                m_pResource->migrationInfo.nodeEntryInfo[i].userName[UserNameBytesMax] = 0;
                // クライアントの状態を NodeInnerState_Rejoining にする
                m_NodeManager.SetInnerState(NodeInnerState_Rejoining, nodeInfo[i].index);
                ++count;
            }
        }
        m_pResource->migrationInfo.nodeCount = count;

        NN_LCS_LOG_DEBUG("--- MigrationInfo ----\n");
        NN_LCS_LOG_DEBUG("NodeCount : %d\n", m_pResource->migrationInfo.nodeCount);
        for (int i = 0; i < m_pResource->migrationInfo.nodeCount; ++i)
        {
            NN_LCS_LOG_DEBUG(" Index : %x\n", m_pResource->migrationInfo.nodeEntryInfo[i].index);
            NN_LCS_LOG_DEBUG(" IpAddress : %x\n", m_pResource->migrationInfo.nodeEntryInfo[i].nodeAddressEntry.ipv4Address);
        }
        NN_LCS_LOG_DEBUG(" --securityParam--\n");
        for (int i = 0; i < 32; ++i)
        {
            NN_LCS_LOG_DEBUG("%x\n", m_pResource->migrationInfo.newNetworkSetting.securityParam[i]);
        }
        NN_LCS_LOG_DEBUG("--- MigrationInfo End ----\n");

        // Host になる端末に連絡
        m_pResource->migrationInfo.role = LcsRole_Host;
        NN_LCS_LOG_DEBUG("SendMigration : %x\n", newHostIndex);
        SendHostMigration(m_NodeManager.GetNodeSocket(newHostIndex), m_pResource->migrationInfo);

        // 残りに連絡
        m_pResource->migrationInfo.role = LcsRole_Client;
        for (int i = 0; i < nodeCount; ++i)
        {
            if (nodeInfo[i].index != newHostIndex && nodeInfo[i].index != m_NodeManager.GetNodeIndex(m_MyAddress))
            {
                NN_LCS_LOG_DEBUG("SendMigration : %x\n", nodeInfo[i].index);
                SendHostMigration(m_NodeManager.GetNodeSocket(nodeInfo[i].index), m_pResource->migrationInfo);
            }
        }
        // 自分の内部状態を Suspending に変更
        m_NodeManager.SetInnerState(NodeInnerState_Suspending, m_NodeManager.GetNodeIndex(m_MyAddress));

        NN_LCS_LOG_DEBUG("Wait Client Disconnect\n");
        while (m_IsOpened)
        {
            int connectedNodesCount = 0;
            if (GetConnectedNodesCount(&connectedNodesCount).IsFailure())
            {
                m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
                ChangeState(State_ContentsShareFailed);
            }
            NN_LCS_LOG_DEBUG("NodeCount : %d\n", connectedNodesCount);
            if (connectedNodesCount == 1)
            {
                break;
            }
            nn::os::TimedWaitEvent(&m_ClientStateChangeEvent,
                nn::TimeSpan::FromMilliSeconds(WaitDisconnectIntervalMilliSeconds));
        }
        NN_LCS_LOG_DEBUG("Start Migration\n");
        // MigrationThread に連絡
        nn::os::SignalEvent(&(m_pResource->migrationEvent));
    }

    bool Host::ConfirmSystemUpdateSchedule() NN_NOEXCEPT
    {
        RejectNodeInfo rejectInfo[NodeCountMax] = {};
        int rejectCount = 0;

        for (int i = 0; i < m_pResource->shareAppInfoCount; ++i)
        {
            if (m_NodeManager.MakeSystemUpdateSchedule(
                &rejectCount, rejectInfo, NodeCountMax, m_MyAddress))
            {
                // 受信ができない端末を切断
                NN_LCS_LOG_DEBUG("--- Reject List : %d ---\n", rejectCount);
                for (int j = 0; j < rejectCount; ++j)
                {
                    NN_LCS_LOG_DEBUG("Index : %x / Reason : %d\n", rejectInfo[i].index, rejectInfo[i].reason);

                    if (rejectInfo[j].reason == ContentsShareFailureReason_NodeLeaved)
                    {
                        Reject(rejectInfo[j].index, ContentsShareFailureReason_NodeLeaved);
                    }
                    else if (rejectInfo[j].reason == ContentsShareFailureReason_CannotUpdateSystem)
                    {
                        Reject(rejectInfo[j].index, ContentsShareFailureReason_CannotUpdateSystem);
                    }
                    else
                    {
                        NN_ABORT("Invalid ContentsShareFailureReason!\n");
                    }
                }
                NN_LCS_LOG_DEBUG("--- Reject List End ---\n");
            }
            else
            {
                // 必須がいない時は false
                return false;
            }
        }
        UpdateAdvertiseProgress();

        return true;
    }

    Result Host::ConfirmContentShareSchedule() NN_NOEXCEPT
    {
        RejectNodeInfo rejectInfo[NodeCountMax] = {};
        int rejectCount = 0;

        for (int i = 0; i < m_pResource->shareAppInfoCount; ++i)
        {
            NN_RESULT_DO(m_NodeManager.MakeContentShareSchedule(
                &rejectCount, rejectInfo, NodeCountMax, m_pResource->shareAppInfo[i], i, m_MyAddress));

            // 受信ができない端末を切断
            NN_LCS_LOG_DEBUG("--- Reject List : %d ---\n", rejectCount);
            for (int j = 0; j < rejectCount; ++j)
            {
                NN_LCS_LOG_DEBUG("Index : %x / Reason : %d\n", rejectInfo[i].index, rejectInfo[i].reason);
                // 自身がパッチを受信できないときの処理
                if (rejectInfo[j].index == m_NodeManager.GetNodeIndex(m_MyAddress))
                {
                    if (rejectInfo[j].reason == ContentsShareFailureReason_NodeLeaved)
                    {
                        //  必須端末がいないから解散させる
                        m_pResource->contentsShareFailureReason = ContentsShareFailureReason_NodeLeaved;
                        ChangeState(State_ContentsShareFailed);
                    }
                    else if (rejectInfo[j].reason == ContentsShareFailureReason_CannotUpdateSystem)
                    {
                        // Master がシステム更新を求められることはないのでアボートさせる
                        NN_ABORT("Master need not update Systems\n");
                    }
                    else if (rejectInfo[j].reason == ContentsShareFailureReason_NotExistDownloadContents)
                    {
                        // とりあえず、受信できずで失敗することを設定。State_Comp になる際に失敗させる
                        m_pResource->contentsShareFailureReason = ContentsShareFailureReason_NotExistDownloadContents;
                        m_NodeManager.ClearRequireDownload(rejectInfo[j].index);
                    }
                }
                else
                {
                    if (rejectInfo[j].reason == ContentsShareFailureReason_NodeLeaved)
                    {
                        Reject(rejectInfo[j].index, ContentsShareFailureReason_NodeLeaved);
                    }
                    else if (rejectInfo[j].reason == ContentsShareFailureReason_CannotUpdateSystem)
                    {
                        Reject(rejectInfo[j].index, ContentsShareFailureReason_CannotUpdateSystem);
                    }
                    else if (rejectInfo[j].reason == ContentsShareFailureReason_NotExistDownloadContents)
                    {
                        Reject(rejectInfo[j].index, ContentsShareFailureReason_NotExistDownloadContents);
                    }
                    else
                    {
                        NN_ABORT("Invalid ContentsShareFailureReason!\n");
                    }
                }
            }
            NN_LCS_LOG_DEBUG("--- Reject List End ---\n");
        }
#ifdef NN_LCS_BUILD_CONFIG_DEBUG_LOG_LEVEL_DEBUG
        NN_LCS_LOG_DEBUG("-----///// ConfirmContentShareSchedule /////-----\n");
        int nodeCount = 0;
        NodeInfo info[NodeCountMax] = {};
        m_NodeManager.GetNodeInfo(&nodeCount, info, NodeCountMax);
        NN_LCS_LOG_DEBUG("NodeCount : %d\n", nodeCount);
        for (int i = 0; i < nodeCount; ++i)
        {
            NodeDetailInfo detailInfo = {};
            if (m_NodeManager.GetNodeDetailInfo(&detailInfo, info[i].index).IsSuccess())
            {
                NN_LCS_LOG_DEBUG("%s[Index : %x / IP : %x]\n",
                    detailInfo.nodeInfo.userName, detailInfo.nodeInfo.index, detailInfo.ipAddress);
                NN_LCS_LOG_DEBUG(" SystemUpdate : %d / From : %x\n",
                    detailInfo.requireSystemDownload.isRequire,
                    detailInfo.requireSystemDownload.formAddress);
                for (int j = 0; j < m_pResource->sessionInfo.contentsCount; ++j)
                {
                    NN_LCS_LOG_DEBUG(" App%dUpdate : %d / From : %x\n", j,
                        detailInfo.shareApp[j].requireAppDownload.isRequire,
                        detailInfo.shareApp[j].requireAppDownload.formAddress);
                }
            }
        }
        NN_LCS_LOG_DEBUG("-----/// ConfirmContentShareSchedule End ///-----\n");
#endif // NN_LCS_BUILD_CONFIG_DEBUG_LOG_LEVEL_DEBUG

        UpdateAdvertiseProgress();

        NN_RESULT_SUCCESS;
    }

    void Host::UpdateHostDetailInfo(Bit64 id) NN_NOEXCEPT
    {
        NodeDetailInfo detailInfo = {};
        uint32_t myIndex = m_NodeManager.GetNodeIndex(m_MyAddress);
        if (m_NodeManager.GetNodeDetailInfo(&detailInfo, myIndex).IsSuccess())
        {
            GetSystemDeliveryInfo(&detailInfo.sysInfo);
            for (int i = 0; i < m_pResource->sessionInfo.contentsCount; ++i)
            {
                GetApplicationDetailInfo(&detailInfo.shareApp[i].info.detailInfo,
                    m_pResource->appControlDataBuffer, AppControlDataSize,
                    m_pResource->sessionInfo.contents[i].applicationId.value);
                GetApplicationDeliveryInfo(
                    &detailInfo.shareApp[i].info.deliveryInfoCount, detailInfo.shareApp[i].info.deliveryInfo,
                    ApplicationDeliveryInfoCountMax, m_pResource->sessionInfo.contents[i].applicationId.value);
            }
            ++detailInfo.downloadedCount;

            m_NodeManager.AddNodeDetail(detailInfo);
        }
    }

    bool Host::ShareSystem(uint32_t receiverIndex, SystemUpdateReason reason) NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("ShareSystem\n");

        NodeDetailInfo detailInfo = {};

        NN_LCS_LOG_DEBUG("ReceiverIndex %x\n", receiverIndex);
        if (m_NodeManager.GetNodeDetailInfo(&detailInfo, receiverIndex).IsFailure())
        {
            return false;
        }
        NN_LCS_LOG_DEBUG("Name : %s\n", detailInfo.nodeInfo.userName);

        if (detailInfo.requireSystemDownload.isRequire)
        {
            Result result;

            NN_LCS_LOG_DEBUG("From %x to %x / MyIP %x\n",
                detailInfo.requireSystemDownload.formAddress,
                detailInfo.ipAddress, m_MyAddress);

            // Host -> Client
            if (detailInfo.requireSystemDownload.formAddress == m_MyAddress &&
                detailInfo.ipAddress != m_MyAddress)
            {
                NN_LCS_LOG_DEBUG("I'm SystemSender to %x\n", detailInfo.ipAddress);

                // SystemDeliveryInfo を取得
                GetSystemDeliveryInfo(&m_pResource->shareComponent.systemDeliveryInfo);

                // 受信者へ受信するか確認連絡
                if (SendShareSystemClear(m_NodeManager.GetNodeSocket(detailInfo.nodeInfo.index), reason).IsFailure())
                {
                    NN_LCS_LOG_DEBUG("SendShareSystemClear Error %x\n", detailInfo.nodeInfo.index);
                    Reject(detailInfo.nodeInfo.index, ContentsShareFailureReason_CommunicationError);
                    return false;
                }

                // 受信者からの返事待ち
                if (!WaitClientResponse())
                {
                    NN_LCS_LOG_DEBUG("WaitClientResponse Error %x\n", detailInfo.nodeInfo.index);
                    Reject(detailInfo.nodeInfo.index, ContentsShareFailureReason_CommunicationError);
                    return false;
                }

                // 受信者が Suspend か確認。
                if (m_NodeManager.IsInnerState(detailInfo.nodeInfo.index, NodeInnerState_Suspending))
                {
                    return false;
                }

                // 送信タスクの生成
                nn::ns::AsyncResult asyncResult;
                RequestSendSystemUpdate(&asyncResult, detailInfo.ipAddress, SharePort,
                    m_pResource->shareComponent.systemDeliveryInfo);

                // 受信者へ連絡
                if (SendShareSystemRecvReq(m_NodeManager.GetNodeSocket(detailInfo.nodeInfo.index), TransferRole_Receiver,
                    detailInfo.requireSystemDownload.formAddress, detailInfo.requireSystemDownload.index,
                    m_pResource->shareComponent.systemDeliveryInfo, 0).IsFailure())
                {
                    NN_LCS_LOG_DEBUG("SendShareSystemClear Error %x\n", detailInfo.nodeInfo.index);
                    asyncResult.Cancel();
                    Reject(detailInfo.nodeInfo.index, ContentsShareFailureReason_CommunicationError);
                    return false;
                }

                // 受信者からの返事待ち
                if (!WaitClientResponse())
                {
                    NN_LCS_LOG_DEBUG("WaitClientResponse Error %x\n", detailInfo.nodeInfo.index);
                    asyncResult.Cancel();
                    Reject(detailInfo.nodeInfo.index, ContentsShareFailureReason_CommunicationError);
                    return false;
                }
                // 受信者の状態を変更する
                m_NodeManager.SetInnerState(NodeInnerState_Transferring, detailInfo.nodeInfo.index);

                // 進捗用の設定
                m_Role = TransferRole_Sender;
                ::std::memset(&m_pResource->contentInfo[m_pResource->contentInfoCount], 0, sizeof(ContentsInfo));
                m_pResource->contentInfo[m_pResource->contentInfoCount].contentsFlag.Reset();
                m_pResource->contentInfo[m_pResource->contentInfoCount].contentsFlag.Set(ContentsType::System::Index);
                // 受信者状態確認用の設定
                m_IsWaitReceiver = true;
                m_NodeIndex = detailInfo.nodeInfo.index;

                // 自身が送信者であることを設定
                m_IsSender = true;

                // 送信完了待ち
                result = WaitTask(&asyncResult, m_NodeIndex);
                if (result.IsFailure())
                {
                    asyncResult.Cancel();
                    m_Role = TransferRole_None;
                    if (ResultAsyncResultError::Includes(result))
                    {
                        ChangeState(State_ContentsShareFailed);
                        return false;
                    }
                    else if (ResultCommunicationError::Includes(result) ||
                        ResultRequestNodeLeaved::Includes(result) || ResultUserCancel::Includes(result))
                    {
                        NN_LCS_LOG_DEBUG("Send System Failed : %08x\n", result.GetInnerValueForDebug());
                        return true;
                    }
                    else
                    {
                        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                    }
                }
                uint64_t sendSize = 0;
                GetSendSystemUpdateProgress(&sendSize, &m_TotalSendSize);
            }
            // Client -> Host
            else if (detailInfo.requireSystemDownload.formAddress != m_MyAddress &&
                detailInfo.ipAddress == m_MyAddress)
            {
                // !!!*** Host は最新になっているので発生することはない ***!!!
                NN_ABORT("Host is Receiver!\n"); // 発生することはないので ABORT させる
            }
            // Client -> Client
            else if (detailInfo.requireSystemDownload.formAddress != m_MyAddress &&
                detailInfo.ipAddress != m_MyAddress)
            {
                NN_LCS_LOG_DEBUG("I'm SystemObserver  from %x to %x\n",
                    detailInfo.requireSystemDownload.formAddress, detailInfo.ipAddress);

                // 受信者へ受信するか確認連絡
                if (SendShareSystemClear(m_NodeManager.GetNodeSocket(detailInfo.nodeInfo.index), reason).IsFailure())
                {
                    NN_LCS_LOG_DEBUG("SendShareSystemClear Error %x\n", detailInfo.nodeInfo.index);
                    Reject(detailInfo.nodeInfo.index, ContentsShareFailureReason_CommunicationError);
                    return false;
                }

                // 受信者からの返事待ち
                if (!WaitClientResponse())
                {
                    NN_LCS_LOG_DEBUG("WaitClientResponse Error %x\n", detailInfo.nodeInfo.index);
                    Reject(detailInfo.nodeInfo.index, ContentsShareFailureReason_CommunicationError);
                    return false;
                }

                // 受信者が Suspend か確認。
                if (m_NodeManager.IsInnerState(detailInfo.nodeInfo.index, NodeInnerState_Suspending))
                {
                    return false;
                }

                // 送信者へ連絡
                if (SendShareSystemSendReq(m_NodeManager.GetNodeSocket(detailInfo.requireSystemDownload.index),
                    TransferRole_Sender, detailInfo.ipAddress, detailInfo.nodeInfo.index, 1).IsFailure())
                {
                    NN_LCS_LOG_DEBUG("SendShareSystemClear Error %x\n", detailInfo.requireSystemDownload.index);
                    Reject(detailInfo.requireSystemDownload.index, ContentsShareFailureReason_CommunicationError);
                    return false;
                }

                // 送信者からの返事待ち
                if (!WaitClientResponse())
                {
                    NN_LCS_LOG_DEBUG("WaitClientResponse Error %x\n", detailInfo.requireSystemDownload.index);
                    Reject(detailInfo.requireSystemDownload.index, ContentsShareFailureReason_CommunicationError);
                    return false;
                }
                // 送信者の状態を変更する
                m_NodeManager.SetInnerState(NodeInnerState_Transferring, detailInfo.requireSystemDownload.index);

                // 受信者へ連絡
                if (SendShareSystemRecvReq(m_NodeManager.GetNodeSocket(detailInfo.nodeInfo.index),
                    TransferRole_Receiver, detailInfo.requireSystemDownload.formAddress,
                    detailInfo.requireSystemDownload.index, m_pResource->shareComponent.systemDeliveryInfo, 1).IsFailure())
                {
                    NN_LCS_LOG_DEBUG("SendShareSystemClear Error %x\n", detailInfo.nodeInfo.index);
                    // ToDo : 送信者にキャンセルを連絡
                    Reject(detailInfo.nodeInfo.index, ContentsShareFailureReason_CommunicationError);
                    return false;
                }

                // 受信者からの返事待ち
                if (!WaitClientResponse())
                {
                    NN_LCS_LOG_DEBUG("WaitClientResponse Error %x\n", detailInfo.nodeInfo.index);
                    // ToDo : 送信者にキャンセルを連絡
                    Reject(detailInfo.nodeInfo.index, ContentsShareFailureReason_CommunicationError);
                    return false;
                }
                // 受信者の状態を変更する
                m_NodeManager.SetInnerState(NodeInnerState_Transferring, detailInfo.nodeInfo.index);
                // 受信者状態確認用の設定
                m_IsWaitReceiver = true;
                m_NodeIndex = detailInfo.nodeInfo.index;
            }

            m_Role = TransferRole_None;

            // 転送完了待ち(タイムアウト作るかは要検討)
            while (m_IsOpened)
            {
                NN_LCS_LOG_DEBUG("SystemReceiver Exsist : %d\n",
                    m_NodeManager.IsExsistNode(detailInfo.requireSystemDownload.index));
                NN_LCS_LOG_DEBUG("SystemReceiver State : %d\n",
                    m_NodeManager.IsInnerState(detailInfo.requireSystemDownload.index, NodeInnerState_Transferring));
                NN_LCS_LOG_DEBUG("SystemSender Exsist : %d\n",
                    m_NodeManager.IsExsistNode(detailInfo.nodeInfo.index));
                NN_LCS_LOG_DEBUG("SystemSender State : %d\n",
                    m_NodeManager.IsInnerState(detailInfo.nodeInfo.index, NodeInnerState_Transferring));
                if ((!m_NodeManager.IsExsistNode(detailInfo.requireSystemDownload.index) ||
                     !m_NodeManager.IsInnerState(detailInfo.requireSystemDownload.index, NodeInnerState_Transferring)) &&
                    (!m_NodeManager.IsExsistNode(detailInfo.nodeInfo.index) ||
                     !m_NodeManager.IsInnerState(detailInfo.nodeInfo.index, NodeInnerState_Transferring)))
                {
                    NN_LCS_LOG_DEBUG("Complete to Transfer System / %x to %x\n",
                        detailInfo.requireSystemDownload.index, detailInfo.nodeInfo.index);
                    break;
                }
                nn::os::WaitEvent(&m_ClientStateChangeEvent);
            }
            // システムを受信した端末の状態を変更
            m_NodeManager.SetInnerState(NodeInnerState_Rebooting, detailInfo.nodeInfo.index);
            m_NodeIndex = 0;
            m_IsSender = false;
            m_IsWaitReceiver = false;

            // Client に完了したことを伝えるために情報送信
            SendSessionInfoToAll();

            UpdateAdvertiseProgress();
        }
        return false;
    }// NOLINT(impl/function_size)

    bool Host::ShareContent(uint32_t receiverIndex, int appIndex) NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("ShareContent\n");

        NodeDetailInfo detailInfo = {};

        NN_LCS_LOG_DEBUG("ReceiverIndex %x / appIndex : %d\n", receiverIndex, appIndex);
        if (m_NodeManager.GetNodeDetailInfo(&detailInfo, receiverIndex).IsFailure())
        {
            return true;
        }
        NN_LCS_LOG_DEBUG("Name : %s\n", detailInfo.nodeInfo.userName);

        if (detailInfo.shareApp[appIndex].requireAppDownload.isRequire)
        {
            Result result;

            NN_LCS_LOG_DEBUG("From %x to %x / MyIP %x\n",
                detailInfo.shareApp[appIndex].requireAppDownload.formAddress,
                detailInfo.ipAddress, m_MyAddress);

            // 送受信の情報を設定
            m_pResource->shareComponent.id = m_pResource->sessionInfo.contents[appIndex].applicationId.value;
            m_pResource->shareComponent.contentsFlag = m_pResource->sessionInfo.contents[appIndex].contentsFlag;

            // Host -> Client
            if (detailInfo.shareApp[appIndex].requireAppDownload.formAddress == m_MyAddress &&
                detailInfo.ipAddress != m_MyAddress)
            {
                NN_LCS_LOG_DEBUG("I'm AppSender to %x\n", detailInfo.ipAddress);

                // 受信者へ受信できるか確認連絡
                if (SendShareContentClear(m_NodeManager.GetNodeSocket(detailInfo.nodeInfo.index),
                    m_pResource->shareComponent.contentsFlag, m_pResource->shareComponent.id).IsFailure())
                {
                    NN_LCS_LOG_DEBUG("SendShareContentClear Error %x\n", detailInfo.nodeInfo.index);
                    Reject(detailInfo.nodeInfo.index, ContentsShareFailureReason_CommunicationError);
                    return true;
                }

                // 受信者からの返事待ち
                if (!WaitClientResponse())
                {
                    NN_LCS_LOG_DEBUG("WaitClientResponse Error %x\n", detailInfo.nodeInfo.index);
                    Reject(detailInfo.nodeInfo.index, ContentsShareFailureReason_CommunicationError);
                    return true;
                }

                // 受信者が Suspend かを見る
                if (m_NodeManager.IsInnerState(detailInfo.nodeInfo.index, NodeInnerState_Suspending))
                {
                    return true;
                }

                // 送信するアプリの状態が変化していないか確認
                for (int i = 0; i < m_pResource->ownAppInfoCount; ++i)
                {
                    if (m_pResource->ownAppInfo[i].detailInfo.id == m_pResource->shareComponent.id)
                    {
                        if (ConfirmSendApplication(m_pResource->ownAppInfo[i]).IsFailure())
                        {
                            NN_LCS_LOG_DEBUG("ConfirmSendApplication Error\n");
                            m_pResource->contentsShareFailureReason = ContentsShareFailureReason_StorageError;
                            ChangeState(State_ContentsShareFailed);
                            return false;
                        }
                    }
                }

                // 送信タスクの生成
                nn::ns::AsyncResult asyncResult;
                if (RequestSendApplication(&asyncResult,
                    detailInfo.ipAddress, SharePort, m_pResource->shareComponent.id).IsFailure())
                {
                    NN_LCS_LOG_DEBUG("RequestSendApplication Error\n");
                    m_pResource->contentsShareFailureReason = ContentsShareFailureReason_StorageError;
                    ChangeState(State_ContentsShareFailed);
                    return false;
                }

                // 送信する ContentMetaKey の生成
                if (ListContentMetaKeyToDeliverApplication(
                    &m_pResource->shareComponent.contentMetaKeyCount, m_pResource->shareComponent.contentMetaKey,
                    ContentMetaKeyMax, m_pResource->shareComponent.id).IsFailure())
                {
                    NN_LCS_LOG_DEBUG("ListContentMetaKeyToDeliverApplication Error\n");
                    asyncResult.Cancel();
                    m_pResource->contentsShareFailureReason = ContentsShareFailureReason_StorageError;
                    ChangeState(State_ContentsShareFailed);
                    return false;
                }

                // 受信者へ連絡
                if (SendShareContentRecvReq(m_NodeManager.GetNodeSocket(detailInfo.nodeInfo.index),
                    TransferRole_Receiver,
                    detailInfo.shareApp[appIndex].requireAppDownload.formAddress,
                    detailInfo.shareApp[appIndex].requireAppDownload.index,
                    m_pResource->shareComponent.id,
                    m_pResource->shareComponent.contentMetaKey, m_pResource->shareComponent.contentMetaKeyCount, 0).IsFailure())
                {
                    NN_LCS_LOG_DEBUG("SendShareContentRecvReq Error %x\n", detailInfo.nodeInfo.index);
                    asyncResult.Cancel();
                    Reject(detailInfo.nodeInfo.index, ContentsShareFailureReason_CommunicationError);
                    return true;
                }

                // 受信者からの返事待ち
                if (!WaitClientResponse())
                {
                    NN_LCS_LOG_DEBUG("WaitClientResponse Error %x\n", detailInfo.nodeInfo.index);
                    asyncResult.Cancel();
                    Reject(detailInfo.nodeInfo.index, ContentsShareFailureReason_CommunicationError);
                    return true;
                }
                // 受信者の状態を変更する
                m_NodeManager.SetInnerState(NodeInnerState_Transferring, detailInfo.nodeInfo.index);

                // 進捗用の設定
                m_Role = TransferRole_Sender;
                ::std::memcpy(&m_pResource->contentInfo[m_pResource->contentInfoCount], &m_pResource->sessionInfo.contents[appIndex], sizeof(ContentsInfo));
                // 受信者状態確認用の設定
                m_IsWaitReceiver = true;
                m_NodeIndex = detailInfo.nodeInfo.index;

                // 自身が送信者であることを設定
                m_IsSender = true;

                // 送信完了待ち
                result = WaitTask(&asyncResult, m_NodeIndex);
                if (result.IsFailure())
                {
                    asyncResult.Cancel();
                    m_Role = TransferRole_None;
                    if (ResultAsyncResultError::Includes(result))
                    {
                        ChangeState(State_ContentsShareFailed);
                        return false;
                    }
                    else if (ResultCommunicationError::Includes(result) ||
                        ResultRequestNodeLeaved::Includes(result) || ResultUserCancel::Includes(result))
                    {
                        NN_LCS_LOG_DEBUG("Send App Failed : %08x\n", result.GetInnerValueForDebug());
                        return true;
                    }
                    else
                    {
                        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                    }
                }
                uint64_t sendSize = 0;
                GetSendApplicationProgress(&sendSize, &m_TotalSendSize,
                    m_pResource->contentInfo[m_pResource->contentInfoCount].applicationId.value);
            }
            // Client -> Host
            else if (detailInfo.shareApp[appIndex].requireAppDownload.formAddress != m_MyAddress &&
                detailInfo.ipAddress == m_MyAddress)
            {
                NN_LCS_LOG_DEBUG("I'm AppReceiver from %x\n",
                    detailInfo.shareApp[appIndex].requireAppDownload.formAddress);

                for (int i = 0; i < m_pResource->shareAppInfoCount; ++i)
                {
                    if (m_pResource->shareAppInfo[i].detailInfo.id == m_pResource->shareComponent.id)
                    {
                        m_RequiredSize = m_pResource->shareAppInfo[i].detailInfo.requiredSize;
                        NN_LCS_LOG_DEBUG("Current m_RequiredSize : %lld\n", m_RequiredSize);
                    }
                }

                if (!HasFreeSpaceSize(m_RequiredSize))
                {
                    m_pResource->suspendedReason = SuspendedReason_StorageSpaceNotEnoughOnRequredNode;
                    ChangeState(State_Suspended);
                    nn::os::WaitEvent(&m_ResumeEvent);
                    if (!m_IsOpened)
                    {
                        return true;
                    }
                }

                // 送信者へ連絡
                if (SendShareContentSendReq(
                    m_NodeManager.GetNodeSocket(detailInfo.shareApp[appIndex].requireAppDownload.index),
                    TransferRole_Sender, detailInfo.ipAddress,
                    detailInfo.nodeInfo.index, m_pResource->shareComponent.id, 0).IsFailure())
                {
                    NN_LCS_LOG_DEBUG("SendShareContentSendReq Error %x\n", detailInfo.shareApp[appIndex].requireAppDownload.index);
                    Reject(detailInfo.shareApp[appIndex].requireAppDownload.index, ContentsShareFailureReason_CommunicationError);
                    return true;
                }

                // 送信者からの返事待ち
                if (!WaitClientResponse())
                {
                    NN_LCS_LOG_DEBUG("WaitClientResponse Error %x\n", detailInfo.shareApp[appIndex].requireAppDownload.index);
                    Reject(detailInfo.shareApp[appIndex].requireAppDownload.index, ContentsShareFailureReason_CommunicationError);
                    return true;
                }
                // 送信者の状態を変更する
                m_NodeManager.SetInnerState(NodeInnerState_Transferring, detailInfo.shareApp[appIndex].requireAppDownload.index);

                result = ReceiveApplication(detailInfo, appIndex);
                if (result.IsFailure())
                {
                    if (ResultUserCancel::Includes(result) || ResultRequestNodeLeaved::Includes(result))
                    {
                        // 継続するので、そのまま
                    }
                    else if (ResultInvalidContentDeliveryRequest::Includes(result) ||
                        nn::ns::ResultReceiveApplicationTaskNotFound::Includes(result))
                    {
                        m_pResource->contentsShareFailureReason = ContentsShareFailureReason_NotExistDownloadContents;
                        ChangeState(State_ContentsShareFailed);
                        return false;
                    }
                    else if (ResultCommunicationError::Includes(result) ||
                        nn::ns::ResultApplicationContentNotDownloaded::Includes(result))
                    {
                        m_pResource->contentsShareFailureReason = ContentsShareFailureReason_CommunicationError;
                        ChangeState(State_ContentsShareFailed);
                        return false;
                    }
                    else if (ResultAsyncResultError::Includes(result))
                    {
                        ChangeState(State_ContentsShareFailed);
                        return false;
                    }
                    else if (nn::ncm::ResultSdCardContentStorageNotActive::Includes(result))
                    {
                        m_pResource->contentsShareFailureReason = ContentsShareFailureReason_SdCardError;
                        ChangeState(State_ContentsShareFailed);
                        return false;
                    }
                    else
                    {
                        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                    }
                }
            }
            // Client -> Client
            else if (detailInfo.shareApp[appIndex].requireAppDownload.formAddress != m_MyAddress &&
                detailInfo.ipAddress != m_MyAddress)
            {
                NN_LCS_LOG_DEBUG("I'm AppObserver from %x to %x\n",
                    detailInfo.shareApp[appIndex].requireAppDownload.formAddress, detailInfo.ipAddress);

                // 受信者へ受信できるか確認連絡
                if (SendShareContentClear(m_NodeManager.GetNodeSocket(detailInfo.nodeInfo.index),
                    m_pResource->shareComponent.contentsFlag, m_pResource->shareComponent.id).IsFailure())
                {
                    NN_LCS_LOG_DEBUG("SendShareContentClear Error %x\n", detailInfo.nodeInfo.index);
                    Reject(detailInfo.nodeInfo.index, ContentsShareFailureReason_CommunicationError);
                    return true;
                }

                // 受信者からの返事待ち
                if (!WaitClientResponse())
                {
                    NN_LCS_LOG_DEBUG("WaitClientResponse Error %x\n", detailInfo.nodeInfo.index);
                    Reject(detailInfo.nodeInfo.index, ContentsShareFailureReason_CommunicationError);
                    return true;
                }

                // 受信者が Suspend かを見る
                if (m_NodeManager.IsInnerState(detailInfo.nodeInfo.index, NodeInnerState_Suspending))
                {
                    return true;
                }

                // 送信者へ連絡
                if (SendShareContentSendReq(
                    m_NodeManager.GetNodeSocket(detailInfo.shareApp[appIndex].requireAppDownload.index),
                    TransferRole_Sender, detailInfo.ipAddress,
                    detailInfo.nodeInfo.index, m_pResource->shareComponent.id, 1).IsFailure())
                {
                    NN_LCS_LOG_DEBUG("SendShareContentSendReq Error %x\n", detailInfo.shareApp[appIndex].requireAppDownload.index);
                    Reject(detailInfo.shareApp[appIndex].requireAppDownload.index, ContentsShareFailureReason_CommunicationError);
                    return true;
                }

                // 送信者からの返事待ち
                if (!WaitClientResponse())
                {
                    NN_LCS_LOG_DEBUG("WaitClientResponse Error %x\n", detailInfo.shareApp[appIndex].requireAppDownload.index);
                    Reject(detailInfo.shareApp[appIndex].requireAppDownload.index, ContentsShareFailureReason_CommunicationError);
                    return true;
                }
                // 送信者の状態を変更する
                m_NodeManager.SetInnerState(NodeInnerState_Transferring, detailInfo.shareApp[appIndex].requireAppDownload.index);

                // 受信者へ連絡
                if (SendShareContentRecvReq(m_NodeManager.GetNodeSocket(detailInfo.nodeInfo.index),
                    TransferRole_Receiver,
                    detailInfo.shareApp[appIndex].requireAppDownload.formAddress,
                    detailInfo.shareApp[appIndex].requireAppDownload.index,
                    m_pResource->shareComponent.id,
                    m_pResource->shareComponent.contentMetaKey, m_pResource->shareComponent.contentMetaKeyCount, 1).IsFailure())
                {
                    NN_LCS_LOG_DEBUG("SendShareContentRecvReq Error %x\n", detailInfo.nodeInfo.index);
                    Reject(detailInfo.nodeInfo.index, ContentsShareFailureReason_CommunicationError);
                    return true;
                }

                // 受信者からの返事待ち
                if (!WaitClientResponse())
                {
                    NN_LCS_LOG_DEBUG("WaitClientResponse Error %x\n", detailInfo.nodeInfo.index);
                    // ToDo : 送信者にキャンセルを連絡
                    Reject(detailInfo.nodeInfo.index, ContentsShareFailureReason_CommunicationError);
                    return true;
                }
                // 送信者の状態を変更する
                m_NodeManager.SetInnerState(NodeInnerState_Transferring, detailInfo.nodeInfo.index);
                // 受信者状態確認用の設定
                m_IsWaitReceiver = true;
                m_NodeIndex = detailInfo.nodeInfo.index;

            }
        }

        m_Role = TransferRole_None;

        // 転送完了待ち(タイムアウト作るかは要検討)
        while (m_IsOpened)
        {
            NN_LCS_LOG_DEBUG("AppReceiver Exsist : %d\n",
                m_NodeManager.IsExsistNode(detailInfo.shareApp[appIndex].requireAppDownload.index));
            NN_LCS_LOG_DEBUG("AppReceiver State : %d\n",
                m_NodeManager.IsInnerState(detailInfo.shareApp[appIndex].requireAppDownload.index, NodeInnerState_Transferring));
            NN_LCS_LOG_DEBUG("AppSender Exsist : %d\n",
                m_NodeManager.IsExsistNode(detailInfo.nodeInfo.index));
            NN_LCS_LOG_DEBUG("AppSender State : %d\n",
                m_NodeManager.IsInnerState(detailInfo.nodeInfo.index, NodeInnerState_Transferring));
            if ((!m_NodeManager.IsExsistNode(detailInfo.shareApp[appIndex].requireAppDownload.index) ||
                 !m_NodeManager.IsInnerState(detailInfo.shareApp[appIndex].requireAppDownload.index, NodeInnerState_Transferring)) &&
                (!m_NodeManager.IsExsistNode(detailInfo.nodeInfo.index) ||
                 !m_NodeManager.IsInnerState(detailInfo.nodeInfo.index, NodeInnerState_Transferring)))
            {
                NN_LCS_LOG_DEBUG("Complete to Transfer Content / %x to %x\n",
                    detailInfo.shareApp[appIndex].requireAppDownload.index, detailInfo.nodeInfo.index);

                // アプリの受信者・送信者共に健在ならばパッチの配信に成功しているので、
                // コンテンツの配信に成功しているので、フラグを立てる。
                if (m_NodeManager.IsExsistNode(detailInfo.shareApp[appIndex].requireAppDownload.index) &&
                    m_NodeManager.IsExsistNode(detailInfo.nodeInfo.index))
                {
                    m_IsContentShareSucceeded = true;
                }
                break;
            }
            nn::os::WaitEvent(&m_ClientStateChangeEvent);
        }
        m_NodeIndex = 0;
        m_IsSender = false;
        m_IsWaitReceiver = false;

        UpdateAdvertiseProgress();

        return true;
    } // NOLINT(impl/function_size)

    Result Host::ReceiveApplication(const NodeDetailInfo& info, int appIndex) NN_NOEXCEPT
    {
        int retryCount = 0;
        bool isSuccess = false;
        // 受信タスクを生成するとき、タイミング的に相手の準備ができていない場合があるので、リトライをする
        for (retryCount = 0; retryCount < ShareTaskRetryCountMax; ++retryCount)
        {
            nn::ns::AsyncResult asyncResult;

            NN_LCS_LOG_DEBUG("RequestReceiveApplication RetryCount : %d\n", retryCount);
            // 受信タスクの生成
            NN_RESULT_DO(RequestReceiveApplication(&asyncResult,
                info.shareApp[appIndex].requireAppDownload.formAddress, SharePort,
                m_pResource->shareComponent.id,
                m_pResource->shareComponent.contentMetaKey, m_pResource->shareComponent.contentMetaKeyCount));

            if (!isSuccess)
            {
                // 進捗用の設定
                m_Role = TransferRole_Receiver;
                m_NodeIndex = info.shareApp[appIndex].requireAppDownload.index;
                ::std::memcpy(&m_pResource->contentInfo[m_pResource->contentInfoCount], &m_pResource->sessionInfo.contents[appIndex], sizeof(ContentsInfo));

                // 受信者になったので、アプリの終了を確認してもらうため Suspend する
                m_pResource->suspendedReason = SuspendedReason_NeedTerminateApplication;
                ChangeState(State_Suspended);
                isSuccess = true;
            }

            // 受信完了待ち
            auto result = WaitTask(&asyncResult, m_NodeIndex);
            if (result.IsSuccess())
            {
                // commit 処理
                result = CommitApplication();
                if (result.IsFailure())
                {
                    if (ResultUserCancel::Includes(result) ||
                        nn::ns::ResultReceiveApplicationTaskNotFound::Includes(result) ||
                        nn::ns::ResultApplicationContentNotDownloaded::Includes(result) ||
                        nn::ncm::ResultSdCardContentStorageNotActive::Includes(result))
                    {
                        asyncResult.Cancel();
                        m_Role = TransferRole_None;
                        m_NodeIndex = 0;
                        return result;
                    }
                    else
                    {
                        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                    }
                }
                // commit 完了を通知するために送信
                SendSessionInfoToAll();
                break;
            }
            else if (ResultAsyncResultError::Includes(result) || ResultUserCancel::Includes(result) ||
                ResultRequestNodeLeaved::Includes(result))
            {
                asyncResult.Cancel();
                m_Role = TransferRole_None;
                m_NodeIndex = 0;
                return result;
            }
            else if(ResultCommunicationError::Includes(result))
            {
                // そのままリトライをするので何もしない。
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
            // リトライするので、少し待つ
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(ShareTaskRetryIntervalMilliSeconds));
        }

        m_Role = TransferRole_None;
        m_NodeIndex = 0;

        if (retryCount == ShareTaskRetryCountMax)
        {
            return ResultCommunicationError();
        }

        NN_RESULT_SUCCESS;
    }

    Result Host::CommitApplication() NN_NOEXCEPT
    {
        if (m_pResource->state.GetState() == State_Suspended &&
            m_pResource->suspendedReason == SuspendedReason_NeedTerminateApplication)
        {
            NN_LCS_LOG_DEBUG("WaitResume\n");
            nn::os::ClearEvent(&m_ResumeEvent);
            nn::os::WaitEvent(&m_ResumeEvent);
            if (!m_IsOpened)
            {
                NN_LCS_LOG_DEBUG("Discard Patch\n");
                return ResultUserCancel();
            }
        }
        NN_LCS_LOG_DEBUG("Commit : %llx\n", m_pResource->shareComponent.id);

        auto result = CommitReceiveApplication(m_pResource->shareComponent.id);
        if (result.IsSuccess())
        {
            // 成功したら AppDeliveryInfo を更新する
            UpdateHostDetailInfo(m_pResource->shareComponent.id);

            if (m_IsShareCompleted)
            {
                NN_LCS_LOG_DEBUG("ContetsShare Completed\n");
                ChangeState(State_Completed);
            }
        }
        else
        {
            NN_LCS_LOG_DEBUG("Commit Error! : %08x\n", result.GetInnerValueForDebug());
            return result;
        }

        // 自身のコンテンツの情報を更新
        for (int i = 0; i < m_pResource->ownAppInfoCount; ++i)
        {
            if (m_pResource->ownAppInfo[i].detailInfo.id == m_pResource->shareComponent.id)
            {
                detail::GetApplicationDetailInfo(&m_pResource->ownAppInfo[i].detailInfo,
                    m_pResource->appControlDataBuffer, detail::AppControlDataSize,
                    m_pResource->shareComponent.id);
                detail::GetApplicationDeliveryInfo(&m_pResource->ownAppInfo[i].deliveryInfoCount,
                    m_pResource->ownAppInfo[i].deliveryInfo, detail::ApplicationDeliveryInfoCountMax,
                    m_pResource->shareComponent.id);
            }
        }

        ++m_pResource->contentInfoCount;
        NN_ABORT_UNLESS_LESS_EQUAL(m_pResource->contentInfoCount, DownloadableContentsCountMax);
        NN_RESULT_SUCCESS;
    }

    bool Host::GetCanShareSystemNodeIndex(uint32_t* pOutIndex) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutIndex);

        int nodeCount = 0;
        NodeInfo info[NodeCountMax] = {};
        NodeDetailInfo detailInfo = {};

        // 更新がいるならもらえるか確認(ホストも見るけど気にしない)
        m_NodeManager.GetNodeInfo(&nodeCount, info, NodeCountMax);
        for (int i = 0; i < nodeCount; ++i)
        {
            m_NodeManager.GetNodeDetailInfo(&detailInfo, info[i].index);
            if (detailInfo.requireSystemDownload.isRequire)
            {
                NN_LCS_LOG_DEBUG("Receiver %s[%x] InnerState : %d\n",
                    detailInfo.nodeInfo.userName, detailInfo.nodeInfo.index, detailInfo.state);
                NodeDetailInfo senderDetailInfo = {};
                m_NodeManager.GetNodeDetailInfo(&senderDetailInfo, detailInfo.requireSystemDownload.index);
                NN_LCS_LOG_DEBUG("Sender   %s[%x] InnerState : %d\n",
                    senderDetailInfo.nodeInfo.userName, senderDetailInfo.nodeInfo.index, senderDetailInfo.state);

                if (m_NodeManager.IsInnerState(detailInfo.nodeInfo.index, NodeInnerState_Connected) &&
                    m_NodeManager.IsInnerState(detailInfo.requireSystemDownload.index, NodeInnerState_Connected))
                {
                    *pOutIndex = info[i].index;
                    return true;
                }
            }
        }
        return false;
    }

    bool Host::GetCanShareContentsNodeIndex(uint32_t* pOutIndex, int* pOutAppIndex) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutIndex);
        NN_SDK_ASSERT_NOT_NULL(pOutAppIndex);

        int nodeCount = 0;
        NodeInfo info[NodeCountMax] = {};
        NodeDetailInfo detailInfo = {};

        // ホストが更新必要か確認
        uint32_t hostIndex = m_NodeManager.GetNodeIndex(m_MyAddress);
        m_NodeManager.GetNodeDetailInfo(&detailInfo, hostIndex);
        for (int j = 0; j < SharableContentsCountMax; ++j)
        {
            if (detailInfo.shareApp[j].requireAppDownload.isRequire)
            {
                if (m_NodeManager.IsInnerState(detailInfo.nodeInfo.index, NodeInnerState_Connected) &&
                    m_NodeManager.IsInnerState(detailInfo.shareApp[j].requireAppDownload.index, NodeInnerState_Connected))
                {
                    *pOutIndex = hostIndex;
                    *pOutAppIndex = j;
                    return true;
                }
                // ホストを優先するのでホストが必要なのに対象がいないなら待つ
                return false;
            }
        }

        // 更新がいるならもらえるか確認(ホストももう一度見るけど気にしない)
        m_NodeManager.GetNodeInfo(&nodeCount, info, NodeCountMax);
        for (int i = 0; i < nodeCount; ++i)
        {
            m_NodeManager.GetNodeDetailInfo(&detailInfo, info[i].index);
            for (int j = 0; j < SharableContentsCountMax; ++j)
            {
                if (detailInfo.shareApp[j].requireAppDownload.isRequire)
                {
                    if (m_NodeManager.IsInnerState(detailInfo.nodeInfo.index, NodeInnerState_Connected) &&
                        m_NodeManager.IsInnerState(detailInfo.shareApp[j].requireAppDownload.index, NodeInnerState_Connected))
                    {
                        *pOutIndex = info[i].index;
                        *pOutAppIndex = j;
                        return true;
                    }
                }
            }
        }
        return false;
    }

    bool Host::WaitClientResponse() NN_NOEXCEPT
    {
        bool isRecv = false;
        nn::os::TimerEventType timeoutEvent;
        nn::os::MultiWaitType       multiWait;
        nn::os::MultiWaitHolderType recvHolder;
        nn::os::MultiWaitHolderType timeoutHolder;
        nn::os::MultiWaitHolderType cancelHolder;

        nn::os::InitializeTimerEvent(&timeoutEvent, nn::os::EventClearMode_AutoClear);
        nn::os::InitializeMultiWait(&multiWait);
        nn::os::InitializeMultiWaitHolder(&recvHolder, &m_RecvResPacketEvent);
        nn::os::InitializeMultiWaitHolder(&timeoutHolder, &timeoutEvent);
        nn::os::InitializeMultiWaitHolder(&cancelHolder, &m_CancelEvent);

        nn::os::LinkMultiWaitHolder(&multiWait, &recvHolder);
        nn::os::LinkMultiWaitHolder(&multiWait, &timeoutHolder);
        nn::os::LinkMultiWaitHolder(&multiWait, &cancelHolder);

        nn::os::ClearTimerEvent(&timeoutEvent);
        nn::TimeSpan interval = nn::TimeSpan::FromMilliSeconds(TimeoutMilliSeconds);
        nn::os::StartOneShotTimerEvent(&timeoutEvent, interval);

        nn::os::ClearEvent(&m_RecvResPacketEvent);
        nn::os::ClearEvent(&m_CancelEvent);

        NN_LCS_LOG_DEBUG("WaitClientResponse\n");
        nn::os::WaitAny(&multiWait);

        nn::os::MultiWaitHolderType *holder = nn::os::WaitAny(&multiWait);

        if (holder == &recvHolder)
        {
            isRecv = true;
        }

        nn::os::UnlinkMultiWaitHolder(&recvHolder);
        nn::os::UnlinkMultiWaitHolder(&timeoutHolder);
        nn::os::UnlinkMultiWaitHolder(&cancelHolder);
        nn::os::FinalizeMultiWaitHolder(&recvHolder);
        nn::os::FinalizeMultiWaitHolder(&timeoutHolder);
        nn::os::FinalizeMultiWaitHolder(&cancelHolder);
        nn::os::FinalizeMultiWait(&multiWait);
        nn::os::FinalizeTimerEvent(&timeoutEvent);

        return isRecv;
    }

    Result Host::WaitTask(nn::ns::AsyncResult* asyncResult, uint32_t index) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(asyncResult);

        NN_LCS_LOG_DEBUG("WaitTask\n");

        // Close されるまで
        while (m_IsOpened)
        {
            // タスクの監視
            if (asyncResult->TryWait())
            {
                NN_LCS_LOG_DEBUG("Task Finished\n");
                Result result = asyncResult->Get();

                if (result.IsFailure())
                {
                    NN_LCS_LOG_DEBUG("Get : %08x\n", result.GetInnerValueForDebug());

                    // ErrorContext を保存
                    nn::err::ErrorContext errorContex = {};
                    asyncResult->GetErrorContext(&errorContex);
                    SetErrorContext(errorContex);

                    // 失敗原因が FS 系だったら失敗理由を入れる
                    if (nn::fs::ResultSdCardAccessFailed::Includes(result))
                    {
                        m_pResource->contentsShareFailureReason = ContentsShareFailureReason_SdCardError;
                    }
                    else if (nn::fs::ResultGameCardAccessFailed::Includes(result) ||
                             nn::ncm::ResultInvalidContentStorage::Includes(result) ||
                             nn::ncm::ResultGameCardContentMetaDatabaseNotActive::Includes(result))
                    {
                        m_pResource->contentsShareFailureReason = ContentsShareFailureReason_GameCardError;
                    }
                    else if (nn::fs::ResultMmcAccessFailed::Includes(result))
                    {
                        m_pResource->contentsShareFailureReason = ContentsShareFailureReason_BuiltInStorageError;
                    }
                    else if (nn::fs::ResultDataCorrupted::Includes(result) ||
                             nn::ncm::ResultContentHashNotMatched::Includes(result))
                    {
                        // 壊れたコンテンツを受け取ったときにエラー
                        m_pResource->contentsShareFailureReason = ContentsShareFailureReason_ContentsCorrupted;
                    }
                    // 容量が足りないときにエラー
                    else if (nn::ncm::ResultNotEnoughInstallSpace::Includes(result))
                    {
                        m_pResource->contentsShareFailureReason = ContentsShareFailureReason_NotEnoughStorageSpace;
                    }
                    else if (nn::nim::ResultLocalCommunicationSessionClosed::Includes(result) ||
                             nn::nim::ResultLocalCommunicationSocketUnexpectedError::Includes(result) ||
                             nn::nim::ResultLocalCommunicationInvalidSignature::Includes(result) ||
                             nn::nim::ResultLocalCommunicationUnexpectedTag::Includes(result))
                    {
                        return ResultCommunicationError();
                    }
                    else if (nn::nim::ResultLocalCommunicationConnectionCanceled::Includes(result))
                    {
                        asyncResult->Cancel();
                    }
                    else
                    {
                        m_pResource->contentsShareFailureReason = ContentsShareFailureReason_UnknownError;
                    }
                    return ResultAsyncResultError();
                }

                // 相手がいなくなってないか確認
                for (int i = 0; i < m_pResource->nodeInfoCount; ++i)
                {
                    // 相手がいたら成功
                    if (m_pResource->nodeInfo[i].index == index)
                    {
                        NN_RESULT_SUCCESS;
                    }
                }
                NN_LCS_LOG_DEBUG("Leave Index : %x\n", index);
                asyncResult->Cancel();
                return ResultRequestNodeLeaved();
            }
            // 相手がいなくなってないか確認
            bool isContinue = false;
            for (int i = 0; i < m_pResource->nodeInfoCount; ++i)
            {
                // 相手がいたら継続
                if (m_pResource->nodeInfo[i].index == index)
                {
                    isContinue = true;
                }
            }
            if (!isContinue)
            {
                NN_LCS_LOG_DEBUG("Leave Index : %x\n", index);
                asyncResult->Cancel();
                return ResultRequestNodeLeaved();
            }

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(AsyncWaitIntervalMilliSeconds));
        }
        return ResultUserCancel();
    }

}}} // namespace nn::lcs
