﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_MemoryFence.h>
#include <nn/os/os_MemoryHeap.h>
#include <nn/os/os_SharedMemory.h>
#include <nn/result/result_HandlingUtility.h>

#include "irsensor_StatusManagerHolder.h"

namespace nn { namespace irsensor { namespace detail {

namespace {

//!< 他プロセスに許可されるメモリアクセス権
const ::nn::os::MemoryPermission OtherPermission =
#if defined(NN_BUILD_CONFIG_OS_WIN)
    // Visual Studio の std::atomic 型が load 時にロックしてしまうため
    ::nn::os::MemoryPermission_ReadWrite;
#else
    ::nn::os::MemoryPermission_ReadOnly;
#endif

} // namespace

StatusManagerHolder::StatusManagerHolder() NN_NOEXCEPT
    : m_IsCreated(false)
{
    // 何もしない
}

StatusManagerHolder::~StatusManagerHolder() NN_NOEXCEPT
{
    // 何もしない
}

::nn::Result StatusManagerHolder::Initialize() NN_NOEXCEPT
{
    auto size = ::nn::util::DivideUp(sizeof(StatusManager),
        ::nn::os::MemoryPageSize) * ::nn::os::MemoryPageSize;

    NN_RESULT_DO(::nn::os::CreateSharedMemory(
        &m_SharedMemory,
        size,
        ::nn::os::MemoryPermission_ReadWrite,
        OtherPermission));

    ::nn::os::MapSharedMemory(&m_SharedMemory,
        ::nn::os::MemoryPermission_ReadWrite);

    m_IsCreated = true;

    auto pManager = reinterpret_cast<StatusManager*>(
        ::nn::os::GetSharedMemoryAddress(&m_SharedMemory));
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    new (pManager) StatusManager();
    // StatusManager の構築の完了を保証
    ::nn::os::FenceMemoryStoreLoad();

    NN_RESULT_SUCCESS;
}

void StatusManagerHolder::Finalize() NN_NOEXCEPT
{
    if (m_IsCreated)
    {
        auto pManager = GetStatusManager();
        pManager->~StatusManager();
        pManager = nullptr;

        // 共有メモリに対するアクセスの完了を保証
        ::nn::os::FenceMemoryAnyStore();
    }

    m_IsCreated = false;

    if (::nn::os::GetSharedMemoryAddress(&m_SharedMemory) != nullptr)
    {
        ::nn::os::UnmapSharedMemory(&m_SharedMemory);
    }

    ::nn::os::DestroySharedMemory(&m_SharedMemory);
}

::nn::Result StatusManagerHolder::Clear() NN_NOEXCEPT
{
    if (m_IsCreated)
    {
        auto pManager = GetStatusManager();
        // コンストラクタ呼び出しで、データをクリア
        new (pManager) StatusManager();
        // 共有メモリに対するアクセスの完了を保証
        ::nn::os::FenceMemoryStoreLoad();
    }
    NN_RESULT_SUCCESS;
}

void StatusManagerHolder::Attach(
    ::nn::os::NativeHandle handle, bool managed) NN_NOEXCEPT
{
    ::nn::os::AttachSharedMemory(&m_SharedMemory,
                                 ::nn::util::DivideUp(
                                         sizeof(StatusManager),
                                         ::nn::os::MemoryPageSize
                                     ) * ::nn::os::MemoryPageSize,
                                 handle,
                                 managed);

    ::nn::os::MapSharedMemory(&m_SharedMemory, OtherPermission);
}

size_t StatusManagerHolder::GetSharedMemorySize() const NN_NOEXCEPT
{
    return ::nn::os::GetSharedMemorySize(&m_SharedMemory);
}

::nn::os::NativeHandle StatusManagerHolder::GetSharedMemoryHandle(
    ) const NN_NOEXCEPT
{
    return ::nn::os::GetSharedMemoryHandle(&m_SharedMemory);
}

StatusManager* StatusManagerHolder::GetStatusManager() const NN_NOEXCEPT
{
    auto pManager = reinterpret_cast<StatusManager*>(
        ::nn::os::GetSharedMemoryAddress(&m_SharedMemory));
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    return pManager;
}

bool StatusManagerHolder::IsSharedMemoryMapped() const NN_NOEXCEPT
{
    return ::nn::os::GetSharedMemoryAddress(&m_SharedMemory) != nullptr;
}

}}} // namespace nn::irsensor::detail
