﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/system/hid_Irsensor.h>
#include <nn/os/os_Thread.h>

#include "irsensor_IrCameraHandle.h"
#include "irsensor_DriverManager.h"
#include "irsensor_RequestHandlerTask.h"
#include "irsensor_AppletResourceManager.h"
#include "../../hid/detail/hid_NpadId.h"

namespace nn { namespace irsensor { namespace detail {

//!< リソースマネージャを扱うクラスです。
class ResourceManager final
{
    NN_DISALLOW_COPY(ResourceManager);
    NN_DISALLOW_MOVE(ResourceManager);

private:
    //!< IR センサーのRequestHandlerスレッドの優先度
    static const int RequestHandlerTaskThreadPriority;

    //!< IR センサータスクのスレッドのスタック領域
    static NN_ALIGNAS(0x1000)
        char s_RequestHandlerTaskThreadStacks[::nn::hid::system::IrSensorSupportedNpadIdsCount][0x1000];

    //!< IrSensor タスク
    RequestHandlerTask m_RequestHandlerTasks[::nn::hid::system::IrSensorSupportedNpadIdsCount];

    //!< IrSensor マネージャ
    DriverManager m_DriverManagers[::nn::hid::system::IrSensorSupportedNpadIdsCount];

    //!< 接続されている IR センサーの数
    int m_DeviceCount;

    //!< StatusManager ホルダー
    StatusManagerHolder m_StatusManagerHolder;

    //!< アプレットリソース操作用の Mutex
    ::nn::os::Mutex m_AppletResourceMutex;

    //!< アプレットリソースID操作用の Mutex
    ::nn::os::Mutex m_AppletResourceIdMutex;

    //!< アプレットリソースマネージャ
    AppletResourceManager m_AppletResourceManager;

public:
    ResourceManager() NN_NOEXCEPT;

    ~ResourceManager() NN_NOEXCEPT;

    //!< IrSensor タスクを返します。
    RequestHandlerTask& GetRequestHandlerTask(
        const IrCameraHandle& handle) NN_NOEXCEPT;

    //!< IrCameraHandle を取得します。
    ::nn::Result GetIrCameraHandle(
        IrCameraHandle* outValue,
        const ::nn::hid::NpadIdType& npadIdType) NN_NOEXCEPT;

    //!< IrCameraHandle が有効であることを表明します。
    ::nn::Result AssertValidIrCameraHandle(
        const IrCameraHandle& handle) NN_NOEXCEPT;

    //!< IrCameraHandle に対応する Xcd ドライバを初期化して、マネージャに登録します。
    ::nn::Result BindIrCameraXcdDriver(
        const IrCameraHandle& handle) NN_NOEXCEPT;

    //!< IrCameraHandle に対応する Xcd ドライバを終了して、マネージャから解除します。
    ::nn::Result UnbindIrCameraXcdDriver(
        const IrCameraHandle& handle) NN_NOEXCEPT;

    //!< 共通のリソースハンドルを取得します
    ::nn::Result GetSharedMemoryHandle(
        ::nn::os::NativeHandle* outValue) NN_NOEXCEPT;

    //!< TransferMemory の破棄
    ::nn::Result DestroyTransferMemory() NN_NOEXCEPT;

    //!< StatusManager のデータをクリアします
    ::nn::Result ClearStatusManager() NN_NOEXCEPT;

    //!< アプレットリソースの Aruid を設定します。
    void SetAppletResourceUserId(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< アプレットリソースのエントリを追加します。
    ::nn::Result AppendAppletResourceEntry(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< アプレットリソースのエントリを削除します。
    void RemoveAppletResourceEntry(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< アプレットリソースユーザ ID を登録します。
    ::nn::Result RegisterAppletResourceUserId(
        ::nn::applet::AppletResourceUserId aruid,
        bool enablesInput) NN_NOEXCEPT;

    //!< アプレットリソースユーザ ID の登録を解除します。
    void UnregisterAppletResourceUserId(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< アプレットリソースの IR センサー入力が有効か否かを設定します。
    void EnableAppletResourceInput(
        ::nn::applet::AppletResourceUserId aruid,
        bool enablesInput) NN_NOEXCEPT;

    //!< IR センサー入力の提供を行うか否かを設定します。
    void EnableAppletToGetInput(
        ::nn::applet::AppletResourceUserId aruid,
        bool enablesInput) NN_NOEXCEPT;

    //!< アプレットリソースのエントリを破棄します。
    void ResetAppletResourceEntry(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< アプレットリソースのエントリを設定します。
    ::nn::Result SetAppletResourceEntry(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;
};

//!< リソースマネージャを返します。
ResourceManager& GetResourceManager() NN_NOEXCEPT;

}}} // namespace nn::irsensor::detail
