﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>

#include "irsensor_InternalErrorHandler.h"
#include "irsensor_IrCameraHandle.h"

namespace nn { namespace irsensor { namespace detail {

InternalErrorHandler::InternalErrorHandler() NN_NOEXCEPT
    : m_Mutex(false)
{
    for (auto i = 0; i < ::nn::hid::system::IrSensorSupportedNpadIdsCount; i++)
    {
        m_InternalStatus[i] = IrCameraInternalStatus_Available;
    }
}

InternalErrorHandler::~InternalErrorHandler() NN_NOEXCEPT
{
    // 何もしない
}

bool InternalErrorHandler::IsLibraryAppletCallEnabled(
    const IrCameraInternalStatus& status,
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    // 複数 handle からの呼び出しを排他して、重複して LA が呼ばれることを防ぐ。
    ::std::lock_guard<decltype(m_Mutex)> statusLocker(m_Mutex);

    bool isLibraryAppletCallEnabled = true;
    if (status == IrCameraInternalStatus_Available)
    {
        // Available 状態の場合は、LA 呼び出しは必要ないので falseを返す。
        isLibraryAppletCallEnabled = false;
    }
    else
    {
        for (auto i = 0; i < ::nn::hid::system::IrSensorSupportedNpadIdsCount; i++)
        {
            // 同じエラーが少なくとも1回以上発生している場合は、
            // すでに LA が起動済みかつ、問題が解決していない状態なので、
            //
            if (m_InternalStatus[i] == status)
            {
                isLibraryAppletCallEnabled = false;
                break;
            }
        }
    }
    // 内部状態を更新する
    m_InternalStatus[GetIrCameraHandlePlayerNumber(handle)] = status;

    return isLibraryAppletCallEnabled;
}

void InternalErrorHandler::SetInternalStatus(
    const IrCameraInternalStatus& status,
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> statusLocker(m_Mutex);
    // 内部状態を更新する
    m_InternalStatus[GetIrCameraHandlePlayerNumber(handle)] = status;
}

}
}
} // namespace nn::irsensor::detail
