﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/util/util_BitFlagSet.h>
#include <nn/irsensor/irsensor_PackedTypes.h>
#include <nn/irsensor/irsensor_ImageTransferProcessorTypes.h>
#include <nn/hid/system/hid_Irsensor.h>

#include "irsensor_ActivationCount.h"
#include "irsensor_StatusManager.h"
#include "irsensor_StatusManagerHolder.h"

namespace nn { namespace irsensor { namespace detail {

//!< IR センサーに対するアプレットリソースのエントリの最大数
const size_t AppletResourceEntryCountMax = 8;

//!< アプレットリソースの制御フラグ定義です。
struct AppletResourceFlag final
{
    typedef ::nn::util::BitFlagSet<32, AppletResourceFlag>::Flag<0>
        IsRegistered;   //!< 登録済みか否か
    typedef ::nn::util::BitFlagSet<32, AppletResourceFlag>::Flag<1>
        IsAvailable;    //!< 利用可能か否か
    typedef ::nn::util::BitFlagSet<32, AppletResourceFlag>::Flag<16>
        EnablesInput;   //!< 入力の提供を行うか否か
    typedef ::nn::util::BitFlagSet<32, AppletResourceFlag>::Flag<17>
        IsForeground;   //!< アプレットが FG か否か
};

//!< アプレットリソースの制御フラグ集合を扱う型です。
typedef ::nn::util::BitFlagSet<32, AppletResourceFlag> AppletResourceFlagSet;

//!< IR センサーのモード
enum class IrSensorMode
{
    None,
    Moment,
    Clustering,
    ImageTransfer,
    Pointing,
    TeraPlugin,
    IrLed,
};

//!< IR センサーの設定値の共用体
union IrSensorConfig
{
    PackedMomentProcessorConfig moment;
    PackedClusteringProcessorConfig clustering;
    PackedImageTransferProcessorExConfig imageTransfer;
    PackedPointingProcessorConfig pointing;
    PackedTeraPluginProcessorConfig teraPlugin;
    PackedIrLedProcessorConfig irLed;
};

//!< 画像転送モードの情報
struct ImageTransferStatus
{
    ::nn::irsensor::ImageTransferProcessorState imageTransferState;
    ::nn::os::TransferMemoryType transferMemory;
    void* workBufferAddress;
    void* imageBufferAddress;
    bool isImageValid;
};

//!< デバイス単位の共有メモリを除いたリソース情報
struct IrSensorDeviceResource
{
    IrSensorMode mode;                       //!< 現在のモード
    IrSensorConfig config;                   //!< 各モードの設定
    PackedFunctionLevel functionLevel;       //!< 設定されたFunctionLevel
    ImageTransferStatus imageTransferStatus; //!< 画像転送モードの情報
};

//!< アプレットリソースのエントリを表す構造体です。
struct AppletResourceEntry final
{
    AppletResourceFlagSet flags;                //!< 制御フラグ
    ::nn::applet::AppletResourceUserId aruid;   //!< ARUID
    int clientStackLevel;                       //!< 重複して呼び出された段数
    StatusManager* address;                     //!< 共有メモリの開始アドレス
    IrSensorDeviceResource deviceResource[::nn::hid::system::IrSensorSupportedNpadIdsCount]; //!< デバイスごとの設定
};


//!< アプレットリソースの解決を担うマネージャを扱うクラスです。
class AppletResourceManager final
{
    NN_DISALLOW_COPY(AppletResourceManager);
    NN_DISALLOW_MOVE(AppletResourceManager);

private:
    //!< ARUID
    ::nn::applet::AppletResourceUserId m_AppletResourceUserId;

    //!< アプレットリソースのエントリ
    AppletResourceEntry m_Entries[AppletResourceEntryCountMax];

    //!< アクティブ化された回数
    ActivationCount m_ActivationCount;

    //!< 共有メモリホルダ
    StatusManagerHolder m_StatusManagerHolder;

public:
    AppletResourceManager() NN_NOEXCEPT;

    //!< ARUID を返します。
    ::nn::applet::AppletResourceUserId GetAppletResourceUserId() NN_NOEXCEPT;

    //!< ARUID を設定します。
    void SetAppletResourceUserId(::nn::applet::AppletResourceUserId value
                                 ) NN_NOEXCEPT;

    //!< アプレットリソースのエントリを返します。
    const AppletResourceEntry (&GetAppletResourceEntries(
        ) NN_NOEXCEPT)[AppletResourceEntryCountMax];

    //!< アプレットリソースユーザ ID を登録します。
    ::nn::Result RegisterAppletResourceUserId(
        ::nn::applet::AppletResourceUserId aruid,
        bool enablesInput) NN_NOEXCEPT;

    //!< アプレットリソースユーザ ID の登録を解除します。
    void UnregisterAppletResourceUserId(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< アプレットリソースのエントリを設定します。
    ::nn::Result SetAppletResourceEntry(
        ::nn::applet::AppletResourceUserId aruid,
        StatusManager* address) NN_NOEXCEPT;

    //!< アプレットリソースのエントリの設定を解除します。
    void ResetAppletResourceEntry(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< アプレットリソースが有効かどうかを表します。
    bool IsAppletResourceAvailable() NN_NOEXCEPT;

    //!< 指定したアプレットリソースが使用中かどうかを表します。
    bool IsAppletResourceUsed(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 画像転送モードでのデータの有効フラグを設定します。
    ::nn::Result SetImageValidity(
        ::nn::applet::AppletResourceUserId aruid,
        IrCameraHandle handle,
        bool isValid) NN_NOEXCEPT;

    //!< 画像転送モードの出力値を設定
    ::nn::Result SetImageTransferState(
        IrCameraHandle handle,
        ImageTransferProcessorState state) NN_NOEXCEPT;

    //!< 画像転送モードのバッファを設定
    ::nn::Result SetImageTransferBuffer(
        IrCameraHandle handle,
        void* address) NN_NOEXCEPT;

    //!< IR センサーの設定値を保存
    ::nn::Result SetIrSensorConfig(
        const ::nn::applet::AppletResourceUserId& aruid,
        IrCameraHandle handle,
        IrSensorConfig config) NN_NOEXCEPT;

    //!< デバイスごとのリソースの状態を取得
    ::nn::Result GetIrSensorDeviceResource(
        IrSensorDeviceResource* deviceResource,
        const ::nn::applet::AppletResourceUserId& aruid,
        IrCameraHandle handle) NN_NOEXCEPT;

    //!< デバイスごとのモードを取得
    ::nn::Result GetIrSensorMode(
        IrSensorMode* pOutMode,
        const ::nn::applet::AppletResourceUserId& aruid,
        IrCameraHandle handle) NN_NOEXCEPT;

    //!< デバイスごとのモードを設定
    ::nn::Result SetIrSensorMode(
        ::nn::applet::AppletResourceUserId aruid,
        IrSensorMode mode,
        IrCameraHandle handle) NN_NOEXCEPT;

    //!< デバイスごとの FunctionLevel を取得
    ::nn::Result GetIrSensorFunctionLevel(
        ::nn::applet::AppletResourceUserId aruid,
        PackedFunctionLevel* pOutLevel,
        IrCameraHandle handle) NN_NOEXCEPT;

    //!< デバイスごとの FunctionLevel を設定
    ::nn::Result SetIrSensorFunctionLevel(
        ::nn::applet::AppletResourceUserId aruid,
        PackedFunctionLevel functionLevel,
        IrCameraHandle handle) NN_NOEXCEPT;

    //!< 直近で Activate された Aruid を取得
    ::nn::applet::AppletResourceUserId GetLatestActivatedAruid() NN_NOEXCEPT;

    //!< TransferMemoryType を取得します。
    ::nn::Result GetTransferMemoryType(
        ::nn::os::TransferMemoryType** pTransferMemory,
        ::nn::applet::AppletResourceUserId aruid,
        IrCameraHandle handle) NN_NOEXCEPT;

    //!< TransferMemory のアドレスを設定します。
    ::nn::Result SetTransferMemoryWorkBufferAddress(
        ::nn::applet::AppletResourceUserId aruid,
        void* address,
        IrCameraHandle handle) NN_NOEXCEPT;

    //!< 共有アプレットリソースの共有メモリのハンドルを返します。
    ::nn::os::NativeHandle GetSharedAppletResourceSharedMemoryHandle(
        ) const NN_NOEXCEPT;

};

}}} // namespace nn::irsensor::detail
