﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_Thread.h>
#include <nn/result/result_HandlingUtility.h>

#include "hidbus_Task.h"

namespace nn { namespace hidbus { namespace server {

Task::Task() NN_NOEXCEPT
    : m_ThreadMutex(false)
    , m_ThreadStack(nullptr)
    , m_ThreadStackSize(0)
    , m_ThreadPriority(::nn::os::InvalidThreadPriority)
{
    // 何もしない
}

Task::~Task() NN_NOEXCEPT
{
    // 何もしない
}

void Task::SetThreadStack(void* stack, size_t stackSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(stack);
    m_ThreadStack = stack;
    m_ThreadStackSize = stackSize;
}

void Task::SetThreadPriority(int priority) NN_NOEXCEPT
{
    m_ThreadPriority = priority;
}

::nn::Result Task::ActivateThread() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_ThreadStack);

    ::std::lock_guard<decltype(m_ThreadMutex)> locker(m_ThreadMutex);

    //NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
    //                       ResultResourceManagerActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施

        // スレッドの作成を試行
        NN_RESULT_DO(::nn::os::CreateThread(&m_Thread,
                                            Task::ThreadFunc,
                                            this,
                                            m_ThreadStack,
                                            m_ThreadStackSize,
                                            m_ThreadPriority));

        // 多重待ちオブジェクトを初期化
        ::nn::os::InitializeMultiWait(&m_MultiWait);

        // 停止イベントをリンク
        m_TerminationEvent.Link(&m_MultiWait);

        // イベントをリンク
        this->LinkEvent(&m_MultiWait);

        // スレッド名を登録
        this->SetThreadName(&m_Thread);

        // 共通サンプリングスレッドを開始
        ::nn::os::StartThread(&m_Thread);
    }

    // アクティブ化回数をインクリメント
    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result Task::DeactivateThread() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_ThreadMutex)> locker(m_ThreadMutex);

    //NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
    //                       ResultResourceManagerDeactivationLowerLimitOver());

    // アクティブ化回数をデクリメント
    --m_ActivationCount;

    if (m_ActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施

        // スレッドの停止を要求
        m_TerminationEvent.Signal();

        // スレッドの停止を待機
        ::nn::os::WaitThread(&m_Thread);

        // イベントをアンリンク
        this->UnlinkEvent();

        // 停止イベントをアンリンク
        m_TerminationEvent.Unlink();

        // 多重待ちオブジェクトを破棄
        ::nn::os::FinalizeMultiWait(&m_MultiWait);

        // スレッドの破棄
        ::nn::os::DestroyThread(&m_Thread);
    }

    NN_RESULT_SUCCESS;
}

void Task::ThreadFunc(void* value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(value);
    static_cast<Task*>(value)->InvokeEventLoop();
}

void Task::InvokeEventLoop() NN_NOEXCEPT
{
    while (NN_STATIC_CONDITION(true))
    {
        ::nn::os::MultiWaitHolderType* waitId = ::nn::os::WaitAny(&m_MultiWait);

        if (waitId == m_TerminationEvent.GetWaitId())
        {
            m_TerminationEvent.Clear();
            // 停止が要求されたならばループを脱出
            return;
        }
        else
        {
            this->HandleEvent(waitId);
        }
    }
}

}}} // namespace nn::hidbus::server
