﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <climits>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>

#include <nn/hid/hid_ResultControllerSupport.h>

namespace nn { namespace hidbus { namespace server {

inline nn::hid::ExternalBusHandle ConvertHandleFromHidbusToHid(nn::hidbus::BusHandle handle)
{
    nn::hid::ExternalBusHandle hidHandle;
    hidHandle._storage = handle._storage;
    return hidHandle;
}

inline nn::hidbus::BusHandle ConvertHandleFromHidToHidbus(nn::hid::ExternalBusHandle handle)
{
    nn::hidbus::BusHandle hidbusHandle;
    hidbusHandle._storage = handle._storage;
    return hidbusHandle;
}

inline nn::Result ConvertHidResultToHidBusResult(nn::Result hidResult)
{
    if (hidResult.IsFailure())
    {
        if (nn::hid::ResultExternalBusActivateFailureNpadBusy::Includes(hidResult))
        {
            return nn::hidbus::ResultActivateFailureNpadBusy();
        }
        else if (nn::hid::ResultExternalBusDeviceNotAttached::Includes(hidResult))
        {
            return nn::hidbus::ResultExternalDeviceNotAttached();
        }
        else if (nn::hid::ResultExternalBusDeviceReadyTimeout::Includes(hidResult))
        {
            return nn::hidbus::ResultExternalDeviceReadyTimeout();
        }
        else if (nn::hid::ResultExternalBusDeviceInquiryTimeout::Includes(hidResult))
        {
            return nn::hidbus::ResultExternalDeviceTimeout();
        }
        else if (nn::hid::ResultExternalBusDeviceTimeout::Includes(hidResult))
        {
            return nn::hidbus::ResultExternalDeviceTimeout();
        }
        else if (nn::hid::ResultExternalBusDeviceNotEnabled::Includes(hidResult))
        {
            return nn::hidbus::ResultExternalDeviceNotEnabled();
        }
        else if (nn::hid::ResultExternalBusDeviceInvalidHandle::Includes(hidResult) ||
                 nn::hid::ResultNoExternalBusFoundOnNpad::Includes(hidResult))
        {
            return nn::hidbus::ResultInvalidHandle();
        }
        else if (nn::hid::ResultControllerFirmwareUpdateFailed::Includes(hidResult))
        {
            return nn::hidbus::ResultFWUpdateFailed();
        }
        else
        {
            return hidResult;
        }
    }
    NN_RESULT_SUCCESS;
}

inline nn::Result ConvertHidResultToHidBusResultForCommunicatin(nn::Result hidResult)
{
    if (hidResult.IsFailure())
    {
        if (nn::hid::ResultExternalBusDeviceNotAttached::Includes(hidResult) ||
            nn::hid::ResultExternalBusDeviceNotEnabled::Includes(hidResult))
        {
            return nn::hidbus::ResultExternalDeviceNotEnabled();
        }
        else if (nn::hid::ResultExternalBusDeviceReadyTimeout::Includes(hidResult) ||
                 nn::hid::ResultExternalBusDeviceInquiryTimeout::Includes(hidResult) ||
                 nn::hid::ResultExternalBusDeviceTimeout::Includes(hidResult))
        {
            return nn::hidbus::ResultExternalDeviceTimeout();
        }
        else if (nn::hid::ResultExternalBusDeviceInvalidHandle::Includes(hidResult))
        {
            return nn::hidbus::ResultInvalidHandle();
        }
        else
        {
            return hidResult;
        }
    }
    NN_RESULT_SUCCESS;
}

inline nn::Result CheckValidExternalBusHandle(nn::hidbus::BusHandle handle)
{
    // handle の有効性チェック
    auto hidHandle = ConvertHandleFromHidbusToHid(handle);
    auto handleCheckResult = nn::hid::CheckValidExternalBusHandle(hidHandle);
    NN_RESULT_DO(ConvertHidResultToHidBusResult(handleCheckResult));

    // InternalIndex に関しては、hid 層でチェックされない (使わない) ため、 hidbus 層でチェックする
    uint32_t upperHandle = static_cast<uint32_t>(handle._storage >> 32);
    // 下位 8bit だけ抜き出す
    auto index = static_cast<int>(upperHandle & 255);
    if (index > nn::hidbus::detail::MaxHidbusNumber)
    {
        return nn::hidbus::ResultInvalidHandle();
    }

    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hidbus::server
