﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/sf/sf_ObjectFactory.h>

#include <nn/hid/hid_ExternalBus.h>
#include <nn/hid/hid_ResultPrivate.h>

#include <nn/hidbus/hidbus_Result.h>
#include <nn/hidbus/hidbus_ResultPrivate.h>
#include <nn/hidbus/detail/hidbus_Log.h>

#include "hidbus_HidbusServer.h"
#include "hidbus_StaticObject.h"
#include "hidbus_ServerUtil.h"

#include "../detail/hidbus_PollingDataAccessor.h"

namespace
{

// SchedulerTaskThread に与えるスタック領域
const size_t SchedulerTaskThreadStackSize = nn::os::StackRegionAlignment;
NN_ALIGNAS(nn::os::StackRegionAlignment) char s_SchedulerTaskThreadStack[SchedulerTaskThreadStackSize];

//!< os ハンドルを管理するか否かを表す値
const bool NeedsToBeManaged =
#ifdef NN_BUILD_CONFIG_OS_HORIZON
false;
#else
false;
#endif

}

namespace nn { namespace hidbus { namespace server {

HidbusServer::HidbusServer() NN_NOEXCEPT
{
    // SharedMemory の作成とマッピング
    GetResourceHolder().Initialize();

    GetSchedulerTask().SetThreadPriority(NN_SYSTEM_THREAD_PRIORITY(hidbus, SchedulerTaskThread));
    GetSchedulerTask().SetThreadStack(&s_SchedulerTaskThreadStack, SchedulerTaskThreadStackSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetSchedulerTask().Initialize(&GetResourceHolder()));
}

HidbusServer::~HidbusServer() NN_NOEXCEPT
{
    // SharedMemory の破棄
    GetResourceHolder().Finalize();
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetSchedulerTask().Finalize());
}
nn::Result HidbusServer::GetBusHandle(nn::sf::Out<nn::hidbus::BusHandle> outHandle, nn::sf::Out<bool> hasHandle,
                                      std::uint32_t npadId, std::uint64_t busTypeId, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    nn::hid::ExternalBusHandle handle;

    // 将来、アプレット対応等の複数クライアント対応する際に、 handle から 対応する aruid が引っ張れるように
    // 追加しておく。今は何も使わない。
    NN_UNUSED(aruid);

    // 定義されている ExternalBusType が指定されているかどうかチェック
    if (static_cast<nn::hid::ExternalBusType>(busTypeId) != nn::hid::ExternalBusType_LeftJoyRail &&
        static_cast<nn::hid::ExternalBusType>(busTypeId) != nn::hid::ExternalBusType_RightJoyRail)
    {
        return nn::hidbus::ResultInvalidArgument();
    }

    *hasHandle = nn::hid::GetExternalBusHandle(&handle, npadId, static_cast<nn::hid::ExternalBusType>(busTypeId));

    outHandle->_storage = handle._storage;
    NN_RESULT_SUCCESS;

}

nn::Result HidbusServer::IsExternalDeviceConnected(nn::sf::Out<bool> isAttached, nn::hidbus::BusHandle handle) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckValidExternalBusHandle(handle));

    bool isConnected;
    auto hidHandle = ConvertHandleFromHidbusToHid(handle);
    auto result = nn::hid::IsExternalBusDeviceConnected(&isConnected, hidHandle);
    if (result.IsFailure())
    {
        return ConvertHidResultToHidBusResult(result);
    }
    *isAttached = isConnected;
    NN_RESULT_SUCCESS;
}

nn::Result HidbusServer::Initialize(nn::hidbus::BusHandle handle, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    // handle の有効性チェック
    NN_RESULT_DO(CheckValidExternalBusHandle(handle));

    auto index = nn::hidbus::detail::GetInternalIndexFromHandle(handle);

    GetResourceHolder().LockInternalStateMutex(index);
    NN_UTIL_SCOPE_EXIT
    {
        GetResourceHolder().UnlockInternalStateMutex(index);
    };

    if (!GetResourceHolder().IsResourceInitialized(index))
    {
        auto hidHandle = ConvertHandleFromHidbusToHid(handle);
        auto result = nn::hid::InitializeExternalBus(hidHandle);
        GetSchedulerTask().Activate(handle);
        GetResourceHolder().SetInitializedAppletResourceUserId(aruid, handle);
        GetResourceHolder().SetResourceInitialized(true, index);
        return ConvertHidResultToHidBusResult(result);
    }
    else
    {
        return nn::hidbus::ResultAlreadyInitialized();
    }
}

nn::Result HidbusServer::Finalize(nn::hidbus::BusHandle handle, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    // 将来、アプレット対応等の複数クライアント対応する際に、誰が Finalize したか管理するために引数に取ってある。
    // 今は何も使わない。
    NN_UNUSED(aruid);

    // handle の有効性チェック
    NN_RESULT_DO(CheckValidExternalBusHandle(handle));

    auto index = nn::hidbus::detail::GetInternalIndexFromHandle(handle);

    GetResourceHolder().LockInternalStateMutex(index);
    NN_UTIL_SCOPE_EXIT
    {
        GetResourceHolder().UnlockInternalStateMutex(index);
    };

    if (GetResourceHolder().GetStatusManager()->IsEnabled(index))
    {
        // Enable 状態の場合、 disable を試みる。(失敗しても、以後の処理は必要なので result はチェックしない)
        EnableExternalDevice(handle, false, 0, GetResourceHolder().GetInFocusAruid());
    }

    if (GetResourceHolder().IsResourceInitialized(index))
    {
        auto hidHandle = ConvertHandleFromHidbusToHid(handle);
        auto result = nn::hid::FinalizeExternalBus(hidHandle);
        GetSchedulerTask().Deactivate(handle);
        GetResourceHolder().SetInitializedAppletResourceUserId(nn::applet::AppletResourceUserId::GetInvalidId(), handle);
        GetResourceHolder().SetResourceInitialized(false, index);
        return ConvertHidResultToHidBusResult(result);
    }
    else
    {
        return nn::hidbus::ResultAlreadyFinalized();
    }
}

nn::Result HidbusServer::EnableExternalDevice(nn::hidbus::BusHandle handle, bool isEnabled, std::uint64_t version, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_DETAIL_HIDBUS_INFO("EnableExternalDevice() in Server\n");

    // handle が無効になっている場合、 SharedMemory の内容は更新されてない可能性があるため、一番初めにチェックする
    NN_RESULT_DO(CheckValidExternalBusHandle(handle));

    auto internalNumber = nn::hidbus::detail::GetInternalIndexFromHandle(handle);

    GetResourceHolder().LockInternalStateMutex(internalNumber);
    NN_UTIL_SCOPE_EXIT
    {
        GetResourceHolder().UnlockInternalStateMutex(internalNumber);
    };

    auto hidHandle = ConvertHandleFromHidbusToHid(handle);

    // Resource が Initialize 済みかチェック
    if (!GetResourceHolder().IsResourceInitialized(internalNumber))
    {
        return nn::hidbus::ResultNotInitialized();
    }

    if (GetResourceHolder().GetInFocusAruid() != aruid)
    {
        return nn::hidbus::ResultNotInForcus();
    }

    if (!isEnabled && GetResourceHolder().GetStatusManager()->IsPollingMode(internalNumber))
    {
        NN_DETAIL_HIDBUS_INFO("Call DisableJoyPollingReceiveMode() in EnableExternalDevice()\n");
        // Disable 要求かつ、PollingMode なら、先に PollingMode を Disable にする。
        NN_RESULT_DO(DisableJoyPollingReceiveMode(handle));
    }

    if (!isEnabled && !GetResourceHolder().GetStatusManager()->IsEnabled(internalNumber))
    {
        // Disable 状態で Disable が呼ばれたときは以降は成功扱いにして返す
        NN_RESULT_SUCCESS;
    }

    auto result = nn::hid::EnableExternalBusDevice(hidHandle, isEnabled, version);

    if (result.IsSuccess())
    {
        if (isEnabled)
        {
            GetResourceHolder().GetStatusManager()->SetEnableState(true, internalNumber);
            GetResourceHolder().SetAppletResourceUserId(aruid, handle);
        }
        else
        {
            GetResourceHolder().GetStatusManager()->SetEnableState(false, internalNumber);
            GetResourceHolder().SetAppletResourceUserId(nn::applet::AppletResourceUserId::GetInvalidId(), handle);
        }
    }
    else if (nn::hid::ResultExternalBusDeviceInquiryTimeout::Includes(result) ||
             nn::hid::ResultExternalBusDeviceReadyTimeout::Includes(result))
    {
        // これらのエラーの場合、電源は Enable になっているものの、 Inquiry が失敗しているので、電源を落としておく
        auto hidResult = nn::hid::EnableExternalBusDevice(hidHandle, false, version);
        if (hidResult.IsFailure())
        {
            return ConvertHidResultToHidBusResult(hidResult);
        }
    }

    return ConvertHidResultToHidBusResult(result);
}

nn::Result HidbusServer::GetExternalDeviceId(nn::sf::Out<std::uint32_t> deviceId, nn::hidbus::BusHandle handle) NN_NOEXCEPT
{
    // handle の有効性チェック
    NN_RESULT_DO(CheckValidExternalBusHandle(handle));

    auto hidHandle = ConvertHandleFromHidbusToHid(handle);
    nn::hid::ExternalBusDeviceInfo info;
    auto result = nn::hid::GetExternalBusDeviceInfo(&info, hidHandle);
    NN_RESULT_DO(ConvertHidResultToHidBusResult(result));
    *deviceId = info.id;

    NN_RESULT_SUCCESS;
}

nn::Result HidbusServer::SendCommandAsync(const nn::sf::InBuffer& inData, nn::hidbus::BusHandle handle) NN_NOEXCEPT
{
    // handle の有効性チェック
    NN_RESULT_DO(CheckValidExternalBusHandle(handle));

    auto hidHandle = ConvertHandleFromHidbusToHid(handle);
    auto result = nn::hid::SendCommandAsyncToExternalBusDevice(reinterpret_cast<const uint8_t*>(inData.GetPointerUnsafe()), inData.GetSize(), hidHandle);
    return ConvertHidResultToHidBusResultForCommunicatin(result);
}

nn::Result HidbusServer::GetSendCommandAsynceResult(nn::sf::Out<std::uint32_t> outSize, const nn::sf::OutBuffer& outData, nn::hidbus::BusHandle handle) NN_NOEXCEPT
{
    // handle の有効性チェック
    NN_RESULT_DO(CheckValidExternalBusHandle(handle));

    size_t hidOutSize;
    auto hidHandle = ConvertHandleFromHidbusToHid(handle);
    auto result = nn::hid::GetSendCommandAsynceResultFromExternalBusDevice(&hidOutSize, reinterpret_cast<uint8_t*>(outData.GetPointerUnsafe()), outData.GetSize(), hidHandle);
    *outSize = static_cast<uint32_t>(hidOutSize);
    return ConvertHidResultToHidBusResultForCommunicatin(result);
}
nn::Result HidbusServer::SetEventForSendCommandAsycResult(nn::sf::Out<nn::sf::NativeHandle> attachmentDataReceiveEventHandle, nn::hidbus::BusHandle handle) NN_NOEXCEPT
{
    // handle の有効性チェック
    NN_RESULT_DO(CheckValidExternalBusHandle(handle));

    auto hidHandle = ConvertHandleFromHidbusToHid(handle);
    auto result = nn::hid::BindEventForExternalBusDeviceSendCommandAsycResult(attachmentDataReceiveEventHandle, hidHandle);
    return ConvertHidResultToHidBusResult(result);
}

nn::Result HidbusServer::GetSharedMemoryHandle(nn::sf::Out<nn::sf::NativeHandle> outValue) NN_NOEXCEPT
{
    auto osHandle = GetResourceHolder().GetSharedMemoryHandle();
    outValue.Set(::nn::sf::NativeHandle(osHandle, NeedsToBeManaged));
    NN_RESULT_SUCCESS;
}

nn::Result HidbusServer::EnableJoyPollingReceiveMode(const nn::sf::InBuffer& inData,
    nn::sf::NativeHandle&& transferMemoryHandle, std::uint32_t transferMemorySize,
    std::uint32_t mode, nn::hidbus::BusHandle handle) NN_NOEXCEPT
{
    // handle の有効性チェック
    NN_RESULT_DO(CheckValidExternalBusHandle(handle));

    auto internalNumber = nn::hidbus::detail::GetInternalIndexFromHandle(handle);
    GetResourceHolder().LockInternalStateMutex(internalNumber);
    NN_UTIL_SCOPE_EXIT
    {
        GetResourceHolder().UnlockInternalStateMutex(internalNumber);
    };

    NN_DETAIL_HIDBUS_INFO("EnableJoyPollingReceiveMode\n");

    auto hidHandle = ConvertHandleFromHidbusToHid(handle);
    auto pTransferMemoryMutex = GetResourceHolder().GetTransferMemoryMutexPointer(internalNumber);
    ::std::lock_guard<decltype(*pTransferMemoryMutex)> locker(*pTransferMemoryMutex);
    // Mutex での排他処理後に裏で Disable になっている可能性があるので、ここで Enable かどうかチェックする
    if (!GetResourceHolder().GetStatusManager()->IsEnabled(internalNumber))
    {
        return nn::hidbus::ResultExternalDeviceNotEnabled();
    }

    auto osHandle = transferMemoryHandle.GetOsHandle();
    auto managed = transferMemoryHandle.IsManaged();
    transferMemoryHandle.Detach();

    auto pTransferMemory = GetResourceHolder().GetTransferMemoryType(handle);

    ::nn::os::AttachTransferMemory(
            pTransferMemory, transferMemorySize, osHandle, managed);

    void* address = nullptr;
    NN_RESULT_DO(::nn::os::MapTransferMemory(
        &address,
        pTransferMemory,
        ::nn::os::MemoryPermission_ReadOnly));
    NN_DETAIL_HIDBUS_INFO("MapTransferMemory in Server\n");

    GetResourceHolder().SetTransferMemory(address, transferMemorySize, handle);

    void* pAccessor = nullptr;

    // 取得したアドレスに PollingDataAccessor を構築する
    switch (static_cast<nn::hidbus::JoyPollingMode>(mode))
    {
    case nn::hidbus::JoyPollingMode_SixAxisSensorEnable:
        pAccessor = reinterpret_cast<nn::hidbus::detail::JoyEnableSixAxisPollingDataAccessor*>(address);
        NN_SDK_REQUIRES_NOT_NULL(pAccessor);

        new (pAccessor) nn::hidbus::detail::JoyEnableSixAxisPollingDataAccessor();
        break;

    case nn::hidbus::JoyPollingMode_SixAxisSensorDisable:
        pAccessor = reinterpret_cast<nn::hidbus::detail::JoyDisableSixAxisPollingDataAccessor*>(address);
        NN_SDK_REQUIRES_NOT_NULL(pAccessor);

        new (pAccessor) nn::hidbus::detail::JoyDisableSixAxisPollingDataAccessor();
        break;

    default:NN_UNEXPECTED_DEFAULT;
    }

    auto result = nn::hid::EnablePollingReceiveModeForAttachmentDevice(reinterpret_cast<const uint8_t*>(inData.GetPointerUnsafe()), inData.GetSize(),
        hidHandle, static_cast<nn::hid::ExternalBusJoyPollingMode>(mode));
    if (result.IsSuccess())
    {
        GetSchedulerTask().StartPollingMode(static_cast<nn::hidbus::JoyPollingMode>(mode), handle);
        GetResourceHolder().GetStatusManager()->SetPollingModeState(true, internalNumber);
    }
    else
    {
        // 実際にポーリングモードに設定できなかったら、 Unmap しておく。
        NN_DETAIL_HIDBUS_INFO("UnmapTransferMemory in Server\n");
        nn::os::UnmapTransferMemory(pTransferMemory);
        nn::os::DestroyTransferMemory(pTransferMemory);
    }
    return ConvertHidResultToHidBusResult(result);
}

nn::Result HidbusServer::DisableJoyPollingReceiveMode(nn::hidbus::BusHandle handle) NN_NOEXCEPT
{
    // handle の有効性チェック
    NN_RESULT_DO(CheckValidExternalBusHandle(handle));

    auto internalNumber = nn::hidbus::detail::GetInternalIndexFromHandle(handle);
    GetResourceHolder().LockInternalStateMutex(internalNumber);
    NN_UTIL_SCOPE_EXIT
    {
        GetResourceHolder().UnlockInternalStateMutex(internalNumber);
    };

    if (!GetResourceHolder().GetStatusManager()->IsPollingMode(internalNumber))
    {
        // PollingMode になってない時はすぐに返る
        NN_RESULT_SUCCESS;
    }

    auto hidHandle = ConvertHandleFromHidbusToHid(handle);
    auto result = nn::hid::DisablePollingReceiveModeForAttachmentDevice(hidHandle);
    if (result.IsSuccess())
    {
        GetSchedulerTask().StopPollingMode(handle);
    }
    auto pTransferMemoryType = GetResourceHolder().GetTransferMemoryType(handle);
    nn::os::UnmapTransferMemory(pTransferMemoryType);
    nn::os::DestroyTransferMemory(pTransferMemoryType);
    GetResourceHolder().GetStatusManager()->SetPollingModeState(false, internalNumber);

    return ConvertHidResultToHidBusResult(result);
}

nn::Result HidbusServer::GetPollingData(const nn::sf::OutArray<nn::hidbus::JoyPollingReceivedData>& pOutArray, nn::hidbus::BusHandle handle) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_WIN)
    // handle の有効性チェック
    NN_RESULT_DO(CheckValidExternalBusHandle(handle));

    auto internalNumber = nn::hidbus::detail::GetInternalIndexFromHandle(handle);

    auto pollingMode = GetResourceHolder().GetStatusManager()->GetPollingMode(internalNumber);
    switch (static_cast<nn::hidbus::JoyPollingMode>(pollingMode))
    {
    case nn::hidbus::JoyPollingMode_SixAxisSensorEnable:
    {
        auto length = pOutArray.GetLength() <= detail::JoyConEnableSixAxisSensorLifoCount ? pOutArray.GetLength() : detail::JoyConEnableSixAxisSensorLifoCount;

        detail::JoyConEnableSixAxisSensorLifoFormat format[detail::JoyConEnableSixAxisSensorLifoCount];
        GetJoyEnableSixAxisPollingDataAccessor(internalNumber)->GetPollingData(format, static_cast<int>(length));

        for (int i = 0; i < static_cast<int>(length); ++i)
        {
            memcpy(pOutArray[i].data, format[i].data, format[i].dataSize);
            pOutArray[i].outSize = format[i].dataSize;
            pOutArray[i].samplingNumber = format[i].samplingNumber;
        }
    }
    break;

    case nn::hidbus::JoyPollingMode_SixAxisSensorDisable:
    {
        auto length = pOutArray.GetLength() <= detail::JoyConDisableSixAxisSensorLifoCount ? pOutArray.GetLength() : detail::JoyConDisableSixAxisSensorLifoCount;

        detail::JoyConDisableSixAxisSensorLifoFormat format[detail::JoyConDisableSixAxisSensorLifoCount];
        GetJoyDisableSixAxisPollingDataAccessor(internalNumber)->GetPollingData(format, static_cast<int>(length));

        for (int i = 0; i < static_cast<int>(length); ++i)
        {
            memcpy(pOutArray[i].data, format[i].data, format[i].dataSize);
            pOutArray[i].outSize = format[i].dataSize;
            pOutArray[i].samplingNumber = format[i].samplingNumber;
        }
    }
    break;

    default:NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
#else
    // 本 API は Windows 環境のみで使用されるため、他の環境では何もしない。
    NN_UNUSED(pOutArray);
    NN_UNUSED(handle);
    NN_RESULT_SUCCESS;
#endif
}

::nn::Result CreateHidbusServerProxy(::nn::sf::SharedPointer<IHidbusServer>* outValue) NN_NOEXCEPT
{
    *outValue = nn::hidbus::detail::StaticObject<
        ::nn::sf::UnmanagedServiceObject<IHidbusServer, HidbusServer>
    >::Get().GetShared();
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hidbus::server
