﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Macro.h>
#include <nn/bluetooth/bluetooth_AddressTypes.h>
#include <nn/hid/system/hid_RegisteredDevice.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/hid/debug/hid_RegisteredDevice.h>
#include <nn/result/result_HandlingUtility.h>

#include "detail/hid_RegisteredDeviceImpl.h"

namespace nn { namespace hid { namespace system {

int GetRegisteredDevices(RegisteredDevice* pOutValues, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValues);
    NN_SDK_REQUIRES_GREATER(count, 0);

    int returnCount = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::hid::detail::GetRegisteredDevices(&returnCount, pOutValues, count));

    return returnCount;
}

int GetConnectableRegisteredDevices(RegisteredDevice* pOutValues, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValues);
    NN_SDK_REQUIRES_GREATER(count, 0);

    int returnCount = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::hid::detail::GetConnectableRegisteredDevices(&returnCount, pOutValues, count));

    return returnCount;
}

void BindConnectionTriggerTimeoutEvent(nn::os::SystemEventType* pSystemEvent, nn::os::EventClearMode clearMode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::hid::detail::BindConnectionTriggerTimeoutEvent(pSystemEvent, clearMode));
}

Result SendConnectionTrigger(nn::bluetooth::Address address) NN_NOEXCEPT
{
    auto result = ::nn::hid::detail::SendConnectionTrigger(address);
    NN_RESULT_TRY(result)
    NN_RESULT_CATCH(system::ResultBluetoothError)
    {
        NN_RESULT_THROW(result);
    }
    NN_RESULT_END_TRY;
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_RESULT_SUCCESS;
}

void BindDeviceRegisteredEventForControllerSupport(nn::os::SystemEventType* pSystemEvent, nn::os::EventClearMode clearMode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::hid::detail::BindDeviceRegisteredEventForControllerSupport(pSystemEvent, clearMode));
}

int GetAllowedBluetoothLinksCount() NN_NOEXCEPT
{
    return ::nn::hid::detail::GetAllowedBluetoothLinksCount();
}

}}} // namespace nn::hid::system

namespace nn { namespace hid { namespace debug {

void AddRegisteredDevice(system::RegisteredDevice device) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::hid::detail::AddRegisteredDevice(device));
}

}}} // namespace nn::hid::debug
