﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <utility>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_Keyboard.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/hid/hid_KeyCode.h>
#include <nn/kpr/kpr_KeyCode.h>
#include <nn/kpr/detail/kpr_KeyCodeMap.h>
#include <nn/kpr/detail/kpr_KeyCodeMap-layout.ChineseTraditional.h>
#include <nn/kpr/detail/kpr_KeyCodeMap-layout.EnglishUk.h>
#include <nn/kpr/detail/kpr_KeyCodeMap-layout.EnglishUs.h>
#include <nn/kpr/detail/kpr_KeyCodeMap-layout.EnglishUsInternational.h>
#include <nn/kpr/detail/kpr_KeyCodeMap-layout.French.h>
#include <nn/kpr/detail/kpr_KeyCodeMap-layout.FrenchCa.h>
#include <nn/kpr/detail/kpr_KeyCodeMap-layout.German.h>
#include <nn/kpr/detail/kpr_KeyCodeMap-layout.Italian.h>
#include <nn/kpr/detail/kpr_KeyCodeMap-layout.Japanese.h>
#include <nn/kpr/detail/kpr_KeyCodeMap-layout.Korean.h>
#include <nn/kpr/detail/kpr_KeyCodeMap-layout.Portuguese.h>
#include <nn/kpr/detail/kpr_KeyCodeMap-layout.Russian.h>
#include <nn/kpr/detail/kpr_KeyCodeMap-layout.Spanish.h>
#include <nn/kpr/detail/kpr_KeyCodeMap-layout.SpanishLatin.h>

#include "../kpr/detail/kpr_KeyCodeImpl.h"

namespace nn { namespace hid {

namespace {

//!< レイアウトの内部値定義です。
struct LayoutValue final
{
    static const uint32_t EnglishUnitedKingdom = 0x08090809;
    static const uint32_t EnglishUnitedStates = 0x04090409;
    static const uint32_t EnglishUnitedStatesInternational = 0xF0010409;
    static const uint32_t French = 0x040C040C;
    static const uint32_t FrenchCanada = 0xF0200C0C;
    static const uint32_t German = 0x04070407;
    static const uint32_t Italian = 0x04100410;
    static const uint32_t Japanese = 0x04110411;
    static const uint32_t Portuguese = 0x08160816;
    static const uint32_t RussianCyrillic = 0x04190419;
    static const uint32_t RussianLatin = 0x04190409;
    static const uint32_t Spanish = 0x040A0C0A;
    static const uint32_t SpanishLatinAmerica = 0x080A580A;
    static const uint32_t KoreanHangul = 0x04120412;
    static const uint32_t KoreanLatin = 0x04120409;
    static const uint32_t ChineseSimplified = 0x08040804;
    static const uint32_t ChineseTraditionalZhuyin = 0x04040404;
    static const uint32_t ChineseTraditionalCangjie = 0x04040C04;
    static const uint32_t ChineseTraditionalLatin = 0x04040409;
};

//!< キーコードマップを返します。
const ::nn::kpr::detail::KeyCodeMapStorage& GetKeyCodeMapStorage(
    KeyboardLayoutType layout) NN_NOEXCEPT;

//!< キーコードマップの変換モードを返します。
::nn::kpr::detail::KeyCodeMapInternalMode GetKeyCodeMapInternalMode(
    KeyboardModifierSet modifiers, KeyboardLayoutType layout) NN_NOEXCEPT;

//!< メタキーが押下中か否かを表す値を返します。
bool IsMetaKeyOn(KeyboardModifierSet modifiers) NN_NOEXCEPT;

//!< ラテン文字を小文字に変換します。
uint16_t ConvertLowerCase(uint16_t code) NN_NOEXCEPT;

} // namespace

const KeyboardLayoutType KeyboardLayout::EnglishUnitedKingdom =
{ LayoutValue::EnglishUnitedKingdom };
const KeyboardLayoutType KeyboardLayout::EnglishUnitedStates =
{ LayoutValue::EnglishUnitedStates };
const KeyboardLayoutType KeyboardLayout::EnglishUnitedStatesInternational =
{ LayoutValue::EnglishUnitedStatesInternational };
const KeyboardLayoutType KeyboardLayout::French =
{ LayoutValue::French };
const KeyboardLayoutType KeyboardLayout::FrenchCanada =
{ LayoutValue::FrenchCanada };
const KeyboardLayoutType KeyboardLayout::German =
{ LayoutValue::German };
const KeyboardLayoutType KeyboardLayout::Italian =
{ LayoutValue::Italian };
const KeyboardLayoutType KeyboardLayout::Japanese =
{ LayoutValue::Japanese };
const KeyboardLayoutType KeyboardLayout::Portuguese =
{ LayoutValue::Portuguese };
const KeyboardLayoutType KeyboardLayout::RussianCyrillic =
{ LayoutValue::RussianCyrillic };
const KeyboardLayoutType KeyboardLayout::RussianLatin =
{ LayoutValue::RussianLatin };
const KeyboardLayoutType KeyboardLayout::Spanish =
{ LayoutValue::Spanish };
const KeyboardLayoutType KeyboardLayout::SpanishLatinAmerica =
{ LayoutValue::SpanishLatinAmerica };
const KeyboardLayoutType KeyboardLayout::KoreanHangul =
{ LayoutValue::KoreanHangul };
const KeyboardLayoutType KeyboardLayout::KoreanLatin =
{ LayoutValue::KoreanLatin };
const KeyboardLayoutType KeyboardLayout::ChineseSimplified =
{ LayoutValue::ChineseSimplified };
const KeyboardLayoutType KeyboardLayout::ChineseTraditionalZhuyin =
{ LayoutValue::ChineseTraditionalZhuyin };
const KeyboardLayoutType KeyboardLayout::ChineseTraditionalCangjie =
{ LayoutValue::ChineseTraditionalCangjie };
const KeyboardLayoutType KeyboardLayout::ChineseTraditionalLatin =
{ LayoutValue::ChineseTraditionalLatin };

void GetKeyCode(uint16_t* pOutValue,
                int usageId,
                KeyboardModifierSet modifiers,
                KeyboardLayoutType layout) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_MINMAX(usageId, 0x0000, 0xFFFF);
    const bool isMetaKeyOn = IsMetaKeyOn(modifiers);
    ::nn::kpr::KeyCode outValue = {};
    ::nn::kpr::detail::GetKeyCode(
        &outValue, usageId, modifiers,
        GetKeyCodeMapStorage(layout),
        isMetaKeyOn ? ::nn::kpr::detail::KeyCodeMapInternalMode::Default
                    : GetKeyCodeMapInternalMode(modifiers, layout));
    *pOutValue = isMetaKeyOn ? ConvertLowerCase(outValue.code) : outValue.code;
}

namespace {

const ::nn::kpr::detail::KeyCodeMapStorage& GetKeyCodeMapStorage(
    KeyboardLayoutType layout) NN_NOEXCEPT
{
    switch (layout.storage)
    {
    case LayoutValue::EnglishUnitedKingdom:
        return ::nn::kpr::detail::KeyCodeMapEnglishUk;
    case LayoutValue::EnglishUnitedStates:
        return ::nn::kpr::detail::KeyCodeMapEnglishUs;
    case LayoutValue::EnglishUnitedStatesInternational:
        return ::nn::kpr::detail::KeyCodeMapEnglishUsInternational;
    case LayoutValue::French:
        return ::nn::kpr::detail::KeyCodeMapFrench;
    case LayoutValue::FrenchCanada:
        return ::nn::kpr::detail::KeyCodeMapFrenchCa;
    case LayoutValue::German:
        return ::nn::kpr::detail::KeyCodeMapGerman;
    case LayoutValue::Italian:
        return ::nn::kpr::detail::KeyCodeMapItalian;
    case LayoutValue::Japanese:
        return ::nn::kpr::detail::KeyCodeMapJapanese;
    case LayoutValue::Portuguese:
        return ::nn::kpr::detail::KeyCodeMapPortuguese;
    case LayoutValue::RussianCyrillic:
    case LayoutValue::RussianLatin:
        return ::nn::kpr::detail::KeyCodeMapRussian;
    case LayoutValue::Spanish:
        return ::nn::kpr::detail::KeyCodeMapSpanish;
    case LayoutValue::SpanishLatinAmerica:
        return ::nn::kpr::detail::KeyCodeMapSpanishLatin;
    case LayoutValue::KoreanHangul:
    case LayoutValue::KoreanLatin:
        return ::nn::kpr::detail::KeyCodeMapKorean;
    case LayoutValue::ChineseSimplified:
        return ::nn::kpr::detail::KeyCodeMapEnglishUs;
    case LayoutValue::ChineseTraditionalZhuyin:
    case LayoutValue::ChineseTraditionalCangjie:
    case LayoutValue::ChineseTraditionalLatin:
        return ::nn::kpr::detail::KeyCodeMapChineseTraditional;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

::nn::kpr::detail::KeyCodeMapInternalMode GetKeyCodeMapInternalMode(
    KeyboardModifierSet modifiers, KeyboardLayoutType layout) NN_NOEXCEPT
{
    switch (layout.storage)
    {
    case LayoutValue::EnglishUnitedKingdom:
        return ::nn::kpr::detail::KeyCodeMapInternalMode::Default;
    case LayoutValue::EnglishUnitedStates:
        return ::nn::kpr::detail::KeyCodeMapInternalMode::Default;
    case LayoutValue::EnglishUnitedStatesInternational:
        return ::nn::kpr::detail::KeyCodeMapInternalMode::Default;
    case LayoutValue::French:
        return ::nn::kpr::detail::KeyCodeMapInternalMode::Default;
    case LayoutValue::FrenchCanada:
        return ::nn::kpr::detail::KeyCodeMapInternalMode::Default;
    case LayoutValue::German:
        return ::nn::kpr::detail::KeyCodeMapInternalMode::Default;
    case LayoutValue::Italian:
        return ::nn::kpr::detail::KeyCodeMapInternalMode::Default;
    case LayoutValue::Japanese:
        return modifiers.Test<KeyboardModifier::Hiragana>()
            ? ::nn::kpr::detail::KeyCodeMapInternalMode::Hiragana
            : (modifiers.Test<KeyboardModifier::Katakana>()
                   ? ::nn::kpr::detail::KeyCodeMapInternalMode::Katakana
                   : ::nn::kpr::detail::KeyCodeMapInternalMode::Default);
    case LayoutValue::Portuguese:
        return ::nn::kpr::detail::KeyCodeMapInternalMode::Default;
    case LayoutValue::RussianCyrillic:
        return ::nn::kpr::detail::KeyCodeMapInternalMode::Cyrillic;
    case LayoutValue::RussianLatin:
        return ::nn::kpr::detail::KeyCodeMapInternalMode::Default;
    case LayoutValue::Spanish:
        return ::nn::kpr::detail::KeyCodeMapInternalMode::Default;
    case LayoutValue::SpanishLatinAmerica:
        return ::nn::kpr::detail::KeyCodeMapInternalMode::Default;
    case LayoutValue::KoreanHangul:
        return ::nn::kpr::detail::KeyCodeMapInternalMode::Hangul;
    case LayoutValue::KoreanLatin:
        return ::nn::kpr::detail::KeyCodeMapInternalMode::Default;
    case LayoutValue::ChineseSimplified:
        return ::nn::kpr::detail::KeyCodeMapInternalMode::Default;
    case LayoutValue::ChineseTraditionalZhuyin:
        return ::nn::kpr::detail::KeyCodeMapInternalMode::Zhuyin;
    case LayoutValue::ChineseTraditionalCangjie:
        return ::nn::kpr::detail::KeyCodeMapInternalMode::Cangjie;
    case LayoutValue::ChineseTraditionalLatin:
        return ::nn::kpr::detail::KeyCodeMapInternalMode::Default;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

bool IsMetaKeyOn(KeyboardModifierSet modifiers) NN_NOEXCEPT
{
    return modifiers.Test<KeyboardModifier::Gui>()
        || modifiers.Test<KeyboardModifier::Control>()
        || modifiers.Test<KeyboardModifier::LeftAlt>()
        || modifiers.Test<KeyboardModifier::RightAlt>();
}

uint16_t ConvertLowerCase(uint16_t code) NN_NOEXCEPT
{
    if (code < 'A' || 'Z' < code)
    {
        return code;
    }
    else
    {
        return code + 'a' - 'A';
    }
}

} // namespace

}} // namespace nn::hid
