﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_Mutex.h>
#include <nn/settings/settings_Xpad.h>

namespace nn { namespace hid { namespace detail {

//!< Xpad 用のマッピングの解決を担うマネージャを扱うクラスです。
class XpadMappingManager final
{
    NN_DISALLOW_COPY(XpadMappingManager);
    NN_DISALLOW_MOVE(XpadMappingManager);

private:
    //!< 基本的な構成を持つ Xpad 用の割り当てを表す構造体です。
    struct BasicXpadMap final
    {
        //!< 汎用コントローラ間の割り当てが初期化済みか否かを表す値
        ::std::atomic<bool> isGenericPadMapInitialized;

        //!< 汎用コントローラ間の割り当て
        ::std::atomic<::nn::settings::BasicXpadGenericPadMap> genericPadMap;

        //!< キーボード間の割り当て
        ::std::atomic<::nn::settings::BasicXpadKeyboardMap> keyboardMap;
    };

    //!< マッピングの初期化処理排他用のミューテックス
    ::nn::os::Mutex m_Mutex;

    //!< 基本的な構成を持つ Xpad 用の割り当て
    BasicXpadMap m_BasicXpadMap;

public:
    XpadMappingManager() NN_NOEXCEPT;

    //!< Xpad-汎用コントローラ間の割り当てが有効か否かを表す値を返します。
    bool IsValidBasicXpadGenericPadMap(
        const ::nn::settings::BasicXpadGenericPadMap& value) NN_NOEXCEPT;

    //!< Xpad-キーボード間の割り当てが有効か否かを表す値を返します。
    bool IsValidBasicXpadKeyboardMap(
        const ::nn::settings::BasicXpadKeyboardMap& value) NN_NOEXCEPT;

    ::nn::Result GetBasicXpadGenericPadMap(
        ::nn::settings::BasicXpadGenericPadMap* pOutValue) NN_NOEXCEPT;

    ::nn::Result SetBasicXpadGenericPadMap(
        const ::nn::settings::BasicXpadGenericPadMap& value) NN_NOEXCEPT;

    ::nn::Result ResetBasicXpadGenericPadMap() NN_NOEXCEPT;

    ::nn::Result GetBasicXpadKeyboardMap(
        ::nn::settings::BasicXpadKeyboardMap* pOutValue) NN_NOEXCEPT;

    ::nn::Result SetBasicXpadKeyboardMap(
        const ::nn::settings::BasicXpadKeyboardMap& value) NN_NOEXCEPT;

    ::nn::Result ResetBasicXpadKeyboardMap() NN_NOEXCEPT;

private:
    //!< Xpad-汎用コントローラ間の割り当ての規定値を取得します。
    static void GetDefaultBasicXpadGenericPadMap(
        ::nn::settings::BasicXpadGenericPadMap* pOutValue) NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
