﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdio>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_Result.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/hid/detail/hid_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringUtil.h>
#include <nn/xcd/xcd.h>

#include "hid_AbstractedPadXcd.h"
#include "hid_XcdFirmwareUpdater.h"

namespace nn { namespace hid { namespace detail {

namespace
{

// 標準で使用する、コントローラーファームウェアのシステムデータ ID
const nn::ncm::SystemDataId SystemDataId = { 0x0100000000000822 };

// デバッグ/巻き戻し用のコントローラーファームウェアのシステムデータ ID
const nn::ncm::SystemDataId SystemDataIdForDebug = { 0x010000000000B22B };

// 標準で使用するシステムデータ用のマウント名
const char MountNameNormal[] = "systemData";

// デバッグ/巻き戻し用システムデータのマウント名
const char MountNameDebug[] = "systemDataD";

// システムデータ内のファームウェアイメージファイル名の最大長
const size_t FirmwareFilenameLengthMax = 64;

// リセット後の再接続失敗と見なすまでの時間
const auto ReconnectionTimeout = nn::TimeSpan::FromSeconds(60);

// LLR が不発した場合にハマるのを防ぐためのタイムアウト
const auto ConnectionTriggerFailureTimeout = nn::TimeSpan::FromSeconds(10) + ReconnectionTimeout;

// 対応している接続インターフェースか確認
nn::Result CheckInterfaceType(nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    nn::xcd::DeviceStatus status;
    NN_RESULT_THROW_UNLESS(
        nn::xcd::GetDeviceStatus(&status, handle).IsSuccess(),
        system::ResultUniquePadDisconnected());

    NN_RESULT_THROW_UNLESS(
        status.interfaceType == nn::xcd::InterfaceType_Bluetooth || status.interfaceType == nn::xcd::InterfaceType_Uart,
        system::ResultUniquePadDisconnected());

    NN_RESULT_THROW_UNLESS(
        nn::xcd::IsBtFirmwareUpdateThroughBtRequired(handle) == false,
        system::ResultFirmwareUpdateRequireBluetooth()
    );

    NN_RESULT_SUCCESS;
}

// IAP 領域の更新要否判定
bool NeedsIapUpdate(nn::xcd::DeviceType type, const nn::xcd::McuVersionData& version) NN_NOEXCEPT
{
    if (version.isIapCorrupted)
    {
        // IAP が壊れているので要更新
        return true;
    }
    else if (version.isCorrupted)
    {
        // 通常の FW 壊れ判定ができていれば新 IAP が生きている
        return false;
    }

    uint8_t oldIapVersionMajor;
    uint8_t oldIapVersionMinor;
    switch (type)
    {
    case nn::xcd::DeviceType_Right:
    case nn::xcd::DeviceType_FullKey:
        {
            // CC/CD 2.3 以前は IAP が古い
            oldIapVersionMajor = 2;
            oldIapVersionMinor = 3;
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    if (version.major < oldIapVersionMajor ||
        (version.major == oldIapVersionMajor && version.minor <= oldIapVersionMinor))
    {
        // 古い IAP なので要更新
        return true;
    }
    else
    {
        // IAP は更新済み
        return false;
    }
}

// MCU 搭載デバイスか判定
bool HasMcuDevice(const nn::xcd::DeviceInfo& deviceInfo) NN_NOEXCEPT
{
    return
        deviceInfo.deviceType == nn::xcd::DeviceType_FullKey ||
        deviceInfo.deviceType == nn::xcd::DeviceType_Right;
}

// デバイス識別子を取得
void GetDeviceIdentifier(
    system::FirmwareVersion* pOutVersion,
    const nn::xcd::DeviceInfo& deviceInfo) NN_NOEXCEPT
{
    // TODO: どこかで一元管理したい
    const char* identifier;
    switch (deviceInfo.deviceType)
    {
    case nn::xcd::DeviceType_Left:
        identifier = "JL";
        break;

    case nn::xcd::DeviceType_Right:
        identifier = "JR";
        break;

    case nn::xcd::DeviceType_FullKey:
        identifier = "FK";
        break;

    case nn::xcd::DeviceType_MiyabiLeft:
    case nn::xcd::DeviceType_MiyabiRight:
        identifier = "MY";
        break;

    case nn::xcd::DeviceType_Tarragon:
        identifier = "LC";
        break;

    default:
        identifier = "";
        break;
    }

    nn::util::Strlcpy(pOutVersion->identifier, identifier, sizeof(pOutVersion->identifier));
}

// Debug 用の UpdateStage を System 用に変換
system::FirmwareUpdateStage ConvertUpdateStageToSystem(debug::FirmwareUpdateStage stage) NN_NOEXCEPT
{
    switch (stage)
    {
    case debug::FirmwareUpdateStage_Preparing:
        return system::FirmwareUpdateStage_Preparing;

    case debug::FirmwareUpdateStage_BluetoothDownload:
    case debug::FirmwareUpdateStage_BluetoothVerify:
    case debug::FirmwareUpdateStage_BluetoothCommit:
    case debug::FirmwareUpdateStage_Reboot1:
    case debug::FirmwareUpdateStage_McuVersionCheck:
    case debug::FirmwareUpdateStage_McuBoot:
    case debug::FirmwareUpdateStage_McuErasing:
    case debug::FirmwareUpdateStage_McuWriting:
    case debug::FirmwareUpdateStage_McuReboot:
    case debug::FirmwareUpdateStage_McuVerify:
        return system::FirmwareUpdateStage_Updating;

    case debug::FirmwareUpdateStage_Reboot2:
        return system::FirmwareUpdateStage_Finalizing;

    case debug::FirmwareUpdateStage_Completed:
        return system::FirmwareUpdateStage_Completed;

    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

// BD アドレスから FirmwareUpdateDeviceHandle を生成する
system::FirmwareUpdateDeviceHandle GetFirmwareUpdateDeviceHandle(const nn::bluetooth::Address& address) NN_NOEXCEPT
{
    system::FirmwareUpdateDeviceHandle handle;
    NN_STATIC_ASSERT(sizeof(nn::bluetooth::Address) < sizeof(system::FirmwareUpdateDeviceHandle));
    std::memcpy(&handle, &address, sizeof(address));
    return handle;
}

// FirmwareUpdateDeviceHandle から BD アドレスを取得する
nn::bluetooth::Address GetBdAddressFromHandle(const system::FirmwareUpdateDeviceHandle& handle) NN_NOEXCEPT
{
    auto address = nn::bluetooth::Address();
    std::memcpy(&address, &handle, sizeof(address));
    return address;
}


}  // anonymous

#if defined(NN_BUILD_CONFIG_COMPILER_VC)

// C4351: 配列メンバの初期化が規定値で行われる旨の警告を抑止
#pragma warning(push)
#pragma warning(disable: 4351)

#endif  // if defined(NN_BUILD_CONFIG_COMPILER_VC)

XcdFirmwareUpdater::XcdFirmwareUpdater() NN_NOEXCEPT
    : m_Handle()
    , m_DeviceInfo()
    , m_InterfaceType()
    , m_ActivationCount()
    , m_pUniquePadManagers()
    , m_ControllerFirmwareAccessorNormal(SystemDataId, MountNameNormal)
    , m_ControllerFirmwareAccessorDebug(SystemDataIdForDebug, MountNameDebug)
    , m_pActiveControllerFirmwareAccessor(&m_ControllerFirmwareAccessorNormal)
    , m_BluetoothFile()
    , m_McuFile()
    , m_FirmwareMemory()
    , m_pXcdBluetoothUpdateEvent(nullptr)
    , m_pXcdMcuUpdateEvent(nullptr)
    , m_pSampleUpdateEvent(nullptr)
    , m_pResetTimeoutEvent(nullptr)
    , m_FirmwareUpdateTarget(FirmwareUpdateTarget::None)
    , m_FirmwareUpdateState(FirmwareUpdateState_NotStarted)
    , m_UpdateOption()
    , m_IsReconnectRequired(false)
    , m_ReconnectionStartTick()
    , m_pInterruptSceneNotifier(nullptr)
    , m_ForegroundAruid()
    , m_IsFirmwareHotfixUpdateSkipFlagEnabled(false)
{
}

#if defined(NN_BUILD_CONFIG_COMPILER_VC)
#pragma warning(pop)
#endif  // if defined(NN_BUILD_CONFIG_COMPILER_VC)

XcdFirmwareUpdater::~XcdFirmwareUpdater() NN_NOEXCEPT
{
}

void XcdFirmwareUpdater::SetBluetoothUpdateEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);

    m_pXcdBluetoothUpdateEvent = pEvent;
}

void XcdFirmwareUpdater::SetMcuUpdateEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);

    m_pXcdMcuUpdateEvent = pEvent;
}

void XcdFirmwareUpdater::SetSampleUpdateEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);

    m_pSampleUpdateEvent = pEvent;
}

void XcdFirmwareUpdater::SetResetTimeoutEvent(nn::os::TimerEventType* pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);

    m_pResetTimeoutEvent = pEvent;
}

void XcdFirmwareUpdater::SetUniquePadManagers(int index, UniquePadManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    m_pUniquePadManagers[index] = pManager;
}

void XcdFirmwareUpdater::SetInterruptSceneNotifier(InterruptSceneNotifier* pNotifier) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNotifier);
    m_pInterruptSceneNotifier = pNotifier;
}

void XcdFirmwareUpdater::NotifyApplicationResourceUserId(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    // ARUID が変化した場合は中断
    if (IsUpdateRunning() && m_ForegroundAruid != aruid)
    {
        AbortFirmwareUpdate();
    }

    m_ForegroundAruid = aruid;
}

Result XcdFirmwareUpdater::ActivateFirmwareUpdate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
        system::ResultFirmwareUpdateActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // 何もしない
    }

    // アクティブ化された回数を増加
    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

Result XcdFirmwareUpdater::DeactivateFirmwareUpdate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
        system::ResultFirmwareUpdateDeactivationLowerLimitOver());

    // アクティブ化された回数を減少
    --m_ActivationCount;

    if (m_ActivationCount.IsZero())
    {
    }

    NN_RESULT_SUCCESS;
}

Result XcdFirmwareUpdater::DiscardSystemDataCacheForRevert() NN_NOEXCEPT
{
    m_ControllerFirmwareAccessorDebug.ClearFirmwareInfo();
    NN_RESULT_SUCCESS;
}

Result XcdFirmwareUpdater::SetFirmwareHotfixUpdateSkipEnabled(bool isEnabled) NN_NOEXCEPT
{
    m_IsFirmwareHotfixUpdateSkipFlagEnabled = isEnabled;
    NN_RESULT_SUCCESS;
}

Result XcdFirmwareUpdater::StartFirmwareUpdateImpl(
    nn::xcd::DeviceHandle xcdHandle) NN_NOEXCEPT
{
    m_Handle = xcdHandle;
    m_UpdateOption.Set<UpdateOption::NeedsMcuFullUpdate>(false);
    m_IsReconnectRequired = false;
    ::nn::os::StopTimerEvent(m_pResetTimeoutEvent);

    // Bluetooth アドレスを読み出す
    // 再起動時のデバイス探査を行うため
    nn::xcd::GetDeviceInfo(&m_DeviceInfo, m_Handle);

    // 接続インターフェースを取得 (更新中のインターフェース変更検知用)
    nn::xcd::DeviceStatus status;
    NN_RESULT_THROW_UNLESS(
        nn::xcd::GetDeviceStatus(&status, m_Handle).IsSuccess(),
        system::ResultUniquePadDisconnected()
    );
    m_InterfaceType = status.interfaceType;

    // システムデータへのアクセス開始
    NN_RESULT_THROW_UNLESS(
        m_pActiveControllerFirmwareAccessor->Mount().IsSuccess(),
        system::ResultFirmwareImageReadFailed()
    );

    // 更新開始前にエラーが発生した場合はマウントを解除する
    bool needsRollback = true;
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            m_pActiveControllerFirmwareAccessor->Unmount();
            nn::xcd::UnsetFirmwareUpdatingDevice();
        }
    };

    // FW 更新対象のデバイスを通知
    NN_RESULT_THROW_UNLESS(
        nn::xcd::SetFirmwareUpdatingDevice(m_Handle).IsSuccess(),
        system::ResultUniquePadDisconnected()
    );

    // 更新の要否判定
    bool needsBtUpdate;
    bool needsMcuUpdate;
    NN_RESULT_DO(NeedsBluetoothUpdate(&needsBtUpdate));
    NN_RESULT_DO(NeedsMcuUpdate(&needsMcuUpdate));

    if (needsBtUpdate)
    {
        m_FirmwareUpdateTarget =
            needsMcuUpdate ?
            FirmwareUpdateTarget::BluetoothAndMcu :
            FirmwareUpdateTarget::Bluetooth;
        NN_RESULT_DO(StartBluetoothUpdate());
        needsRollback = false;
    }
    else
    {
        if (needsMcuUpdate)
        {
            // MCU の更新から開始
            m_FirmwareUpdateTarget = FirmwareUpdateTarget::Mcu;
            NN_RESULT_DO(PrepareMcuUpdate());
            needsRollback = false;
        }
        else
        {
            // いずれも最新のため更新不要
            m_FirmwareUpdateTarget = FirmwareUpdateTarget::None;
            NN_RESULT_THROW(system::ResultFirmwareUpdateNotNeeded());
        }
    }

    NN_RESULT_SUCCESS;
}

Result XcdFirmwareUpdater::StartFirmwareUpdateImplByTransferMemory(
    nn::xcd::DeviceHandle xcdHandle,
    FirmwareUpdateTarget target) NN_NOEXCEPT
{
    m_Handle = xcdHandle;
    m_UpdateOption.Set<UpdateOption::NeedsMcuFullUpdate>(false);
    m_IsReconnectRequired = false;
    ::nn::os::StopTimerEvent(m_pResetTimeoutEvent);

    // Bluetooth アドレスを読み出す
    // 再起動時のデバイス探査を行うため
    nn::xcd::GetDeviceInfo(&m_DeviceInfo, m_Handle);

    // 接続インターフェースを取得 (更新中のインターフェース変更検知用)
    nn::xcd::DeviceStatus status;
    NN_RESULT_THROW_UNLESS(
        nn::xcd::GetDeviceStatus(&status, m_Handle).IsSuccess(),
        system::ResultUniquePadDisconnected()
    );
    m_InterfaceType = status.interfaceType;

    // FW 更新対象のデバイスを通知
    NN_RESULT_THROW_UNLESS(
        nn::xcd::SetFirmwareUpdatingDevice(m_Handle).IsSuccess(),
        system::ResultUniquePadDisconnected()
    );

    bool needsRollback = true;
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            nn::xcd::UnsetFirmwareUpdatingDevice();
        }
    };

    // 更新の要否判定
    bool needsUpdate = false;
    switch (target)
    {
    case FirmwareUpdateTarget::Bluetooth:
        {
            NN_RESULT_DO(NeedsBluetoothUpdate(&needsUpdate));
            if (needsUpdate)
            {
                m_FirmwareUpdateTarget = FirmwareUpdateTarget::Bluetooth;
                NN_RESULT_DO(StartBluetoothUpdate());
                needsRollback = false;
            }
        }
        break;

    case FirmwareUpdateTarget::Mcu:
        {
            NN_RESULT_DO(NeedsMcuUpdate(&needsUpdate));
            if (needsUpdate)
            {
                m_FirmwareUpdateTarget = FirmwareUpdateTarget::Mcu;
                NN_RESULT_DO(PrepareMcuUpdate());
                needsRollback = false;
            }
        }
        break;

    default:
        // 単独更新以外は非対応
        NN_RESULT_THROW(system::ResultFirmwareUpdateFailed());
    }

    if (!needsUpdate)
    {
        // 更新不要
        m_FirmwareUpdateTarget = FirmwareUpdateTarget::None;
        NN_RESULT_THROW(system::ResultFirmwareUpdateNotNeeded());
    }

    NN_RESULT_SUCCESS;
}

Result XcdFirmwareUpdater::StartFirmwareUpdateForSystem(
    system::FirmwareUpdateDeviceHandle* pOutDeviceHandle,
    system::UniquePadId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutDeviceHandle);

    if (IsUpdateRunning())
    {
        NN_RESULT_THROW(system::ResultFirmwareUpdateBusy());
    }

    auto handle = GetXcdDeviceHandle(id);
    NN_RESULT_DO(CheckInterfaceType(handle));

    // 電池残量判定
    nn::xcd::DeviceStatus status;
    NN_RESULT_THROW_UNLESS(
        nn::xcd::GetDeviceStatus(&status, handle).IsSuccess(),
        system::ResultUniquePadDisconnected()
    );

    // 充電中 or 電池残量が Low 以上でないと更新を許可しない
    if (!status.charged &&
        status.batteryLevel < nn::xcd::BatteryLevel_Low)
    {
        NN_RESULT_THROW(system::ResultFirmwareUpdateLowBattery());
    }

    // System 版 (本体機能用) は、バージョン判定あり、IAP 更新なし、完了後再接続
    m_UpdateOption.Reset();
    m_UpdateOption.Set<UpdateOption::VersionRestriction>(true);
    m_UpdateOption.Set<UpdateOption::PermitMcuFullUpdate>(false);
    m_UpdateOption.Set<UpdateOption::Reconnect>(true);

    // 標準のシステムデータを使う
    m_pActiveControllerFirmwareAccessor = &m_ControllerFirmwareAccessorNormal;

    auto result = StartFirmwareUpdateImpl(handle);
    if (result.IsSuccess())
    {
        // デバイス毎に固有になるように、BD アドレスからハンドルを生成する
        *pOutDeviceHandle = GetFirmwareUpdateDeviceHandle(m_DeviceInfo.address);
    }

    NN_RESULT_THROW(result);
}

Result XcdFirmwareUpdater::StartFirmwareUpdateForDebug(
    system::UniquePadId id) NN_NOEXCEPT
{
    if (IsUpdateRunning())
    {
        NN_RESULT_THROW(system::ResultFirmwareUpdateBusy());
    }

    auto handle = GetXcdDeviceHandle(id);
    NN_RESULT_DO(CheckInterfaceType(handle));

    // Debug 版 (ツール用) は、バージョン制限なし、IAP 更新あり、完了後切断
    m_UpdateOption.Reset();
    m_UpdateOption.Set<UpdateOption::VersionRestriction>(false);
    m_UpdateOption.Set<UpdateOption::PermitMcuFullUpdate>(true);
    m_UpdateOption.Set<UpdateOption::Reconnect>(false);

    // 標準のシステムデータを使う
    m_pActiveControllerFirmwareAccessor = &m_ControllerFirmwareAccessorNormal;

    return StartFirmwareUpdateImpl(handle);
}

Result XcdFirmwareUpdater::StartFirmwareUpdateForRevert(
    system::UniquePadId id) NN_NOEXCEPT
{
    if (IsUpdateRunning())
    {
        NN_RESULT_THROW(system::ResultFirmwareUpdateBusy());
    }

    auto handle = GetXcdDeviceHandle(id);
    NN_RESULT_DO(CheckInterfaceType(handle));

    // 巻き戻し版 (ツール用) は、バージョン制限なし、IAP 更新あり、完了後切断
    m_UpdateOption.Reset();
    m_UpdateOption.Set<UpdateOption::VersionRestriction>(false);
    m_UpdateOption.Set<UpdateOption::PermitMcuFullUpdate>(true);
    m_UpdateOption.Set<UpdateOption::Reconnect>(false);

    // デバッグ用のシステムデータを使う
    m_pActiveControllerFirmwareAccessor = &m_ControllerFirmwareAccessorDebug;

    return StartFirmwareUpdateImpl(handle);
}

Result XcdFirmwareUpdater::StartFirmwareUpdateByTransferMemory(
    system::FirmwareUpdateDeviceHandle* pOutDeviceHandle,
    system::UniquePadId id,
    FirmwareUpdateTarget target,
    const XcdFirmwareImageInfo& imageInfo) NN_NOEXCEPT
{
    if (IsUpdateRunning())
    {
        NN_RESULT_THROW(system::ResultFirmwareUpdateBusy());
    }

    auto handle = GetXcdDeviceHandle(id);
    NN_RESULT_DO(CheckInterfaceType(handle));

    NN_RESULT_DO(m_FirmwareMemory.Map(imageInfo));

    bool needsRollback = true;
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            m_FirmwareMemory.Unmap();
        }
    };

    // TransferMemory 版は、バージョン制限なし、IAP 更新なし、完了後切断
    m_UpdateOption.Reset();
    m_UpdateOption.Set<UpdateOption::VersionRestriction>(false);
    m_UpdateOption.Set<UpdateOption::PermitMcuFullUpdate>(false);
    m_UpdateOption.Set<UpdateOption::Reconnect>(false);
    m_UpdateOption.Set<UpdateOption::UseTransferMemory>(true);

    NN_RESULT_DO(StartFirmwareUpdateImplByTransferMemory(handle, target));

    // デバイス毎に固有になるように、BD アドレスからハンドルを生成する
    *pOutDeviceHandle = GetFirmwareUpdateDeviceHandle(m_DeviceInfo.address);

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

Result XcdFirmwareUpdater::StartBluetoothUpdate() NN_NOEXCEPT
{
    if (m_UpdateOption.Test<UpdateOption::UseTransferMemory>())
    {
        // 呼ばれた時点で、イメージは既にマップ済み

        NN_DETAIL_HID_INFO("Starting bluetooth update by TransferMemory.\n");

        // アップデートを開始
        nn::xcd::FirmwareImage image = {};
        image.imageType      = nn::xcd::FirmwareImageType_Memory;
        image.pFirmwareImage = reinterpret_cast<char*>(m_FirmwareMemory.GetImageData());
        image.imageSize      = m_FirmwareMemory.GetImageSize();
        NN_RESULT_THROW_UNLESS(
            nn::xcd::StartBtFirmwareUpdate(m_Handle, image, m_pXcdBluetoothUpdateEvent).IsSuccess(),
            system::ResultUniquePadDisconnected()
        );
    }
    else
    {
        // ファームウェアイメージを開く
        char imagePath[FirmwareFilenameLengthMax];
        NN_RESULT_DO(
            m_pActiveControllerFirmwareAccessor->GetBluetoothImageFilePath(
                imagePath,
                sizeof(imagePath),
                m_Handle)
        );
        NN_RESULT_THROW_UNLESS(
            m_BluetoothFile.Open(imagePath).IsSuccess(),
            system::ResultFirmwareImageReadFailed()
        );

        bool needsRollback = true;
        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                m_BluetoothFile.Close();
            }
        };

        NN_DETAIL_HID_INFO("Starting bluetooth update.\n");

        // アップデートを開始
        nn::xcd::FirmwareImage image = {};
        image.imageType  = nn::xcd::FirmwareImageType_File;
        image.fileHandle = m_BluetoothFile.GetHandle();
        NN_RESULT_THROW_UNLESS(
            nn::xcd::StartBtFirmwareUpdate(m_Handle, image, m_pXcdBluetoothUpdateEvent).IsSuccess(),
            system::ResultUniquePadDisconnected()
        );

        needsRollback = false;
    }

    m_FirmwareUpdateState = FirmwareUpdateState_BtUpdate;

    NN_RESULT_SUCCESS;
}

Result XcdFirmwareUpdater::AbortFirmwareUpdate() NN_NOEXCEPT
{
    if (IsUpdateRunning())
    {
        // 実行中の FW 更新を中断
        nn::xcd::AbortMcuUpdate(m_Handle);
        nn::xcd::AbortBtFirmwareUpdate(m_Handle);

        // 切断扱いにする
        FailFirmwareUpdateAsDisconnected();
    }

    NN_RESULT_SUCCESS;
}

void XcdFirmwareUpdater::FinishFirmwareUpdateProcess() NN_NOEXCEPT
{
    nn::xcd::UnsetFirmwareUpdatingDevice();
    UnmountSystemData();
}

void XcdFirmwareUpdater::FailFirmwareUpdateAsDisconnected() NN_NOEXCEPT
{
    // 再接続待ち状態も解除する
    m_IsReconnectRequired   = false;
    m_ReconnectionStartTick = ::nn::os::Tick();
    m_FirmwareUpdateState   = FirmwareUpdateState_Disconnected;
    ::nn::os::StopTimerEvent(m_pResetTimeoutEvent);

    FinishFirmwareUpdateProcess();
}

Result XcdFirmwareUpdater::GetFirmwareVersion(
    system::FirmwareVersion* pOutValue,
    system::UniquePadId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    bool isIapCorrupted;
    auto xcdHandle = GetXcdDeviceHandle(id);
    NN_RESULT_DO(
        GetFirmwareVersionImpl(pOutValue, &isIapCorrupted, xcdHandle));
    NN_UNUSED(isIapCorrupted);

    NN_RESULT_SUCCESS;
}

Result XcdFirmwareUpdater::GetFirmwareVersionImpl(
    debug::FirmwareVersion* pOutValue,
    bool* pOutIsMcuIapCorrupted,
    ::nn::xcd::DeviceHandle xcdHandle) NN_NOEXCEPT
{
    std::memset(pOutValue, 0, sizeof(debug::FirmwareVersion));

    // Bluetooth FW バージョン
    ::nn::xcd::BtFirmwareVersion btFirmwareVersion;
    NN_RESULT_THROW_UNLESS(
        ::nn::xcd::GetBtFirmwareVersion(&btFirmwareVersion, xcdHandle).IsSuccess(),
        system::ResultUniquePadDisconnected()
    );
    pOutValue->major = btFirmwareVersion.major;
    pOutValue->minor = btFirmwareVersion.minor;

    ::nn::xcd::DeviceInfo deviceInfo;
    NN_RESULT_THROW_UNLESS(
        ::nn::xcd::GetDeviceInfo(&deviceInfo, xcdHandle).IsSuccess(),
        system::ResultUniquePadDisconnected()
    );

    // MCU FW バージョン
    if (HasMcuDevice(deviceInfo))
    {
        ::nn::xcd::McuVersionData mcuVersion;
        if (::nn::xcd::GetMcuVersion(&mcuVersion, xcdHandle).IsFailure())
        {
            ::nn::xcd::RequestMcuVersion(xcdHandle);
            NN_RESULT_THROW(system::ResultFirmwareVersionReading());
        }

        pOutValue->micro       = static_cast<uint8_t>(mcuVersion.major);
        pOutValue->revision    = static_cast<uint8_t>(mcuVersion.minor);
        *pOutIsMcuIapCorrupted = mcuVersion.isIapCorrupted;
    }
    else
    {
        pOutValue->micro       = 0;
        pOutValue->revision    = 0;
        *pOutIsMcuIapCorrupted = false;
    }

    // デバイス識別子
    GetDeviceIdentifier(pOutValue, deviceInfo);

    NN_RESULT_SUCCESS;
}

Result XcdFirmwareUpdater::GetDestinationFirmwareVersion(
    system::FirmwareVersion* pOutVersion,
    system::UniquePadId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutVersion);

    auto xcdHandle = GetXcdDeviceHandle(id);
    return GetDestinationFirmwareVersionImpl(
        pOutVersion,
        xcdHandle,
        &m_ControllerFirmwareAccessorNormal);
}

Result XcdFirmwareUpdater::GetDestinationFirmwareVersionForRevert(
    debug::FirmwareVersion* pOutVersion,
    system::UniquePadId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutVersion);

    auto xcdHandle = GetXcdDeviceHandle(id);
    return GetDestinationFirmwareVersionImpl(
        pOutVersion,
        xcdHandle,
        &m_ControllerFirmwareAccessorDebug);
}

Result XcdFirmwareUpdater::GetDestinationFirmwareVersionImpl(
    system::FirmwareVersion* pOutVersion,
    nn::xcd::DeviceHandle handle,
    ControllerFirmwareAccessor* pAccessor) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutVersion);
    NN_SDK_REQUIRES_NOT_NULL(pAccessor);

    NN_RESULT_DO(pAccessor->GetDestinationVersion(pOutVersion, handle));

    ::nn::xcd::DeviceInfo deviceInfo;
    NN_RESULT_THROW_UNLESS(
        ::nn::xcd::GetDeviceInfo(&deviceInfo, handle).IsSuccess(),
        system::ResultUniquePadDisconnected()
    );

    GetDeviceIdentifier(pOutVersion, deviceInfo);

    NN_RESULT_SUCCESS;
}

Result XcdFirmwareUpdater::IsFirmwareUpdateAvailable(
    bool* pOutIsAvailable,
    system::UniquePadId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutIsAvailable);

    auto xcdHandle = GetXcdDeviceHandle(id);

    // デバイスが接続されているか
    NN_RESULT_THROW_UNLESS(
        xcdHandle.IsValid(),
        system::ResultUniquePadDisconnected()
    );

    {
        // デバイスが接続されているのに Disconnected の場合、非サポートのインターフェースなので更新なしとして扱う
        auto result = CheckInterfaceType(xcdHandle);
        if (system::ResultUniquePadDisconnected::Includes(result))
        {
            *pOutIsAvailable = false;
            NN_RESULT_SUCCESS;
        }
        else if (result.IsFailure())
        {
            NN_RESULT_THROW(result);
        }
    }

    // BT or MCU のどちらかが要更新であれば更新が必要と判定

    // MCU FW の更新要否判定
    // HardwareError を優先するため、BT FW より先に MCU FW を判定
    bool needsMcuUpdate;
    bool isIapCorrupted;
    NN_RESULT_DO(
        NeedsMcuUpdateImpl(
            &needsMcuUpdate,
            &isIapCorrupted,
            xcdHandle,
            &m_ControllerFirmwareAccessorNormal)
    );
    NN_RESULT_THROW_UNLESS(
        !isIapCorrupted,
        system::ResultFirmwareUpdateHardwareError()
    );

    if (needsMcuUpdate)
    {
        *pOutIsAvailable = true;
        NN_RESULT_SUCCESS;
    }

    // BT FW の更新要否判定
    bool needsBtUpdate;
    NN_RESULT_DO(
        NeedsBluetoothUpdateImpl(
            &needsBtUpdate,
            xcdHandle,
            &m_ControllerFirmwareAccessorNormal)
    );
    if (needsBtUpdate)
    {
        *pOutIsAvailable = true;
        NN_RESULT_SUCCESS;
    }

    // BT, MCU ともに最新
    *pOutIsAvailable = false;

    NN_RESULT_SUCCESS;
}

Result XcdFirmwareUpdater::CheckFirmwareUpdateRequired(
    system::FirmwareUpdateRequiredReason* pOutReason,
    system::UniquePadId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutReason);

    auto xcdHandle = GetXcdDeviceHandle(id);

    // デバイスが接続されているか
    NN_RESULT_THROW_UNLESS(
        xcdHandle.IsValid(),
        system::ResultUniquePadDisconnected()
    );

    {
        // デバイスが接続されているのに Disconnected の場合、非サポートのインターフェースなので更新なしとして扱う
        auto result = CheckInterfaceType(xcdHandle);
        if (system::ResultUniquePadDisconnected::Includes(result))
        {
            *pOutReason = system::FirmwareUpdateRequiredReason_Nothing;
            NN_RESULT_SUCCESS;
        }
        else if (result.IsFailure())
        {
            NN_RESULT_THROW(result);
        }
    }

    // MCU FW の更新要求判定
    // HardwareError を優先するため、BT FW より先に判定
    {
        system::FirmwareUpdateRequiredReason reason;
        NN_RESULT_DO(CheckMcuFirmwareUpdateRequired(&reason, xcdHandle));
        if (reason != system::FirmwareUpdateRequiredReason_Nothing)
        {
            *pOutReason = reason;
            NN_RESULT_SUCCESS;
        }
    }

    // BT FW の更新要求判定
    {
        system::FirmwareUpdateRequiredReason reason;
        NN_RESULT_DO(CheckBluetoothFirmwareUpdateRequired(&reason, xcdHandle));
        if (reason != system::FirmwareUpdateRequiredReason_Nothing)
        {
            *pOutReason = reason;
            NN_RESULT_SUCCESS;
        }
    }

    *pOutReason = system::FirmwareUpdateRequiredReason_Nothing;
    NN_RESULT_SUCCESS;
}

Result XcdFirmwareUpdater::CheckBluetoothFirmwareUpdateRequired(
    system::FirmwareUpdateRequiredReason* pOutReason,
    nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutReason);

    nn::xcd::BtFirmwareVersion btVersion;
    NN_RESULT_THROW_UNLESS(
        nn::xcd::GetBtFirmwareVersion(&btVersion, handle).IsSuccess(),
        system::ResultUniquePadDisconnected()
    );

    // デバッグ API 使用時は更新要否判定がないので、誤動作防止のために常に標準のデータを参照する
    system::FirmwareVersion version;
    NN_RESULT_DO(m_ControllerFirmwareAccessorNormal.GetExpectVersion(&version, handle));

    int currentVersion = (btVersion.major << 8) + btVersion.minor;
    int expectVersion  = (version.major << 8) + version.minor;

    // m_IsFirmwareHotfixUpdateSkipFlagEnabled がTrueの時はexpectVersion未満だとしても、Hotfix扱いにしない
    if (currentVersion < expectVersion && m_IsFirmwareHotfixUpdateSkipFlagEnabled == false)
    {
        // 更新すべき BT FW がある
        *pOutReason = system::FirmwareUpdateRequiredReason_Hotfix;
        NN_RESULT_SUCCESS;
    }

    *pOutReason = system::FirmwareUpdateRequiredReason_Nothing;
    NN_RESULT_SUCCESS;
}

Result XcdFirmwareUpdater::CheckMcuFirmwareUpdateRequired(
    system::FirmwareUpdateRequiredReason* pOutReason,
    nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutReason);

    ::nn::xcd::DeviceInfo deviceInfo;
    NN_RESULT_THROW_UNLESS(
        ::nn::xcd::GetDeviceInfo(&deviceInfo, handle).IsSuccess(),
        system::ResultUniquePadDisconnected()
    );

    if (!HasMcuDevice(deviceInfo))
    {
        // MCU 非搭載なら MCU 更新判定をしない
        *pOutReason = system::FirmwareUpdateRequiredReason_Nothing;
        NN_RESULT_SUCCESS;
    }

    // MCU バージョン確認
    nn::xcd::McuVersionData mcuVersion;
    NN_RESULT_TRY(nn::xcd::GetMcuVersion(&mcuVersion, handle))
        NN_RESULT_CATCH(nn::xcd::ResultMcuVersionNotAvailable)
        {
            ::nn::xcd::RequestMcuVersion(handle);
            NN_RESULT_THROW(system::ResultFirmwareVersionReading());
        }
        NN_RESULT_CATCH_ALL
        {
            NN_RESULT_THROW(system::ResultUniquePadDisconnected());
        }
    NN_RESULT_END_TRY;

    NN_RESULT_THROW_UNLESS(
        !mcuVersion.isIapCorrupted,
        system::ResultFirmwareUpdateHardwareError()
    );

    if (mcuVersion.isCorrupted)
    {
        // FW 壊れなので要更新
        *pOutReason = system::FirmwareUpdateRequiredReason_Corrupted;
        NN_RESULT_SUCCESS;
    }

    // デバッグ API 使用時は更新要否判定がないので、誤動作防止のために常に標準のデータを参照する
    system::FirmwareVersion version;
    NN_RESULT_DO(m_ControllerFirmwareAccessorNormal.GetExpectVersion(&version, handle));

    int currentVersion = (mcuVersion.major << 8) + mcuVersion.minor;
    int expectVersion  = (version.micro << 8) + version.revision;  // MCU バージョンは micro, revision
    if (currentVersion < expectVersion && m_IsFirmwareHotfixUpdateSkipFlagEnabled == false)
    {
        // 更新すべき MCU FW がある
        *pOutReason = system::FirmwareUpdateRequiredReason_Hotfix;
        NN_RESULT_SUCCESS;
    }

    *pOutReason = system::FirmwareUpdateRequiredReason_Nothing;
    NN_RESULT_SUCCESS;
}

Result XcdFirmwareUpdater::GetFirmwareUpdateState(
    system::FirmwareUpdateState* pOutState,
    system::FirmwareUpdateDeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutState);

    // (仮) FirmwareUpdateDeviceHandle の実体は xcd ハンドル
    auto bdAddr = GetBdAddressFromHandle(handle);
    if (m_DeviceInfo.address != bdAddr)
    {
        // 実行中のデバイスと異なっていたら切断扱い
        NN_RESULT_THROW(system::ResultUniquePadDisconnected());
    }

    debug::FirmwareUpdateStage stage;
    uint8_t progress;
    NN_RESULT_DO(GetFirmwareUpdateStage(&stage, &progress));

    pOutState->stage = ConvertUpdateStageToSystem(stage);

    // Stage に応じた進捗率の取得
    bool isSingle = m_FirmwareUpdateTarget != FirmwareUpdateTarget::BluetoothAndMcu;
    switch (stage)
    {
    case debug::FirmwareUpdateStage_Preparing:
        pOutState->progress = 0;
        break;

    case debug::FirmwareUpdateStage_BluetoothDownload:
        if (isSingle)
        {
            // BT のみの場合は 0-49 の間
            pOutState->progress = static_cast<uint8_t>(progress * 49 / 100);
        }
        else
        {
            // BT + MCU の場合は 0-4 の間
            pOutState->progress = static_cast<uint8_t>(progress * 4 / 100);
        }
        break;

    case debug::FirmwareUpdateStage_BluetoothVerify:
        if (isSingle)
        {
            // BT のみの場合は 50-99 の間
            pOutState->progress = 50 + static_cast<uint8_t>(progress * 49 / 100);
        }
        else
        {
            // BT + MCU の場合は 5-9 の間
            pOutState->progress = 5 + static_cast<uint8_t>(progress * 4 / 100);
        }
        break;

    case debug::FirmwareUpdateStage_BluetoothCommit:
    case debug::FirmwareUpdateStage_Reboot1:
        if (isSingle)
        {
            // BT のみの場合は 99
            pOutState->progress = 99;
        }
        else
        {
            // BT + MCU の場合は 9
            pOutState->progress = 9;
        }
        break;

    case debug::FirmwareUpdateStage_McuVersionCheck:
        if (isSingle)
        {
            // MCU のみの場合は 0
            pOutState->progress = 0;
        }
        else
        {
            // BT + MCU の場合は 10
            pOutState->progress = 10;
        }
        break;

    case debug::FirmwareUpdateStage_McuBoot:
    case debug::FirmwareUpdateStage_McuErasing:
    case debug::FirmwareUpdateStage_McuWriting:
    case debug::FirmwareUpdateStage_McuReboot:
    case debug::FirmwareUpdateStage_McuVerify:
        if (isSingle)
        {
            // MCU のみの場合は 0-99 の間
            pOutState->progress = static_cast<uint8_t>(progress * 99 / 100);
        }
        else
        {
            // BT + MCU の場合は 10-99 の間
            pOutState->progress = 10 + static_cast<uint8_t>(progress * 89 / 100);
        }
        break;

    case debug::FirmwareUpdateStage_Reboot2:
        pOutState->progress = 99;
        break;

    case debug::FirmwareUpdateStage_Completed:
        pOutState->progress = 100;
        NN_RESULT_SUCCESS;

    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;

}  // NOLINT(impl/function_size)

Result XcdFirmwareUpdater::GetFirmwareUpdateStage(debug::FirmwareUpdateStage* pOutStage, uint8_t* pOutProgress) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutStage);
    NN_SDK_REQUIRES_NOT_NULL(pOutProgress);

    nn::xcd::FirmwareUpdateStage stage;
    nn::xcd::McuUpdateStateInfo updateInfo;
    int progress;
    Result result;

    switch (m_FirmwareUpdateState)
    {
    case FirmwareUpdateState_NotStarted:
        NN_RESULT_THROW(system::ResultFirmwareUpdateNotStarted());
    case FirmwareUpdateState_BtUpdate:
        result = nn::xcd::GetBtFirmwareUpdateProgress(m_Handle, &stage, &progress);
        if (result.IsSuccess())
        {
            switch (stage)
            {
            case nn::xcd::FirmwareUpdateStage_Download:
                *pOutStage = debug::FirmwareUpdateStage_BluetoothDownload;
                break;
            case nn::xcd::FirmwareUpdateStage_Verify:
                *pOutStage = debug::FirmwareUpdateStage_BluetoothVerify;
                break;
            case nn::xcd::FirmwareUpdateStage_Commit:
            case nn::xcd::FirmwareUpdateStage_Completed:
                *pOutStage = debug::FirmwareUpdateStage_BluetoothCommit;
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
            *pOutProgress = static_cast<uint8_t>(progress);
        }
        else
        {
            if (::nn::xcd::ResultBtFirmwareUpdateVerifyError().Includes(result))
            {
                NN_RESULT_THROW(ResultFirmwareUpdateBluetoothVerifyError());
            }
            else if (::nn::xcd::ResultBtFirmwareUpdateNotStarted().Includes(result))
            {
                // BtUpdate ステートでアップデート開始前の状態は Preparing
                *pOutStage = debug::FirmwareUpdateStage_Preparing;
            }
            else
            {
                NN_RESULT_THROW(ResultFirmwareUpdateBluetoothUnknownError());
            }
        }
        break;
    case FirmwareUpdateState_Reboot1_WaitingDisconnect:
    case FirmwareUpdateState_Reboot1_WaitingReconnect:
        *pOutStage = debug::FirmwareUpdateStage_Reboot1;
        *pOutProgress = 50;
        break;

    case FirmwareUpdateState_Reboot2:
        *pOutStage = debug::FirmwareUpdateStage_Reboot2;
        *pOutProgress = 50;
        break;
    case FirmwareUpdateState_McuVersionCheck:
        *pOutStage = debug::FirmwareUpdateStage_McuVersionCheck;
        *pOutProgress = 50;
        break;
    case FirmwareUpdateState_McuUpdate:
    case FirmwareUpdateState_McuVerify:
        if (nn::xcd::GetMcuUpdateState(&updateInfo, m_Handle).IsFailure())
        {
            NN_RESULT_THROW(system::ResultUniquePadDisconnected());
        }

        if (m_FirmwareUpdateState == FirmwareUpdateState_McuVerify)
        {
            *pOutStage = debug::FirmwareUpdateStage_McuVerify;
        }
        else
        {
            switch (updateInfo.state)
            {
            case nn::xcd::McuUpdateState_Boot:
                *pOutStage = debug::FirmwareUpdateStage_McuBoot;
                break;
            case nn::xcd::McuUpdateState_RomErasing:
                *pOutStage = debug::FirmwareUpdateStage_McuErasing;
                break;
            case nn::xcd::McuUpdateState_Writing:
                *pOutStage = debug::FirmwareUpdateStage_McuWriting;
                break;
            case nn::xcd::McuUpdateState_Reboot:
            case nn::xcd::McuUpdateState_End:
                *pOutStage = debug::FirmwareUpdateStage_McuReboot;
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
        *pOutProgress = static_cast<uint8_t>(updateInfo.progress);
        break;
    case FirmwareUpdateState_Complete:
        *pOutStage = debug::FirmwareUpdateStage_Completed;
        *pOutProgress = 100;
        break;
    case FirmwareUpdateState_BtVerifyError:
        NN_RESULT_THROW(ResultFirmwareUpdateBluetoothVerifyError());
        break;
    case FirmwareUpdateState_BtUnknownError:
        NN_RESULT_THROW(ResultFirmwareUpdateBluetoothUnknownError());
        break;
    case FirmwareUpdateState_McuVerifyError:
        NN_RESULT_THROW(ResultFirmwareUpdateMcuVerifyError());
        break;
    case FirmwareUpdateState_McuHardwareError:
        NN_RESULT_THROW(system::ResultFirmwareUpdateHardwareError());
        break;
    case FirmwareUpdateState_McuUnknownError:
        NN_RESULT_THROW(system::ResultFirmwareUpdateFailed());
        break;
    case FirmwareUpdateState_Disconnected:
        NN_RESULT_THROW(system::ResultUniquePadDisconnected());
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;

}  // NOLINT(impl/function_size)

Result XcdFirmwareUpdater::IsFirmwareUpdatingDevice(
    bool* pOutIsUpdating,
    system::UniquePadId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutIsUpdating);

    if (IsUpdateRunning())
    {
        auto xcdHandle = GetXcdDeviceHandle(id);
        *pOutIsUpdating = xcdHandle == m_Handle;
    }
    else
    {
        *pOutIsUpdating = false;
    }

    NN_RESULT_SUCCESS;
}

void XcdFirmwareUpdater::BluetoothFirmwareUpdateCompleted() NN_NOEXCEPT
{
    m_BluetoothFile.Close();
    m_FirmwareMemory.Unmap();

    nn::xcd::FirmwareUpdateStage stage;
    int progress;
    auto result = nn::xcd::GetBtFirmwareUpdateProgress(m_Handle, &stage, &progress);
    if (result.IsSuccess())
    {
        NN_DETAIL_HID_INFO("Bluetooth firmware update is completed.\n");
        if (m_FirmwareUpdateTarget == FirmwareUpdateTarget::BluetoothAndMcu &&
            HasMcuDevice(m_DeviceInfo))
        {
            // MCU 更新があるので、再接続必須
            m_FirmwareUpdateState = FirmwareUpdateState_Reboot1_WaitingDisconnect;
            RebootUpdatingDevice(true);
        }
        else
        {
            // 再接続が必要であれば再接続待ち、再接続不要なら更新完了
            m_FirmwareUpdateState =
                m_UpdateOption.Test<UpdateOption::Reconnect>() ?
                FirmwareUpdateState_Reboot1_WaitingDisconnect :
                FirmwareUpdateState_Reboot2;
            RebootUpdatingDevice(m_UpdateOption.Test<UpdateOption::Reconnect>());
        }
    }
    else
    {
        // 失敗時は再接続待ちシーケンスに入らないので、そのまま切断
        if (nn::xcd::ResultBtFirmwareUpdateVerifyError().Includes(result))
        {
            m_FirmwareUpdateState = FirmwareUpdateState_BtVerifyError;
            RebootUpdatingDevice(false);
        }
        else if (nn::xcd::ResultBtFirmwareUpdateUnknownError().Includes(result))
        {
            m_FirmwareUpdateState = FirmwareUpdateState_BtUnknownError;
            RebootUpdatingDevice(false);
        }
        else
        {
            m_FirmwareUpdateState = FirmwareUpdateState_Disconnected;
        }

        FinishFirmwareUpdateProcess();
    }
}

void XcdFirmwareUpdater::McuFirmwareUpdateCompleted() NN_NOEXCEPT
{
    // MCU の Verify はイメージ不要なので閉じる
    m_McuFile.Close();
    m_FirmwareMemory.Unmap();

    m_FirmwareUpdateState = FirmwareUpdateState_McuVerify;
}

void XcdFirmwareUpdater::SampleRecieved() NN_NOEXCEPT
{
    if (m_FirmwareUpdateState == FirmwareUpdateState_McuVersionCheck)
    {
        ProcessMcuVersionCheck();
    }
    else if (m_FirmwareUpdateState == FirmwareUpdateState_McuVerify)
    {
        ProcessMcuVerify();
    }
}

void XcdFirmwareUpdater::DeviceUpdated() NN_NOEXCEPT
{
    nn::xcd::DeviceList deviceList;
    if (nn::xcd::ListDevices(&deviceList).IsFailure())
    {
        deviceList.deviceCount = 0;
    }

    int index;
    auto isConnected = IsDeviceConnected(deviceList, &index);

    switch (m_FirmwareUpdateState)
    {
    case FirmwareUpdateState_NotStarted:
    case FirmwareUpdateState_Disconnected:
        break;

    case FirmwareUpdateState_BtUpdate:
    case FirmwareUpdateState_McuUpdate:
    case FirmwareUpdateState_McuVersionCheck:
    case FirmwareUpdateState_McuVerify:
    case FirmwareUpdateState_BtVerifyError:
    case FirmwareUpdateState_BtUnknownError:
    case FirmwareUpdateState_McuVerifyError:
    case FirmwareUpdateState_McuHardwareError:
    case FirmwareUpdateState_McuUnknownError:
        if (isConnected == false)
        {
            m_FirmwareUpdateState = FirmwareUpdateState_Disconnected;
            FinishFirmwareUpdateProcess();
        }
        break;

    case FirmwareUpdateState_Reboot1_WaitingDisconnect:
        if (!isConnected)
        {
            NN_DETAIL_HID_INFO("Device is disconnected on Reboot1_WaitingDisconnect.\n");
            if (m_IsReconnectRequired)
            {
                // 切断が完了したら再接続トリガー
                TriggerReconnect(true);
            }

            m_FirmwareUpdateState = FirmwareUpdateState_Reboot1_WaitingReconnect;
        }
        break;

    case FirmwareUpdateState_Reboot1_WaitingReconnect:
        if (isConnected)
        {
            // ハンドルを更新
            m_Handle = deviceList.handleList[index];

            nn::xcd::DeviceStatus status;
            auto result = nn::xcd::GetDeviceStatus(&status, m_Handle);
            if (result.IsFailure() ||
                status.interfaceType != m_InterfaceType)
            {
                // 接続インターフェースが変化していたら切断と見なす
                NN_DETAIL_HID_INFO("Device interface is changed. Disconnected.\n");
                m_FirmwareUpdateState = FirmwareUpdateState_Disconnected;
                FinishFirmwareUpdateProcess();
            }
            else if (HasMcuDevice(m_DeviceInfo))
            {
                if (PrepareMcuUpdate().IsFailure())
                {
                    NN_DETAIL_HID_WARN("Failed to prepare for updating MCU.\n");
                    m_FirmwareUpdateState = FirmwareUpdateState_McuUnknownError;
                    FinishFirmwareUpdateProcess();
                }
            }
            else
            {
                // MCU がない場合は更新完了
                NN_DETAIL_HID_INFO("No MCU. Update is completed.\n");
                m_FirmwareUpdateState = FirmwareUpdateState_Complete;
                FinishFirmwareUpdateProcess();
            }

            m_IsReconnectRequired   = false;
            m_ReconnectionStartTick = ::nn::os::Tick();
            ::nn::os::StopTimerEvent(m_pResetTimeoutEvent);

            // 念のためジョイコン取り外し通知の抑止を解除する
            m_pInterruptSceneNotifier->SetIgnoreJoyDetachOnBluetoothOffEvent(false);
        }
        break;

    case FirmwareUpdateState_Reboot2:
        if (isConnected)
        {
            m_IsReconnectRequired   = false;
            m_ReconnectionStartTick = ::nn::os::Tick();
            ::nn::os::StopTimerEvent(m_pResetTimeoutEvent);
        }
        else
        {
            m_FirmwareUpdateState = FirmwareUpdateState_Complete;
            FinishFirmwareUpdateProcess();

            // 再接続が必要な場合は再接続トリガー
            if (m_IsReconnectRequired)
            {
                TriggerReconnect(true);
            }
        }
        break;
    case FirmwareUpdateState_Complete:
        if (!isConnected)
        {
            // 更新対象のデバイスが切断したら完了状態から抜ける
            m_FirmwareUpdateState = FirmwareUpdateState_NotStarted;
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

}  // NOLINT(impl/function_size)

void XcdFirmwareUpdater::ConnectionTriggerTimedOut() NN_NOEXCEPT
{
    if (IsRebooting() && m_IsReconnectRequired)
    {
        auto diffTime = (::nn::os::GetSystemTick() - m_ReconnectionStartTick).ToTimeSpan();
        if (diffTime > ReconnectionTimeout)
        {
            // 再接続開始から規定時間が経過していたら切断扱いで失敗
            NN_DETAIL_HID_WARN("Failed to reconnect.\n");
            FailFirmwareUpdateAsDisconnected();
        }
        else
        {
            // 接続トリガーを再発行
            NN_DETAIL_HID_INFO("ConnectionTrigger is timed out. Retrying.\n");
            TriggerReconnect(false);
        }
    }
}

void XcdFirmwareUpdater::ResetTimedOut() NN_NOEXCEPT
{
    if (IsRebooting())
    {
        // レール接続中の再アタッチがタイムアウトした場合は切断扱いで失敗
        NN_DETAIL_HID_WARN("Failed to attach after reset.\n");
        FailFirmwareUpdateAsDisconnected();
    }
}

bool XcdFirmwareUpdater::IsDeviceConnected(nn::xcd::DeviceList& deviceList, int* pIndex) NN_NOEXCEPT
{
    for (int i = 0; i < deviceList.deviceCount; i++)
    {
        nn::xcd::DeviceInfo deviceInfo;
        nn::xcd::GetDeviceInfo(&deviceInfo, deviceList.handleList[i]);

        if (std::memcmp(&deviceInfo, &m_DeviceInfo, sizeof(nn::xcd::DeviceInfo)) == 0)
        {
            *pIndex = i;
            return true;
        }
    }

    return false;
}

bool XcdFirmwareUpdater::IsUpdateRunning() NN_NOEXCEPT
{
    return
        m_FirmwareUpdateState != FirmwareUpdateState_NotStarted &&
        m_FirmwareUpdateState != FirmwareUpdateState_Complete &&
        m_FirmwareUpdateState != FirmwareUpdateState_BtVerifyError &&
        m_FirmwareUpdateState != FirmwareUpdateState_BtUnknownError &&
        m_FirmwareUpdateState != FirmwareUpdateState_McuVerifyError &&
        m_FirmwareUpdateState != FirmwareUpdateState_McuHardwareError &&
        m_FirmwareUpdateState != FirmwareUpdateState_McuUnknownError &&
        m_FirmwareUpdateState != FirmwareUpdateState_Disconnected;
}

bool XcdFirmwareUpdater::IsRebooting() NN_NOEXCEPT
{
    return
        m_FirmwareUpdateState == FirmwareUpdateState_Reboot1_WaitingDisconnect ||
        m_FirmwareUpdateState == FirmwareUpdateState_Reboot1_WaitingReconnect ||
        m_FirmwareUpdateState == FirmwareUpdateState_Reboot2;
}

void XcdFirmwareUpdater::UnmountSystemData() NN_NOEXCEPT
{
    m_BluetoothFile.Close();
    m_McuFile.Close();
    m_pActiveControllerFirmwareAccessor->Unmount();
    m_FirmwareMemory.Unmap();
}

Result XcdFirmwareUpdater::NeedsBluetoothUpdate(bool* pOutNeedsUpdate) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutNeedsUpdate);

    // Firmware 更新が可能かどうかインタフェースをチェックする
    NN_RESULT_DO(CheckInterfaceType(m_Handle));

    if (!m_UpdateOption.Test<UpdateOption::VersionRestriction>())
    {
        // バージョン判定しない場合は常に更新
        *pOutNeedsUpdate = true;
        NN_RESULT_SUCCESS;
    }

    return NeedsBluetoothUpdateImpl(
        pOutNeedsUpdate,
        m_Handle,
        m_pActiveControllerFirmwareAccessor);
}

Result XcdFirmwareUpdater::NeedsBluetoothUpdateImpl(
    bool* pOutNeedsUpdate,
    ::nn::xcd::DeviceHandle xcdHandle,
    ControllerFirmwareAccessor* pAccessor) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutNeedsUpdate);
    NN_SDK_REQUIRES_NOT_NULL(pAccessor);

    system::FirmwareVersion destinationVersion;
    NN_RESULT_DO(
        pAccessor->GetDestinationVersion(&destinationVersion, xcdHandle)
    );

    nn::xcd::BtFirmwareVersion btVersion;
    NN_RESULT_THROW_UNLESS(
        nn::xcd::GetBtFirmwareVersion(&btVersion, xcdHandle).IsSuccess(),
        system::ResultUniquePadDisconnected()
    );

    *pOutNeedsUpdate =
        btVersion.major < destinationVersion.major ||
        (btVersion.major == destinationVersion.major &&
            btVersion.minor < destinationVersion.minor);

    NN_RESULT_SUCCESS;
}

Result XcdFirmwareUpdater::NeedsMcuUpdate(bool* pOutNeedsUpdate) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutNeedsUpdate);

    // Firmware 更新が可能かどうかインタフェースをチェックする
    NN_RESULT_DO(CheckInterfaceType(m_Handle));

    if (!m_UpdateOption.Test<UpdateOption::VersionRestriction>())
    {
        // バージョン判定しない場合、MCU が搭載されていれば常に更新
        *pOutNeedsUpdate = HasMcuDevice(m_DeviceInfo);
        NN_RESULT_SUCCESS;
    }

    bool needsIapUpdate;
    auto result = NeedsMcuUpdateImpl(
        pOutNeedsUpdate,
        &needsIapUpdate,
        m_Handle,
        m_pActiveControllerFirmwareAccessor);
    NN_UNUSED(needsIapUpdate);

    return result;
}

Result XcdFirmwareUpdater::NeedsMcuUpdateImpl(
    bool* pOutNeedsUpdate,
    bool* pOutNeedsIapUpdate,
    ::nn::xcd::DeviceHandle xcdHandle,
    ControllerFirmwareAccessor* pAccessor) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutNeedsUpdate);
    NN_SDK_REQUIRES_NOT_NULL(pOutNeedsIapUpdate);
    NN_SDK_REQUIRES_NOT_NULL(pAccessor);

    nn::xcd::DeviceInfo deviceInfo;
    NN_RESULT_THROW_UNLESS(
        nn::xcd::GetDeviceInfo(&deviceInfo, xcdHandle).IsSuccess(),
        system::ResultUniquePadDisconnected()
    );

    *pOutNeedsIapUpdate = false;

    // MCU 非搭載デバイスは更新不要
    if (!HasMcuDevice(deviceInfo))
    {
        *pOutNeedsUpdate = false;
        NN_RESULT_SUCCESS;
    }

    nn::xcd::McuVersionData mcuVersion;
    NN_RESULT_TRY(nn::xcd::GetMcuVersion(&mcuVersion, xcdHandle))
        NN_RESULT_CATCH(nn::xcd::ResultMcuVersionNotAvailable)
        {
            ::nn::xcd::RequestMcuVersion(xcdHandle);
            NN_RESULT_THROW(system::ResultFirmwareVersionReading());
        }
        NN_RESULT_CATCH_ALL
        {
            NN_RESULT_THROW(system::ResultUniquePadDisconnected());
        }
    NN_RESULT_END_TRY;

    if (mcuVersion.isCorrupted)
    {
        // FW 壊れなので更新必須
        *pOutNeedsUpdate    = true;
        *pOutNeedsIapUpdate = NeedsIapUpdate(deviceInfo.deviceType, mcuVersion);
        NN_RESULT_SUCCESS;
    }

    // TransferMemory 使用時は、FW 壊れでなければ更新完了と見なす
    if (m_UpdateOption.Test<UpdateOption::UseTransferMemory>())
    {
        NN_DETAIL_HID_INFO("Using TransferMemory, version check is skipped.\n");
        *pOutNeedsUpdate = false;
        NN_RESULT_SUCCESS;
    }

    system::FirmwareVersion destinationVersion;
    NN_RESULT_DO(
        pAccessor->GetDestinationVersion(&destinationVersion, xcdHandle)
    );

    // MCU の destination バージョンは micro と revision
    *pOutNeedsUpdate =
        mcuVersion.major < destinationVersion.micro ||
        (mcuVersion.major == destinationVersion.micro &&
            mcuVersion.minor < destinationVersion.revision);

    NN_RESULT_SUCCESS;
};

void XcdFirmwareUpdater::ProcessMcuVersionCheck() NN_NOEXCEPT
{
    bool needsUpdate;
    {
        auto result = NeedsMcuUpdate(&needsUpdate);
        if (system::ResultFirmwareVersionReading::Includes(result))
        {
            // バージョン取得中
            return;
        }
        else if (result.IsFailure())
        {
            // 想定外のエラー
            m_FirmwareUpdateState = FirmwareUpdateState_McuUnknownError;
            FinishFirmwareUpdateProcess();
            return;
        }
    }

    if (!needsUpdate)
    {
        // MCU ファームウェア更新が必要ない場合は完了
        m_FirmwareUpdateState = FirmwareUpdateState_Complete;
        FinishFirmwareUpdateProcess();
        return;
    }

    nn::xcd::McuVersionData version;
    if (nn::xcd::GetMcuVersion(&version, m_Handle).IsFailure())
    {
        return;
    }

    bool needsIapUpdate = NeedsIapUpdate(m_DeviceInfo.deviceType, version);
    if (needsIapUpdate &&
        !m_UpdateOption.Test<UpdateOption::PermitMcuFullUpdate>())
    {
        // IAP が壊れているが更新不能の場合は HW エラー扱い
        NN_DETAIL_HID_WARN("%s: IAP seems to be broken.\n", NN_CURRENT_FUNCTION_NAME);
        m_FirmwareUpdateState = FirmwareUpdateState_McuHardwareError;
        FinishFirmwareUpdateProcess();
        return;
    }

    // MCU 更新開始
    m_UpdateOption.Set<UpdateOption::NeedsMcuFullUpdate>(needsIapUpdate);
    if (StartMcuUpdate().IsFailure())
    {
        m_FirmwareUpdateState = FirmwareUpdateState_McuUnknownError;
        FinishFirmwareUpdateProcess();
    }
};

void XcdFirmwareUpdater::ProcessMcuVerify() NN_NOEXCEPT
{
    // FW 更新が必要なくなっていれば更新成功と見なす
    bool needsUpdate;
    bool needsIapUpdate;
    auto result = NeedsMcuUpdateImpl(
        &needsUpdate,
        &needsIapUpdate,
        m_Handle,
        m_pActiveControllerFirmwareAccessor);
    NN_UNUSED(needsIapUpdate);

    if (system::ResultFirmwareVersionReading::Includes(result))
    {
        // バージョン取得中
        return;
    }
    else if (result.IsFailure())
    {
        // デバイス切断
        NN_DETAIL_HID_WARN("MCU verify: Device disconnected.\n");
        m_FirmwareUpdateState = FirmwareUpdateState_Disconnected;
        return;
    }

    if (needsUpdate)
    {
        NN_DETAIL_HID_WARN("MCU firmware is corrupted.\n");
        m_FirmwareUpdateState = FirmwareUpdateState_McuVerifyError;
    }
    else
    {
        NN_DETAIL_HID_INFO("MCU firmware update is completed.\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(EndMcuUpdate());
    }
};

Result XcdFirmwareUpdater::PrepareMcuUpdate() NN_NOEXCEPT
{
    m_FirmwareUpdateState = FirmwareUpdateState_McuVersionCheck;

    nn::xcd::SetSamplingEvent(m_pSampleUpdateEvent, m_Handle);

    NN_RESULT_THROW_UNLESS(
        nn::xcd::RequestMcuVersion(m_Handle).IsSuccess(),
        system::ResultUniquePadDisconnected()
    );

    NN_RESULT_SUCCESS;
}

Result XcdFirmwareUpdater::StartMcuUpdate() NN_NOEXCEPT
{
    m_FirmwareUpdateState = FirmwareUpdateState_McuUpdate;

    auto checkUpdateResult = [](nn::Result result, UpdateOptionSet option) -> nn::Result
    {
        NN_UNUSED(option);  // Release ビルド時のエラー対策

        NN_RESULT_TRY(result)
            NN_RESULT_CATCH(::nn::xcd::ResultNotConnected)
            {
                NN_RESULT_THROW(system::ResultUniquePadDisconnected());
            }
            NN_RESULT_CATCH_ALL
            {
                NN_DETAIL_HID_ERROR(
                    "nn::xcd::StartMcuUpdate%s() returns 0x%08X",
                    option.Test<UpdateOption::NeedsMcuFullUpdate>() ? "Full" : "",
                    result.GetInnerValueForDebug());
                NN_RESULT_THROW(ResultFirmwareUpdateMcuUnknownError());
            }
        NN_RESULT_END_TRY;

        NN_RESULT_SUCCESS;
    };

    if (m_UpdateOption.Test<UpdateOption::UseTransferMemory>())
    {
        // 呼ばれた時点で、イメージは既にマップ済み
        NN_DETAIL_HID_INFO("Starting normal MCU update by TransferMemory.\n");

        nn::xcd::FirmwareImage image = {};
        image.imageType      = nn::xcd::FirmwareImageType_Memory;
        image.pFirmwareImage = reinterpret_cast<char*>(m_FirmwareMemory.GetImageData());
        image.imageSize      = m_FirmwareMemory.GetImageSize();

        auto result = nn::xcd::StartMcuUpdate(m_Handle, image, m_pXcdMcuUpdateEvent);
        NN_RESULT_DO(checkUpdateResult(result, m_UpdateOption));
    }
    else
    {
        // ファームウェアイメージを開く
        char imagePath[FirmwareFilenameLengthMax];
        NN_RESULT_THROW_UNLESS(
            m_pActiveControllerFirmwareAccessor->GetMcuImageFilePath(
                imagePath,
                sizeof(imagePath),
                m_Handle,
                m_UpdateOption.Test<UpdateOption::NeedsMcuFullUpdate>()).IsSuccess(),
            system::ResultFirmwareImageReadFailed()
        );
        NN_RESULT_THROW_UNLESS(
            m_McuFile.Open(imagePath).IsSuccess(),
            system::ResultFirmwareImageReadFailed()
        );

        bool needsRollback = true;
        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                m_McuFile.Close();
            }
        };

        // アップデート開始
        nn::xcd::FirmwareImage image = {};
        image.imageType  = nn::xcd::FirmwareImageType_File;
        image.fileHandle = m_McuFile.GetHandle();

        nn::Result result;
        if (m_UpdateOption.Test<UpdateOption::NeedsMcuFullUpdate>())
        {
            NN_DETAIL_HID_INFO("Starting full MCU update.\n");
            result = nn::xcd::StartMcuUpdateFull(m_Handle, image, m_pXcdMcuUpdateEvent);
        }
        else
        {
            NN_DETAIL_HID_INFO("Starting normal MCU update.\n");
            result = nn::xcd::StartMcuUpdate(m_Handle, image, m_pXcdMcuUpdateEvent);
        }

        NN_RESULT_DO(checkUpdateResult(result, m_UpdateOption));

        needsRollback = false;
    }

    NN_RESULT_SUCCESS;
}

Result XcdFirmwareUpdater::EndMcuUpdate() NN_NOEXCEPT
{
    if (m_UpdateOption.Test<UpdateOption::Reconnect>())
    {
        // 内部で MCU リセットをかけているため、接続維持する場合は再接続不要
        m_FirmwareUpdateState = FirmwareUpdateState_Complete;
        FinishFirmwareUpdateProcess();
    }
    else
    {
        m_FirmwareUpdateState = FirmwareUpdateState_Reboot2;
        RebootUpdatingDevice(false);
    }

    NN_RESULT_SUCCESS;
}

::nn::xcd::DeviceHandle XcdFirmwareUpdater::GetXcdDeviceHandle(system::UniquePadId id) NN_NOEXCEPT
{
    for (auto& pManager : m_pUniquePadManagers)
    {
        if (pManager->GetId()._storage == id._storage)
        {
            IAbstractedPad* pPad;
            if (pManager->GetIAbstractedPad(&pPad))
            {
                if (pPad->GetType() == AbstractedPadType_Xcd)
                {
                    return static_cast<AbstractedPadXcd*>(pPad)->GetXcdDeviceHandle();
                }
            }

        }
    }

    // 接続していない場合はダミーハンドルを返却
    return ::nn::xcd::DeviceHandle::GetInvalidHandle();
}

void XcdFirmwareUpdater::RebootUpdatingDevice(bool needsReconnect) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pResetTimeoutEvent);

    ::nn::xcd::DeviceStatus status;
    auto result = ::nn::xcd::GetDeviceStatus(&status, m_Handle);
    if (result.IsSuccess() &&
        status.interfaceType == ::nn::xcd::InterfaceType_Uart)
    {
        // ジョイント状態の場合は自動で再接続されるので、接続トリガー不要
        // ただしジョイコンの取り外し通知は抑止する
        m_IsReconnectRequired = false;
        m_pInterruptSceneNotifier->SetIgnoreJoyDetachOnBluetoothOffEvent(true);

        // 再接続待ちをする場合はタイムアウト判定を開始
        if (needsReconnect)
        {
            ::nn::os::StartOneShotTimerEvent(m_pResetTimeoutEvent, ReconnectionTimeout);
        }
    }
    else
    {
        // 切断が完了するまで起動トリガーをかけられないので、トリガー発行を遅延する
        m_IsReconnectRequired = needsReconnect;
    }

    ::nn::xcd::Reboot(m_Handle, false);
}

void XcdFirmwareUpdater::TriggerReconnect(bool isFirstTrigger) NN_NOEXCEPT
{
    // 再接続前の切断時にしか来ないはずなので、Result はハンドリングしない
    ::nn::xcd::SendConnectionTrigger(m_DeviceInfo.address);

    if (isFirstTrigger)
    {
        m_ReconnectionStartTick = ::nn::os::GetSystemTick();

        // LLR が不発した場合に備えたタイムアウト
        ::nn::os::StartOneShotTimerEvent(m_pResetTimeoutEvent, ConnectionTriggerFailureTimeout);
    }
}

}}} // namespace nn::hid::detail
