﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_TransferMemory.h>

namespace nn { namespace hid { namespace detail {

struct XcdFirmwareImageInfo
{
    nn::os::NativeHandle    handle;
    bool                    isHandleManaged;
    size_t                  size;
};

/**
 * @brief   ファームウェアイメージを TransferMemory にマップする操作を提供するクラスです。
 */
class XcdFirmwareMemory final
{
public:
    XcdFirmwareMemory() NN_NOEXCEPT;

    ~XcdFirmwareMemory() NN_NOEXCEPT;

    /**
     * @brief   TransferMemory で渡されたファームウェアイメージを、アクセス可能なメモリにマップします。
     */
    nn::Result Map(const XcdFirmwareImageInfo& info) NN_NOEXCEPT;

    /**
     * @brief   ファームウェアイメージのマップを解除します。
     */
    void Unmap() NN_NOEXCEPT;

    /**
     * @brief  ファームウェアのサイズを取得します。
     */
    size_t GetImageSize() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsMapped);
        return m_ImageInfo.size;
    }

    /**
     * @brief   マップされたファームウェアのアドレスを取得します。
     */
    void* GetImageData() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsMapped);
        return m_pMappedData;
    }

private:
    void AttachImage(
        nn::os::TransferMemoryType* pImageMemory,
        const XcdFirmwareImageInfo& info) NN_NOEXCEPT;

    nn::Result MapImage(void** ppOutImage) NN_NOEXCEPT;

    nn::Result MapTransferMemory(
        void** ppOutImage,
        nn::os::TransferMemoryType* pImageMemory) NN_NOEXCEPT;

private:
    nn::os::TransferMemoryType  m_TransferMemory;   //!< ファームウェアの TransferMemory オブジェクト
    XcdFirmwareImageInfo        m_ImageInfo;        //!< ファームウェアイメージ
    void*                       m_pMappedData;      //!< ファームウェアのマップ先アドレス
    bool                        m_IsAttached;       //!< イメージ attach 済み
    bool                        m_IsMapped;         //!< イメージマップ済み
};

}}} // namespace nn::hid::detail
