﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os/os_MemoryHeap.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_TransferMemory.h>

#include "hid_XcdFirmwareMemory.h"

namespace nn { namespace hid { namespace detail {

XcdFirmwareMemory::XcdFirmwareMemory() NN_NOEXCEPT
    : m_TransferMemory()
    , m_ImageInfo()
    , m_pMappedData(nullptr)
    , m_IsAttached(false)
    , m_IsMapped(false)
{
}

XcdFirmwareMemory::~XcdFirmwareMemory() NN_NOEXCEPT
{
    Unmap();
}

nn::Result XcdFirmwareMemory::Map(const XcdFirmwareImageInfo& info) NN_NOEXCEPT
{
    m_ImageInfo = info;

    bool needsRollback = true;

    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            Unmap();
        }
    };

    NN_RESULT_DO(MapImage(&m_pMappedData));

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

void XcdFirmwareMemory::Unmap() NN_NOEXCEPT
{
    if (m_IsMapped)
    {
        nn::os::UnmapTransferMemory(&m_TransferMemory);
        m_pMappedData = nullptr;
        m_IsMapped    = false;
    }

    if (m_IsAttached)
    {
        nn::os::DestroyTransferMemory(&m_TransferMemory);
        m_IsAttached = false;
    }
}

void XcdFirmwareMemory::AttachImage(
    nn::os::TransferMemoryType* pImageMemory,
    const XcdFirmwareImageInfo& info) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pImageMemory);

    nn::os::AttachTransferMemory(
        pImageMemory,
        nn::util::align_up(static_cast<size_t>(info.size), nn::os::MemoryPageSize),
        info.handle,
        info.isHandleManaged
    );
}

nn::Result XcdFirmwareMemory::MapImage(void** ppOutImage) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ppOutImage);

    AttachImage(&m_TransferMemory, m_ImageInfo);
    m_IsAttached = true;

    auto result = MapTransferMemory(ppOutImage, &m_TransferMemory);
    if (result.IsSuccess())
    {
        m_IsMapped = true;
    }

    return result;
}

nn::Result XcdFirmwareMemory::MapTransferMemory(
    void** ppOutImage,
    nn::os::TransferMemoryType* pImageMemory) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ppOutImage);
    NN_SDK_REQUIRES_NOT_NULL(pImageMemory);

    bool needsRollback = true;

    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            nn::os::DestroyTransferMemory(pImageMemory);
        }
    };

    NN_RESULT_DO(
        nn::os::MapTransferMemory(
            ppOutImage,
            pImageMemory,
            nn::os::MemoryPermission_ReadOnly
        )
    );

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
