﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/hid/hid_DebugPad.h>
#include <nn/hid/hid_Keyboard.h>
#include <nn/hid/hid_Xpad.h>
#include <nn/hid/system/hid_Keyboard.h>
#include <nn/settings/settings_DebugPad.h>
#include <nn/settings/settings_Xpad.h>

namespace nn { namespace hid { namespace detail {

//!< Windows のキーボードを扱うためのクラスです。
class WindowsKeyboard final
{
    NN_DISALLOW_COPY(WindowsKeyboard);
    NN_DISALLOW_MOVE(WindowsKeyboard);

private:
    //!< Keyboard が接続状態にあるか否かを表す値
    bool m_IsConnected;

    //!< Keyboard のキーの状態
    KeyboardKeySet m_Keys;

    //!< Keyboard の修飾情報
    KeyboardModifierSet m_Modifiers;

    //!< 配列表現された割り当て
    int m_RawMaps[32];

    //!< 配列表現された割り当ての数
    int m_RawMapCount;

public:
    WindowsKeyboard() NN_NOEXCEPT;

    //!< Keyboard が接続状態にあるか否かを表す値を取得します。
    bool IsConnected() const NN_NOEXCEPT
    {
        return m_IsConnected;
    }

    //!< Keyboard のキーの状態を取得します。
    KeyboardKeySet GetKeys() const NN_NOEXCEPT
    {
        return m_Keys;
    }

    //!< Keyboard の修飾情報を取得します。
    KeyboardModifierSet GetModifiers() const NN_NOEXCEPT
    {
        return m_Modifiers;
    }

    //!< 指定の割り当てに基づいたデジタルボタンの押下状態を返します。
    DebugPadButtonSet GetButtons(
        ::nn::settings::DebugPadKeyboardMap& map) NN_NOEXCEPT;

    //!< 指定の割り当てに基づいたデジタルボタンの押下状態を返します。
    BasicXpadButtonSet GetButtons(
        ::nn::settings::BasicXpadKeyboardMap& map) NN_NOEXCEPT;

    //!< Keyboard が接続状態について内部状態を更新します。
    void UpdateConnectionStatus() NN_NOEXCEPT;

    //!< Keyboard のキーの状態について内部状態を更新します。
    void UpdateKeys() NN_NOEXCEPT;

    //!< Keyboard の修飾情報について内部状態を更新します。
    void UpdateModifiers() NN_NOEXCEPT;

    //!< 内部状態をリセットします。
    void Reset() NN_NOEXCEPT;

    //!< Keyboard のロックキーイベントを送信します。
    void SendLockKeyEvent(
        ::nn::hid::system::KeyboardLockKeyEventSet value) NN_NOEXCEPT;

private:
    //!< 配列表現された割り当てに基づいたデジタルボタンの押下状態を返します。
    template<typename T>
    T GetButtons() const NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
