﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Windows.h>
#include <nn/hid/hid_AnalogStickState.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/settings/settings_GenericPadAxis.h>
#include <nn/settings/settings_GenericPadButton.h>
#include <XInput.h>

#include "hid_CommonStateUtility.h"
#include "hid_WindowsHidSdi-os.win.h"
#include "hid_WindowsGenericPad-os.win.h"
#include "hid_WindowsGenericPadAccessor-os.win.h"

#pragma comment(lib, "xinput9_1_0.lib")

namespace nn { namespace hid { namespace detail {

namespace {

//!< XInput 対応コントローラのボタン入力をマップします
void MapXInputButton(
    WindowsGenericPadButtonSet* pOutValue, WORD buttons) NN_NOEXCEPT;

//!< ジョイスティックのハットスイッチの傾きをマップします
void MapJoystickPov(
    WindowsGenericPadButtonSet* pOutValue, int pov) NN_NOEXCEPT;

//!< ジョイスティックの座標値をマップします
void MapJoystickPositions(
    int (&outPositions)[6], const int (&positions)[6],
    const WindowsGenericPadAbility& ability) NN_NOEXCEPT;

} // namespace

WindowsGenericPad::WindowsGenericPad() NN_NOEXCEPT
    : m_IsDetected(false)
    , m_GenericPadId()
    , m_Handle(CreateHidSdiHandle())
    , m_IsStateValid(false)
    , m_State()
    , m_Buttons()
    , m_StickL()
    , m_StickR()
{
    ::std::fill(::std::begin(m_Positions), ::std::end(m_Positions), 0);
}

void WindowsGenericPad::SetGenericPadId(int genericPadId) NN_NOEXCEPT
{
    if (m_IsDetected && m_GenericPadId != genericPadId)
    {
        // 対象とするジョイスティックが変更された場合は未検出状態に
        m_IsDetected = false;
    }

    m_GenericPadId = genericPadId;
}

bool WindowsGenericPad::IsDetected() const NN_NOEXCEPT
{
    return m_IsDetected;
}

bool WindowsGenericPad::IsXInput() const NN_NOEXCEPT
{
    return m_Ability.flags.Test<WindowsGenericPadAbilityFlag::IsXInput>();
}

bool WindowsGenericPad::IsButtonDown(
    ::nn::settings::GenericPadButton button) const NN_NOEXCEPT
{
    switch (button)
    {
    case ::nn::settings::GenericPadButton_1:
    case ::nn::settings::GenericPadButton_2:
    case ::nn::settings::GenericPadButton_3:
    case ::nn::settings::GenericPadButton_4:
    case ::nn::settings::GenericPadButton_5:
    case ::nn::settings::GenericPadButton_6:
    case ::nn::settings::GenericPadButton_7:
    case ::nn::settings::GenericPadButton_8:
    case ::nn::settings::GenericPadButton_9:
    case ::nn::settings::GenericPadButton_10:
    case ::nn::settings::GenericPadButton_11:
    case ::nn::settings::GenericPadButton_12:
    case ::nn::settings::GenericPadButton_13:
    case ::nn::settings::GenericPadButton_14:
    case ::nn::settings::GenericPadButton_15:
    case ::nn::settings::GenericPadButton_16:
        return m_Buttons.Test(
            static_cast<int>(button) -
            static_cast<int>(::nn::settings::GenericPadButton_1));

    case ::nn::settings::GenericPadButton_Left:
        return m_Buttons.Test<WindowsGenericPadButton::Left>();

    case ::nn::settings::GenericPadButton_Up:
        return m_Buttons.Test<WindowsGenericPadButton::Up>();

    case ::nn::settings::GenericPadButton_Right:
        return m_Buttons.Test<WindowsGenericPadButton::Right>();

    case ::nn::settings::GenericPadButton_Down:
        return m_Buttons.Test<WindowsGenericPadButton::Down>();

    case ::nn::settings::GenericPadButton_TriggerL:
        return m_Buttons.Test<WindowsGenericPadButton::TriggerL>();

    case ::nn::settings::GenericPadButton_TriggerR:
        return m_Buttons.Test<WindowsGenericPadButton::TriggerR>();

    default:
        return false;
    }
}

WindowsGenericPadButtonSet WindowsGenericPad::GetButtons() const NN_NOEXCEPT
{
    return m_Buttons;
}

AnalogStickState WindowsGenericPad::GetStickL() const NN_NOEXCEPT
{
    return m_StickL;
}

AnalogStickState WindowsGenericPad::GetStickR() const NN_NOEXCEPT
{
    return m_StickR;
}

void WindowsGenericPad::Update() NN_NOEXCEPT
{
    // 無入力状態で初期化
    m_Buttons.Reset();
    m_StickL = AnalogStickState();
    m_StickR = AnalogStickState();
    ::std::fill(::std::begin(m_Positions), ::std::end(m_Positions), 0);

    typedef WindowsGenericPadAbilityFlag AbilityFlag;

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        GetWindowsGenericPadAbility(&m_Ability, m_GenericPadId));

    m_IsDetected = false;

    if (!IsHidSdiHandleValid(m_Handle))
    {
        if (m_Ability.flags.Test<AbilityFlag::IsJoystick>() &&
            OpenHidSdiDevice(&m_Handle, m_Ability.deviceName).IsSuccess())
        {
            m_IsStateValid = false;
        }
    }
    else
    {
        if (!m_Ability.flags.Test<AbilityFlag::IsJoystick>())
        {
            CloseHidSdiDevice(&m_Handle);
        }
    }

    if (m_Ability.flags.Test<AbilityFlag::IsXInput>())
    {
        XINPUT_STATE state = {};

        if (::XInputGetState(m_Ability.index, &state) == ERROR_SUCCESS)
        {
            m_IsDetected = true;

            MapXInputButton(&m_Buttons, state.Gamepad.wButtons);

            m_Buttons.Set<WindowsGenericPadButton::TriggerL>(
                state.Gamepad.bLeftTrigger > XINPUT_GAMEPAD_TRIGGER_THRESHOLD);
            m_Buttons.Set<WindowsGenericPadButton::TriggerR>(
                state.Gamepad.bRightTrigger > XINPUT_GAMEPAD_TRIGGER_THRESHOLD);

            m_StickL.x = state.Gamepad.sThumbLX;
            m_StickL.y = state.Gamepad.sThumbLY;
            m_StickL = LimitAnalogStick(
                m_StickL, XINPUT_GAMEPAD_LEFT_THUMB_DEADZONE, 0x8000, 0x8000);

            m_StickR.x = state.Gamepad.sThumbRX;
            m_StickR.y = state.Gamepad.sThumbRY;
            m_StickR = LimitAnalogStick(
                m_StickR, XINPUT_GAMEPAD_RIGHT_THUMB_DEADZONE, 0x8000, 0x8000);
        }
    }
    else if (
        m_Ability.flags.Test<AbilityFlag::IsJoystick>() &&
        IsHidSdiHandleValid(m_Handle))
    {
        ::nn::Result result = GetHidSdiGamePadState(&m_Handle, &m_State);

        while (result.IsSuccess())
        {
            m_IsStateValid = true;

            result = GetHidSdiGamePadState(&m_Handle, &m_State);
        }

        if (result.IsFailure() &&
            !::nn::hid::ResultWin32ApiIoPending::Includes(result))
        {
            CloseHidSdiDevice(&m_Handle);
        }
        else
        {
            m_IsDetected = true;

            if (m_IsStateValid)
            {
                m_Buttons._storage[0] = m_State.buttons;

                MapJoystickPov(&m_Buttons, m_State.pov);

                MapJoystickPositions(m_Positions, m_State.positions, m_Ability);
            }
        }
    }
}

void WindowsGenericPad::Reset() NN_NOEXCEPT
{
    m_IsDetected = false;
}

int WindowsGenericPad::GetPosition(
    ::nn::settings::GenericPadAxis axis,
    const ::nn::settings::GenericPadAxisAttributeSet& attribute
    ) const NN_NOEXCEPT
{
    switch (axis)
    {
    case ::nn::settings::GenericPadAxis_X:
        return attribute.Test<
            ::nn::settings::GenericPadAxisAttribute::IsXAxisInverted
            >() ? -m_Positions[0] : m_Positions[0];
    case ::nn::settings::GenericPadAxis_Y:
        return attribute.Test<
            ::nn::settings::GenericPadAxisAttribute::IsYAxisInverted
            >() ? -m_Positions[1] : m_Positions[1];
    case ::nn::settings::GenericPadAxis_Z:
        return attribute.Test<
            ::nn::settings::GenericPadAxisAttribute::IsZAxisInverted
            >() ? -m_Positions[2] : m_Positions[2];
    case ::nn::settings::GenericPadAxis_R:
        return attribute.Test<
            ::nn::settings::GenericPadAxisAttribute::IsRAxisInverted
            >() ? -m_Positions[3] : m_Positions[3];
    case ::nn::settings::GenericPadAxis_U:
        return attribute.Test<
            ::nn::settings::GenericPadAxisAttribute::IsUAxisInverted
            >() ? -m_Positions[4] : m_Positions[4];
    case ::nn::settings::GenericPadAxis_V:
        return attribute.Test<
            ::nn::settings::GenericPadAxisAttribute::IsVAxisInverted
            >() ? -m_Positions[5] : m_Positions[5];
    default:
        return 0;
    }
}

namespace {

void MapXInputButton(
    WindowsGenericPadButtonSet* pOutValue, WORD buttons) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    pOutValue->Set<WindowsGenericPadButton::Up>(
        (buttons & XINPUT_GAMEPAD_DPAD_UP) != 0);
    pOutValue->Set<WindowsGenericPadButton::Down>(
        (buttons & XINPUT_GAMEPAD_DPAD_DOWN) != 0);
    pOutValue->Set<WindowsGenericPadButton::Left>(
        (buttons & XINPUT_GAMEPAD_DPAD_LEFT) != 0);
    pOutValue->Set<WindowsGenericPadButton::Right>(
        (buttons & XINPUT_GAMEPAD_DPAD_RIGHT) != 0);
    pOutValue->Set<WindowsGenericPadButton::Start>(
        (buttons & XINPUT_GAMEPAD_START) != 0);
    pOutValue->Set<WindowsGenericPadButton::Back>(
        (buttons & XINPUT_GAMEPAD_BACK) != 0);
    pOutValue->Set<WindowsGenericPadButton::ThumbL>(
        (buttons & XINPUT_GAMEPAD_LEFT_THUMB) != 0);
    pOutValue->Set<WindowsGenericPadButton::ThumbR>(
        (buttons & XINPUT_GAMEPAD_RIGHT_THUMB) != 0);
    pOutValue->Set<WindowsGenericPadButton::L>(
        (buttons & XINPUT_GAMEPAD_LEFT_SHOULDER) != 0);
    pOutValue->Set<WindowsGenericPadButton::R>(
        (buttons & XINPUT_GAMEPAD_RIGHT_SHOULDER) != 0);
    pOutValue->Set<WindowsGenericPadButton::A>(
        (buttons & XINPUT_GAMEPAD_A) != 0);
    pOutValue->Set<WindowsGenericPadButton::B>(
        (buttons & XINPUT_GAMEPAD_B) != 0);
    pOutValue->Set<WindowsGenericPadButton::X>(
        (buttons & XINPUT_GAMEPAD_X) != 0);
    pOutValue->Set<WindowsGenericPadButton::Y>(
        (buttons & XINPUT_GAMEPAD_Y) != 0);
}

void MapJoystickPov(
    WindowsGenericPadButtonSet* pOutValue, int pov) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    if (0 <= pov)
    {
        pOutValue->Set<WindowsGenericPadButton::Up>(pov < 2 || pov > 6);
        pOutValue->Set<WindowsGenericPadButton::Down>(pov > 2 && pov < 6);
        pOutValue->Set<WindowsGenericPadButton::Left>(4 < pov);
        pOutValue->Set<WindowsGenericPadButton::Right>(pov > 0 && pov < 4);
    }
}

int GetJoystickPosition(int position, int32_t min, int32_t max) NN_NOEXCEPT
{
    const int srcRange = static_cast<int>(max - min);

    const int dstRange = AnalogStickMax * 2 + 1;

    int dstPosition = position - (srcRange + 1) / 2;

    if (0 < srcRange && srcRange < dstRange)
    {
        dstPosition *= dstRange / srcRange;
    }
    else if (0 < dstRange && dstRange < srcRange)
    {
        dstPosition /= srcRange / dstRange;
    }

    if (dstPosition > AnalogStickMax)
    {
        dstPosition = AnalogStickMax;
    }
    else if (dstPosition < -AnalogStickMax)
    {
        dstPosition = -AnalogStickMax;
    }

    return dstPosition;
}

void MapJoystickPositions(
    int (&outPositions)[6], const int (&positions)[6],
    const WindowsGenericPadAbility& ability) NN_NOEXCEPT
{
    outPositions[0] = GetJoystickPosition(
        positions[0], ability.xMin, ability.xMax);

    outPositions[1] = GetJoystickPosition(
        positions[1], ability.yMin, ability.yMax);

    typedef WindowsGenericPadAbilityFlag AbilityFlag;

    if (ability.flags.Test<AbilityFlag::HasZ>())
    {
        outPositions[2] = GetJoystickPosition(
            positions[2], ability.zMin, ability.zMax);
    }

    if (ability.flags.Test<AbilityFlag::HasR>())
    {
        outPositions[3] = GetJoystickPosition(
            positions[3], ability.rMin, ability.rMax);
    }

    if (ability.flags.Test<AbilityFlag::HasU>())
    {
        outPositions[4] = GetJoystickPosition(
            positions[4], ability.uMin, ability.uMax);
    }

    if (ability.flags.Test<AbilityFlag::HasV>())
    {
        outPositions[5] = GetJoystickPosition(
            positions[5], ability.vMin, ability.vMax);
    }
}

} // namespace

}}} // namespace nn::hid::detail
