﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/result/result_HandlingUtility.h>
#include <nn/hid/hid_ResultPrivate.h>
#include "hid_VibratorXcdDriver.h"

namespace nn { namespace hid { namespace detail {

Result VibratorXcdDriver::IncreaseReferenceCount() NN_NOEXCEPT
{
    m_ReferenceCount++;

    if (m_ReferenceCount == 1)
    {
        //初期化時に振動を停止する
        VibrationValue value = VibrationValue::Make();
        SendVibrationValue(value);
    }

    if (m_IsActivated)
    {
        (void)SetEnabled(true);
    }

    NN_RESULT_SUCCESS;
}

Result VibratorXcdDriver::DecreaseReferenceCount() NN_NOEXCEPT
{
    if (m_ReferenceCount == 1)
    {
        //アプリ終了時に振動を停止する
        VibrationValue value = VibrationValue::Make();
        SendVibrationValue(value);
    }

    m_ReferenceCount--;

    if (m_ReferenceCount <= 0)
    {
        m_ReferenceCount = 0;
        (void)SetEnabled(false);
    }

    NN_RESULT_SUCCESS;
}

void VibratorXcdDriver::ForceStopVibration() NN_NOEXCEPT
{
    VibrationValue value = VibrationValue::Make();
    SendVibrationValue(value);
}

Result VibratorXcdDriver::ActivateVibrator(VibratorXcdConfig vibratorXcdConfig) NN_NOEXCEPT
{
    m_Config = vibratorXcdConfig;
    m_IsActivated = true;

    if (m_ReferenceCount > 0)
    {
        (void)SetEnabled(true);
    }

    //アクティベートされた時に振動を一旦停止する
    VibrationValue value = VibrationValue::Make();
    SendVibrationValue(value);

    NN_RESULT_SUCCESS;
}

Result VibratorXcdDriver::DeactivateVibrator() NN_NOEXCEPT
{
    //ディアクティベートされた時に振動を停止する
    VibrationValue value = VibrationValue::Make();
    SendVibrationValue(value);

    m_IsActivated = false;

    if (m_ReferenceCount <= 0)
    {
        (void)SetEnabled(false);
    }

    NN_RESULT_SUCCESS;
}

Result VibratorXcdDriver::SetEnabled(bool isEnabled) NN_NOEXCEPT
{
    if(!m_IsActivated)
    {
        NN_RESULT_SUCCESS;
    }

    bool current;
    NN_RESULT_DO(nn::xcd::IsVibratorEnabled(&current, m_Config.position, m_Config.handle));

    if(current != isEnabled)
    {
        return nn::xcd::SetVibratorEnabled(isEnabled, m_Config.position, m_Config.handle);
    }
    NN_RESULT_SUCCESS;
}

Result VibratorXcdDriver::SendVibrationValue(const VibrationValue& value) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_ReferenceCount > 0, ResultVibrationDeviceNotInitialized());

    if (!m_IsActivated)
    {
        NN_RESULT_SUCCESS;
    }

    nn::xcd::VibrationValue xcdValue;
    xcdValue.amplitudeLow = value.amplitudeLow;
    xcdValue.frequencyLow = value.frequencyLow;
    xcdValue.amplitudeHigh = value.amplitudeHigh;
    xcdValue.frequencyHigh = value.frequencyHigh;

    return nn::xcd::SendVibrationValue(xcdValue, m_Config.position, m_Config.handle);
}

Result VibratorXcdDriver::GetActualVibrationValue(VibrationValue* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_RESULT_THROW_UNLESS(m_ReferenceCount > 0, ResultVibrationDeviceNotInitialized());

    if (!m_IsActivated)
    {
        nn::hid::VibrationValue zero = nn::hid::VibrationValue::Make();
        *pOutValue = zero;
        NN_RESULT_SUCCESS;
    }

    nn::xcd::VibrationValue xcdValue;
    NN_RESULT_DO(nn::xcd::GetActualVibrationValue(&xcdValue, m_Config.position, m_Config.handle));

    pOutValue->amplitudeLow = xcdValue.amplitudeLow;
    pOutValue->frequencyLow = xcdValue.frequencyLow;
    pOutValue->amplitudeHigh = xcdValue.amplitudeHigh;
    pOutValue->frequencyHigh = xcdValue.frequencyHigh;
    NN_RESULT_SUCCESS;
}

Result VibratorXcdDriver::StartVibrationOnConnect() NN_NOEXCEPT
{
    NN_RESULT_DO(nn::xcd::StartVibrationOnConnect(m_Config.handle));
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
