﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/result/result_HandlingUtility.h>
#include <nn/hid/hid_ResultPrivate.h>
#include "hid_VibratorGcDriver.h"

namespace nn { namespace hid { namespace detail {

void VibratorGcDriver::SetGcAdapterDriver(GcAdapterDriver* pDriver) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDriver);
    m_pDriver = pDriver;
}

Result VibratorGcDriver::IncreaseReferenceCount() NN_NOEXCEPT
{
    m_ReferenceCount++;

    if (m_ReferenceCount == 1)
    {
        //初期化時に振動を停止する
        SendCommand(VibrationGcErmCommand_Stop);
    }

    NN_RESULT_SUCCESS;
}

Result VibratorGcDriver::DecreaseReferenceCount() NN_NOEXCEPT
{
    if (m_ReferenceCount == 1)
    {
        //アプリ終了時に振動を停止する
        SendCommand(VibrationGcErmCommand_Stop);
    }

    m_ReferenceCount--;

    if (m_ReferenceCount <= 0)
    {
        m_ReferenceCount = 0;
    }

    NN_RESULT_SUCCESS;
}

void VibratorGcDriver::ForceStopVibration() NN_NOEXCEPT
{
    SendCommand(VibrationGcErmCommand_Stop);
}

Result VibratorGcDriver::ActivateVibrator(GcControllerIndex index) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    m_Index = index;
    m_IsActivated = true;

    //アクティベートされた時に振動を一旦停止する
    SendCommand(VibrationGcErmCommand_Stop);

    NN_RESULT_SUCCESS;
}

Result VibratorGcDriver::DeactivateVibrator() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    //ディアクティベートされた時に振動を停止する
    SendCommand(VibrationGcErmCommand_Stop);

    m_IsActivated = false;

    NN_RESULT_SUCCESS;
}

Result VibratorGcDriver::SendCommand(const VibrationGcErmCommand& command) NN_NOEXCEPT
{
//    NN_RESULT_THROW_UNLESS(m_ReferenceCount > 0, ResultVibrationDeviceNotInitialized());
    if (!m_IsActivated)
    {
        NN_RESULT_SUCCESS;
    }

    switch (command)
    {
    case VibrationGcErmCommand_Stop:
        m_pDriver->ControlMotor(m_Index, GcControllerMotorState_Off);
        break;
    case VibrationGcErmCommand_Start:
        m_pDriver->ControlMotor(m_Index, GcControllerMotorState_On);
        break;
    case VibrationGcErmCommand_StopHard:
        m_pDriver->ControlMotor(m_Index, GcControllerMotorState_Brake);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
}

Result VibratorGcDriver::GetActualCommand(VibrationGcErmCommand* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
//    NN_RESULT_THROW_UNLESS(m_ReferenceCount > 0, ResultVibrationDeviceNotInitialized());

    if (!m_IsActivated)
    {
        *pOutValue = VibrationGcErmCommand_Stop;
        NN_RESULT_SUCCESS;
    }

    auto driverState = m_pDriver->GetMotorState(m_Index);
    *pOutValue = static_cast<VibrationGcErmCommand>(driverState);
    NN_RESULT_SUCCESS;
}

Result VibratorGcDriver::StartVibrationOnConnect() NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
