﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/system/hid_UniquePad.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/util/util_BitPack.h>

#include "hid_ActivationCount.h"
#include "hid_MultiWaitEvent.h"
#include "hid_UniquePadManager.h"

namespace nn { namespace hid { namespace detail {

//!< UniquePad デバイス割り当ての解決を担うマネージャを扱うクラスです。
class UniquePadAssignmentManager final
{
    NN_DISALLOW_COPY(UniquePadAssignmentManager) NN_NOEXCEPT;
    NN_DISALLOW_MOVE(UniquePadAssignmentManager) NN_NOEXCEPT;

private:
    //!< UniquePad Manager
    UniquePadManager* m_pUniquePadManagers[system::UniquePadIdCountMax];

    //!< UniquePad コントロールマネージャ がアクティブ化された回数
    ActivationCount m_ActivationCount;

    //!< UniquePad 接続状態変化通知イベント
    ::nn::os::SystemEvent m_UniquePadConnectionEvent;

    //!< IAbstractedPad
    IAbstractedPad* m_pAbstractedPads[IAbstractedPadCountMax];

    //!< IAbstractedPad の数
    int m_AbstractedPadCount;

    //!< 使用中の UniquePadId (Ocean リビルドされたら修正する)
    util::BitPack16 m_UsedIds;

    //!< 最後に割り振った UniquePadId (Ocean リビルドされたら修正する)
    system::UniquePadId m_LatestId;

public:
    UniquePadAssignmentManager() NN_NOEXCEPT;

    ~UniquePadAssignmentManager() NN_NOEXCEPT;

    //!< IAbstractedPad をセットします
    void AddIAbstractedPad(IAbstractedPad* pPads) NN_NOEXCEPT;

    //!< UniquePadManager をセットします。
    void SetUniquePadManagers(UniquePadManager* pUniquePadManagers, int count) NN_NOEXCEPT;

    //!< マネージャをアクティブ化します。
    ::nn::Result Activate() NN_NOEXCEPT;

    //!< マネージャを非アクティブ化します。
    ::nn::Result Deactivate() NN_NOEXCEPT;

    //!< UniquePad の接続状態変化通知イベントをバインドします。
    ::nn::Result AcquireUniquePadConnectionEventHandle(::nn::os::NativeHandle* pOutHandle) NN_NOEXCEPT;

    //!< デバイスの解決状態を更新します。
    void UpdateDeviceMap() NN_NOEXCEPT;

    //!< UniquePad ID を取得します。
    ::nn::Result GetUniquePadIds(int* pOutCount,
                                 system::UniquePadId* outUniquePadIds,
                                 int count) const NN_NOEXCEPT;

    //!< USB コントローラを接続します。
    ::nn::Result ConnectUsbPadsAsync() NN_NOEXCEPT;

    //!< USB コントローラを切断します。
    ::nn::Result DisconnectUsbPadsAsync() NN_NOEXCEPT;

    //!< Bluetooth 接続可能なコントローラーの数を取得します
    int GetAllowedBluetoothLinksCount() NN_NOEXCEPT;

private:
    //!< 切断またはインタフェースの更新されているデバイスかどうか確認し、切断されている場合はディアクティベートをします。
    void CheckForDeviceDetach() NN_NOEXCEPT;

    //!< 指定されたデバイスが UniquePadManager に未登録のデバイスかどうか
    bool IsUnregisteredAbstractedPad(const AbstractedPadId& id) NN_NOEXCEPT;

    //!< UniquePad に対して デバイスを追加します。
    void AddAbstractedPadToUniquePad(IAbstractedPad* pPad) NN_NOEXCEPT;

    //!< UniquePadId を発行する
    system::UniquePadId PublishUniquePadId() NN_NOEXCEPT;

    //!< 使用済みの UniquePadId を解放する
    void ReleaseUniquePadId(system::UniquePadId id) NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
