﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/util/util_TypedStorage.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/hid_NpadCommonTypes.h>
#include <nn/hid/hid_NpadJoyCommon.h>
#include <nn/hid/hid_Result.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>

#include "hid_BluetoothMode.h"
#include "hid_UniquePadAssignmentManager.h"
#include "hid_UniquePadId.h"

namespace nn { namespace hid { namespace detail {

UniquePadAssignmentManager::UniquePadAssignmentManager() NN_NOEXCEPT
    : m_ActivationCount()
    , m_UniquePadConnectionEvent(::nn::os::EventClearMode_ManualClear, true)
    , m_AbstractedPadCount(0)
{
    m_UsedIds.Clear();
    m_LatestId._storage = 0;
}

UniquePadAssignmentManager::~UniquePadAssignmentManager() NN_NOEXCEPT
{
    // 何もしない
}

void UniquePadAssignmentManager::AddIAbstractedPad(IAbstractedPad* pPads) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPads);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pAbstractedPads[m_AbstractedPadCount] = pPads;
    m_AbstractedPadCount++;
}

void UniquePadAssignmentManager::SetUniquePadManagers(UniquePadManager* pUniquePadManagers, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pUniquePadManagers);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    NN_SDK_REQUIRES_GREATER_EQUAL(static_cast<int>(NN_ARRAY_SIZE(m_pUniquePadManagers)), count);

    for (int i = 0; i < count; ++i)
    {
        m_pUniquePadManagers[i] = &pUniquePadManagers[i];
    }
}

::nn::Result UniquePadAssignmentManager::Activate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultGamePadDriverActivationUpperLimitOver());
    NN_SDK_REQUIRES_NOT_NULL(m_pAbstractedPads[0]);
    NN_SDK_REQUIRES_NOT_NULL(m_pUniquePadManagers[0]);

    // アクティブ化した回数をインクリメント
    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result UniquePadAssignmentManager::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultGamePadDriverDeactivationLowerLimitOver());
    NN_SDK_REQUIRES_NOT_NULL(m_pAbstractedPads[0]);
    NN_SDK_REQUIRES_NOT_NULL(m_pUniquePadManagers[0]);

    // アクティブ化した回数をデクリメント
    --m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result UniquePadAssignmentManager::AcquireUniquePadConnectionEventHandle(
    ::nn::os::NativeHandle* pOutHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutHandle);

    *pOutHandle = m_UniquePadConnectionEvent.GetReadableHandle();

    // ハンドルの取得直後はシグナル状態であることを保証
    m_UniquePadConnectionEvent.Signal();

    NN_RESULT_SUCCESS;
}

void UniquePadAssignmentManager::UpdateDeviceMap() NN_NOEXCEPT
{
    if (m_ActivationCount.IsZero())
    {
        return;
    }

    // 切断されたデバイスをディアクティベート
    CheckForDeviceDetach();

    // 新規に接続されたデバイスを追加
    for(int i = 0; i < m_AbstractedPadCount; i++)
    {
        if (m_pAbstractedPads[i]->IsConnected() == false)
        {
            continue;
        }
        if (IsUnregisteredAbstractedPad(m_pAbstractedPads[i]->GetId()))
        {
            AddAbstractedPadToUniquePad(m_pAbstractedPads[i]);
        }
    }
}

::nn::Result UniquePadAssignmentManager::GetUniquePadIds(int* pOutCount,
                                                      system::UniquePadId* outUniquePadIds,
                                                      int count) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER(count, 0);

    *pOutCount = 0;
    for (auto& pManager : m_pUniquePadManagers)
    {
        if (pManager->IsConnected())
        {
            outUniquePadIds[*pOutCount] = pManager->GetId();
            (*pOutCount)++;

            if (*pOutCount == count)
            {
                break;
            }
        }
    }

    NN_RESULT_SUCCESS;
}

void UniquePadAssignmentManager::CheckForDeviceDetach() NN_NOEXCEPT
{
    for (auto& pManager : m_pUniquePadManagers)
    {
        auto detachedId = pManager->GetId();
        if (pManager->CheckForDeviceDetach() == true)
        {
            ReleaseUniquePadId(detachedId);
            // 切断されている場合は、更新を通知
            m_UniquePadConnectionEvent.Signal();
        }
    }
}

bool UniquePadAssignmentManager::IsUnregisteredAbstractedPad(const AbstractedPadId& id) NN_NOEXCEPT
{
    // UniquePadManager を探索して接続済みかどうかをチェック
    for (auto& pManager : m_pUniquePadManagers)
    {
        if(pManager->IsConnectedAbstractedPad(id) == true)
        {
            return false;
        }
    }

    return true;
}

void UniquePadAssignmentManager::AddAbstractedPadToUniquePad(IAbstractedPad* pPad) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPad);

    for (auto& pManager : m_pUniquePadManagers)
    {
        if (pManager->IsConnected() == false)
        {
            pManager->AttachAbstractedPad(pPad, PublishUniquePadId());
            m_UniquePadConnectionEvent.Signal();
            return;
        }
    }
}

::nn::Result UniquePadAssignmentManager::ConnectUsbPadsAsync() NN_NOEXCEPT
{
    for (int i = 0; i < m_AbstractedPadCount; i++)
    {
        auto pAbstractedPad = m_pAbstractedPads[i];

        if(pAbstractedPad->IsConnected() == false &&
           pAbstractedPad->GetInterfaceType() == ::nn::hid::system::InterfaceType_Usb &&
           pAbstractedPad->GetType() == ::nn::hid::detail::AbstractedPadType_Usb &&
           pAbstractedPad->IsAttached() == true &&
           pAbstractedPad->GetDeviceType().Test<::nn::hid::system::DeviceType::FullKeyController>())
        {
            pAbstractedPad->Connect();
        }
    }

    NN_RESULT_SUCCESS;
}

::nn::Result UniquePadAssignmentManager::DisconnectUsbPadsAsync() NN_NOEXCEPT
{
    for (int i = 0; i < m_AbstractedPadCount; i++)
    {
        auto pAbstractedPad = m_pAbstractedPads[i];

        if(pAbstractedPad->IsConnected() == true &&
           pAbstractedPad->GetInterfaceType() == ::nn::hid::system::InterfaceType_Usb &&
           pAbstractedPad->GetType() == ::nn::hid::detail::AbstractedPadType_Usb &&
           pAbstractedPad->IsAttached() == true &&
           pAbstractedPad->GetDeviceType().Test<::nn::hid::system::DeviceType::FullKeyController>())
        {
            pAbstractedPad->Detach();
        }
    }

    NN_RESULT_SUCCESS;
}

int UniquePadAssignmentManager::GetAllowedBluetoothLinksCount() NN_NOEXCEPT
{
    static const int MaxControllers = 8;
    int bluetoothDeviceCount = 0;
    int usbDeviceCount = 0;
    int bleDeviceCount = 0;

    // Bluetooth / Usb でそれぞれ接続されているデバイスをカウント
    for (int i = 0; i < m_AbstractedPadCount; i++)
    {
        if (m_pAbstractedPads[i]->IsConnected() == false)
        {
            continue;
        }
        if (m_pAbstractedPads[i]->GetDeviceType().Test<system::DeviceType::Palma>() == true)
        {
            ++bleDeviceCount;
            continue;
        }
        switch (m_pAbstractedPads[i]->GetInterfaceType())
        {
        case system::InterfaceType_Bluetooth:
            bluetoothDeviceCount++;
            break;
        case system::InterfaceType_Usb:
            usbDeviceCount++;
            break;
        default:
            // 何もしない
            ;
        }
    }

    // Bluetooth 接続できる数 か コントローラー接続できる数の小さいほうを返す
    return std::min(
        MaxControllers - (bluetoothDeviceCount + usbDeviceCount + bleDeviceCount),     // 残り接続できるコントローラーの数
        GetBluetoothMaxConnectionCount() - bluetoothDeviceCount       // 残り接続できる Bluetooth デバイスの数
    );
}

system::UniquePadId UniquePadAssignmentManager::PublishUniquePadId() NN_NOEXCEPT
{
    auto nextId = m_LatestId;

    for (int i = 0; i < system::UniquePadIdCountMax; ++i)
    {
        ++(nextId._storage);
        if (nextId._storage == system::UniquePadIdCountMax)
        {
            nextId._storage = 0;
        }
        if (m_UsedIds.GetBit(static_cast<int>(nextId._storage)) == false)
        {
            // Bit が使われていなかったら確定
            break;
        }
    }
    m_UsedIds.SetBit(static_cast<int>(nextId._storage), true);
    m_LatestId = nextId;
    return nextId;
}

void UniquePadAssignmentManager::ReleaseUniquePadId(system::UniquePadId id) NN_NOEXCEPT
{
    m_UsedIds.SetBit(static_cast<int>(id._storage), false);
}


}}} // namespace nn::hid::detail
