﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd.h>

#include "hid_CommonStateUtility.h"
#include "hid_UniquePadAnalogStickDriver.h"
#include "hid_UniquePadUtility.h"

namespace nn { namespace hid { namespace detail {

namespace {

const int ReleasePositionThreshold  = 400;    // リリース位置の範囲

AnalogStickState CastAnalogStickState(::nn::xcd::AnalogStickState& xcdState)
{
    nn::hid::AnalogStickState state = { static_cast<int32_t>(xcdState.x), static_cast<int32_t>(xcdState.y) };
    return state;
}

::nn::xcd::AnalogStickState CastAnalogStickState(AnalogStickState& xcdState)
{
    ::nn::xcd::AnalogStickState state = { static_cast<int16_t>(xcdState.x), static_cast<int16_t>(xcdState.y) };
    return state;
}

} // namespace

UniquePadAnalogStickDriver::UniquePadAnalogStickDriver() NN_NOEXCEPT
    : m_pAbstractedPadHolder(nullptr)
    , m_pHandheldManager(nullptr)
    , m_Position(system::AnalogStickPosition_Left)
    , m_CalibrationManager()
    , m_Stage(system::AnalogStickManualCalibrationStage_Completed)
    , m_State()
{
    m_Flags.Reset();
}

UniquePadAnalogStickDriver::~UniquePadAnalogStickDriver() NN_NOEXCEPT
{
    // 何もしない
}

void UniquePadAnalogStickDriver::SetHandheldManager(HandheldManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    m_pHandheldManager = pManager;
}

void UniquePadAnalogStickDriver::SetUniquePadAbstractedPadHolder(UniquePadAbstractedPadHolder* pHolder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHolder);
    m_pAbstractedPadHolder = pHolder;
}

void UniquePadAnalogStickDriver::SetPosition(system::AnalogStickPosition position) NN_NOEXCEPT
{
    m_Position = position;
}

::nn::Result UniquePadAnalogStickDriver::StartAnalogStickManualCalibration() NN_NOEXCEPT
{
    NN_RESULT_DO(this->IsAnalogStickCalibrationSupportedImpl());

    nn::xcd::DeviceHandle xcdHandle;
    NN_RESULT_THROW_UNLESS(
        m_pAbstractedPadHolder->GetXcdDeviceHandle(&xcdHandle),
        system::ResultAnalogStickManualCalibrationNotSupported());

    ::nn::xcd::DeviceInfo deviceInfo;
    if (::nn::xcd::GetDeviceInfo(&deviceInfo, xcdHandle).IsFailure())
    {
        NN_RESULT_THROW(system::ResultAnalogStickDeviceNotConnected());
    }

    ::nn::xcd::AnalogStickDeviceParameter parameter;
    switch (m_Position)
    {
    case system::AnalogStickPosition_Left:
        if (deviceInfo.deviceType == ::nn::xcd::DeviceType_Right)
        {
            NN_RESULT_THROW(system::ResultAnalogStickDeviceNotConnected());
        }
        ::nn::xcd::GetLeftAnalogStickDeviceParameter(&parameter, xcdHandle);
        break;
    case system::AnalogStickPosition_Right:
        if (deviceInfo.deviceType == ::nn::xcd::DeviceType_Left)
        {
            NN_RESULT_THROW(system::ResultAnalogStickDeviceNotConnected());
        }
        ::nn::xcd::GetLeftAnalogStickDeviceParameter(&parameter, xcdHandle);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    auto originMin = CastAnalogStickState(parameter.originRangeMin);
    auto originMax = CastAnalogStickState(parameter.originRangeMax);
    auto strokePositive = CastAnalogStickState(parameter.minimumStrokePositive);
    auto strokeNegative = CastAnalogStickState(parameter.minimumStrokeNegative);

    // ステートを初期化
    m_State = AnalogStickState();
    m_Flags.Reset();

    NN_RESULT_DO(
        m_CalibrationManager.StartAnalogStickManualCalibration(
            originMin,
            originMax,
            strokePositive,
            strokeNegative,
            ReleasePositionThreshold
        )
    );

    m_Stage = system::AnalogStickManualCalibrationStage_ReleaseFromRight;
    NN_RESULT_SUCCESS;
}

::nn::Result UniquePadAnalogStickDriver::RetryCurrentAnalogStickManualCalibrationStage() NN_NOEXCEPT
{
    NN_RESULT_DO(m_CalibrationManager.RetryCurrentAnalogStickManualCalibration());
    NN_RESULT_SUCCESS;
}

::nn::Result UniquePadAnalogStickDriver::CancelAnalogStickManualCalibration() NN_NOEXCEPT
{
    NN_RESULT_DO(m_CalibrationManager.CancelAnalogStickManualCalibration());
    InitializeAnalogStickManualCalibration();
    NN_RESULT_SUCCESS;
}

::nn::Result UniquePadAnalogStickDriver::ResetAnalogStickManualCalibration() NN_NOEXCEPT
{
    //Reset は Result を返さないので内部ステートだけ変更して結果は必ず NN_RESULT_SUCCESS
    if (this->ResetAnalogStickManualCalibrationImpl().IsSuccess())
    {
        m_Stage = system::AnalogStickManualCalibrationStage_Clear;
    }
    else
    {
        m_Stage = system::AnalogStickManualCalibrationStage_ClearCompleted;
    }

    NN_RESULT_SUCCESS;
}

void UniquePadAnalogStickDriver::GetAnalogStickState(AnalogStickState* pOutValue) NN_NOEXCEPT
{
    // 無入力で初期化
    *pOutValue = AnalogStickState();

    IAbstractedPad* pPad;
    if (m_pAbstractedPadHolder->GetIAbstractedPad(&pPad) == false)
    {
        // コントローラー未接続時は無入力
        return;
    }

    if (m_pHandheldManager->IsHandheldEnabled() == false &&
        pPad->GetInterfaceType() == system::InterfaceType_Rail)
    {
        // 携帯機操作が無効な場合は無入力
        return;
    }

    if (m_Stage == system::AnalogStickManualCalibrationStage_Completed ||
        m_Stage == system::AnalogStickManualCalibrationStage_ClearCompleted)
    {
        // キャリブレーションを行っていない時は、補正済みの値を返す
        auto padState = pPad->GetPadState();

        switch (m_Position)
        {
        case system::AnalogStickPosition_Left:
            if (pPad->GetDeviceType().Test<system::DeviceType::JoyConRight>() ||
                pPad->GetDeviceType().Test<system::DeviceType::HandheldJoyRight>())
            {
                return;
            }
            *pOutValue = padState.analogStickL;
            break;
        case system::AnalogStickPosition_Right:
            if (pPad->GetDeviceType().Test<system::DeviceType::JoyConLeft>() ||
                pPad->GetDeviceType().Test<system::DeviceType::HandheldJoyLeft>())
            {
                return;
            }
            *pOutValue = padState.analogStickR;
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        return;
    }

    // キャリブレーション中は、保持している値を返す
    *pOutValue = m_State;
}

::nn::Result UniquePadAnalogStickDriver::GetAnalogStickManualCalibrationStage(system::AnalogStickManualCalibrationStage* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_pAbstractedPadHolder->IsConnected(), system::ResultUniquePadDisconnected());

    *pOutValue = m_Stage;
    NN_RESULT_SUCCESS;
}

bool UniquePadAnalogStickDriver::IsAnalogStickButtonPressed() NN_NOEXCEPT
{
    IAbstractedPad* pPad;
    if (m_pAbstractedPadHolder->GetIAbstractedPad(&pPad) == false)
    {
        // コントローラー未接続時は無入力
        return false;
    }

    if (m_pHandheldManager->IsHandheldEnabled() == false &&
        pPad->GetInterfaceType() == system::InterfaceType_Rail)
    {
        // 携帯機操作が無効な場合は無入力
        return false;
    }

    auto padState = pPad->GetPadState();

    switch (m_Position)
    {
    case system::AnalogStickPosition_Left:
        if (pPad->GetDeviceType().Test<system::DeviceType::JoyConRight>() ||
            pPad->GetDeviceType().Test<system::DeviceType::HandheldJoyRight>())
        {
            return false;
        }
        return padState.buttons.Test<AbstractedPadButton::StickL>();
    case system::AnalogStickPosition_Right:
        if (pPad->GetDeviceType().Test<system::DeviceType::JoyConLeft>() ||
            pPad->GetDeviceType().Test<system::DeviceType::HandheldJoyLeft>())
        {
            return false;
        }
        return padState.buttons.Test<AbstractedPadButton::StickR>();
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

bool UniquePadAnalogStickDriver::IsAnalogStickInReleasePosition() NN_NOEXCEPT
{
    return m_Flags.Test<AnalogStickCalibrationFlags::IsAnalogStickInReleasePosition>();
}

bool UniquePadAnalogStickDriver::IsAnalogStickInCircumference() NN_NOEXCEPT
{
    return m_Flags.Test<AnalogStickCalibrationFlags::IsAnalogStickInCircumference>();
}

void UniquePadAnalogStickDriver::Update() NN_NOEXCEPT
{
    // コントローラーがつながっていないときは何もしない
    IAbstractedPad* pPad;
    if (m_pAbstractedPadHolder->GetIAbstractedPad(&pPad) == false)
    {
        return;
    }

    // 補正が行われていないときも何もしない
    if (m_Stage == system::AnalogStickManualCalibrationStage_Completed ||
        m_Stage == system::AnalogStickManualCalibrationStage_ClearCompleted)
    {
        return;
    }

    // ポジションとデバイスタイプが一致しない場合はスキップ
    switch (m_Position)
    {
    case system::AnalogStickPosition_Left:
        if (pPad->GetDeviceType().Test<system::DeviceType::JoyConRight>() ||
            pPad->GetDeviceType().Test<system::DeviceType::HandheldJoyRight>())
        {
            return;
        }
        break;
    case system::AnalogStickPosition_Right:
        if (pPad->GetDeviceType().Test<system::DeviceType::JoyConLeft>() ||
            pPad->GetDeviceType().Test<system::DeviceType::HandheldJoyLeft>())
        {
            return;
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    // キャリブレーション中は内部状態を更新する
    m_Flags.Reset();
    m_State = AnalogStickState();
    auto currentStage = m_Stage;
    m_Stage = system::AnalogStickManualCalibrationStage_Completed;

    nn::xcd::DeviceHandle xcdHandle;
    if (m_pAbstractedPadHolder->GetXcdDeviceHandle(&xcdHandle) == false)
    {
        // 通常はここにはこない
        return;
    }

    ::nn::xcd::PadState xcdState;
    ::nn::xcd::AnalogStickState* pAnalogStickState;
    ::nn::xcd::GetPadState(&xcdState, xcdHandle);
    switch (m_Position)
    {
    case system::AnalogStickPosition_Left:
        pAnalogStickState = &xcdState.analogStickL;
        break;
    case system::AnalogStickPosition_Right:
        pAnalogStickState = &xcdState.analogStickR;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    AnalogStickState convertedState;

    switch (currentStage)
    {
    case system::AnalogStickManualCalibrationStage_Completed:
    case system::AnalogStickManualCalibrationStage_ClearCompleted:
        // ここには到達しない
        break;
    case system::AnalogStickManualCalibrationStage_ReleaseFromRight:
    case system::AnalogStickManualCalibrationStage_ReleaseFromBottom:
    case system::AnalogStickManualCalibrationStage_ReleaseFromLeft:
    case system::AnalogStickManualCalibrationStage_ReleaseFromTop:
    case system::AnalogStickManualCalibrationStage_Rotate:
        convertedState = CastAnalogStickState(*pAnalogStickState);
        ProceedAnalogStickManualCalibration(convertedState);
        break;
    case system::AnalogStickManualCalibrationStage_Update:
    case system::AnalogStickManualCalibrationStage_Clear:
        ProceedAnalogStickCalibrationUpdate(currentStage);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void UniquePadAnalogStickDriver::Reset() NN_NOEXCEPT
{
    m_State = AnalogStickState();
    m_Flags.Reset();
    m_Stage = system::AnalogStickManualCalibrationStage_Completed;
    m_CalibrationManager.Reset();
}

void UniquePadAnalogStickDriver::ProceedAnalogStickManualCalibration(AnalogStickState& state) NN_NOEXCEPT
{
    if (this->IsAnalogStickCalibrationSupportedImpl().IsFailure())
    {
        return;
    }

    nn::xcd::DeviceHandle xcdHandle;
    if (m_pAbstractedPadHolder->GetXcdDeviceHandle(&xcdHandle) == false)
    {
        return;
    }

    // キャリブレーション補正処理
    bool isInReleasePosition = false;
    bool isInCircumference = false;
    m_CalibrationManager.ProceedAnalogStickCalibration(&m_Stage, &m_State, &isInReleasePosition, &isInCircumference, state);

    m_Flags.Set<AnalogStickCalibrationFlags::IsAnalogStickInReleasePosition>(isInReleasePosition);
    m_Flags.Set<AnalogStickCalibrationFlags::IsAnalogStickInCircumference>(isInCircumference);
    // キャリブレーションが完了したかどうかをチェック
    if (m_Stage == system::AnalogStickManualCalibrationStage_Completed)
    {
        m_Stage = system::AnalogStickManualCalibrationStage_Update;

        // キャリブレーションが行われた結果の取得
        AnalogStickState origin;
        AnalogStickState circuitMin;
        AnalogStickState circuitMax;
        m_CalibrationManager.GetAnalogStickCalibrationValue(&origin, &circuitMin, &circuitMax);

        // Xcd の型に変換
        ::nn::xcd::AnalogStickValidRange updateValue = { CastAnalogStickState(origin),
                                                         CastAnalogStickState(circuitMin),
                                                         CastAnalogStickState(circuitMax) };

        // 更新を行う
        switch (m_Position)
        {
        case system::AnalogStickPosition_Left:
            ::nn::xcd::UpdateLeftAnalogStickValidRange(updateValue, xcdHandle);
            break;
        case system::AnalogStickPosition_Right:
            ::nn::xcd::UpdateRightAnalogStickValidRange(updateValue, xcdHandle);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

void UniquePadAnalogStickDriver::ProceedAnalogStickCalibrationUpdate(system::AnalogStickManualCalibrationStage currentStage) NN_NOEXCEPT
{
    if (this->IsAnalogStickCalibrationSupportedImpl().IsFailure())
    {
        return;
    }

    nn::xcd::DeviceHandle xcdHandle;
    if (m_pAbstractedPadHolder->GetXcdDeviceHandle(&xcdHandle) == false)
    {
        return;
    }

    // ステートの更新
    m_Stage = currentStage;
    m_State = AnalogStickState();

    ::nn::xcd::CalibrationUpdateStatus status;
    // 更新状況を取得
    switch (m_Position)
    {
    case system::AnalogStickPosition_Left:
        ::nn::xcd::GetLeftAnalogStickValidRangeUpdateStatus(&status, xcdHandle);
        break;
    case system::AnalogStickPosition_Right:
        ::nn::xcd::GetRightAnalogStickValidRangeUpdateStatus(&status, xcdHandle);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    // 更新が行われていれば完了へ
    if (status == ::nn::xcd::CalibrationUpdateStatus_None)
    {
        m_Stage = (currentStage == system::AnalogStickManualCalibrationStage_Update) ?
            system::AnalogStickManualCalibrationStage_Completed :
            system::AnalogStickManualCalibrationStage_ClearCompleted;
    }
}

void UniquePadAnalogStickDriver::InitializeAnalogStickManualCalibration() NN_NOEXCEPT
{
    m_Stage = system::AnalogStickManualCalibrationStage_Completed;
}

::nn::Result UniquePadAnalogStickDriver::IsAnalogStickCalibrationSupportedImpl() const NN_NOEXCEPT
{
    IAbstractedPad* pPad;
    NN_RESULT_THROW_UNLESS(m_pAbstractedPadHolder->GetIAbstractedPad(&pPad), system::ResultUniquePadDisconnected());
    NN_RESULT_THROW_UNLESS(pPad->GetFeatureSet().Test<AbstractedPadFeature::AnalogStickUserCal>(),
        system::ResultAnalogStickManualCalibrationNotSupported());
    NN_RESULT_SUCCESS;
}

::nn::Result UniquePadAnalogStickDriver::ResetAnalogStickManualCalibrationImpl() NN_NOEXCEPT
{
    NN_RESULT_DO(this->IsAnalogStickCalibrationSupportedImpl());

    nn::xcd::DeviceHandle xcdHandle;
    NN_RESULT_THROW_UNLESS(
        m_pAbstractedPadHolder->GetXcdDeviceHandle(&xcdHandle),
        system::ResultAnalogStickManualCalibrationNotSupported());

    ::nn::xcd::DeviceInfo deviceInfo;
    if (::nn::xcd::GetDeviceInfo(&deviceInfo, xcdHandle).IsFailure())
    {
        NN_RESULT_THROW(system::ResultAnalogStickDeviceNotConnected());
    }

    switch (m_Position)
    {
    case system::AnalogStickPosition_Left:
        if (deviceInfo.deviceType == ::nn::xcd::DeviceType_Right)
        {
            NN_RESULT_THROW(system::ResultAnalogStickDeviceNotConnected());
        }
        ::nn::xcd::ResetLeftAnalogStickValidRange(xcdHandle);
        break;
    case system::AnalogStickPosition_Right:
        if (deviceInfo.deviceType == ::nn::xcd::DeviceType_Left)
        {
            NN_RESULT_THROW(system::ResultAnalogStickDeviceNotConnected());
        }
        ::nn::xcd::ResetRightAnalogStickValidRange(xcdHandle);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
