﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/hid/hid_Gesture.h>
#include <nn/hid/hid_TouchScreen.h>
#include <nn/hid/debug/hid_TouchScreen.h>
#include <nn/util/util_BitFlagSet.h>

#include "hid_ActivationCount.h"
#include "hid_AppletResourceManager.h"
#include "hid_GestureRecognizer.h"
#include "hid_HandheldManager.h"
#include "hid_InputDetectorManager.h"
#include "hid_ITouchScreenDriver.h"
#include "hid_SharedMemoryFormat.h"

namespace nn { namespace hid { namespace detail {

//!< Gesture のコンテキストの制御フラグ定義です。
struct GestureContextFlag final
{
    typedef ::nn::util::BitFlagSet<32, GestureContextFlag>::Flag<0>
            IsAvailable;    //!< 利用可能か否か
    typedef ::nn::util::BitFlagSet<32, GestureContextFlag>::Flag<1>
            IsLocked;       //!< ロック中か否か
};

//!< Gesture のコンテキストの制御フラグ集合を扱う型です。
typedef ::nn::util::BitFlagSet<32, GestureContextFlag> GestureContextFlagSet;

//!< Gesture のコンテキストを表す構造体です。
struct GestureContext final
{
    GestureContextFlagSet flags;    //!< 制御フラグ
    int32_t version;                //!< ジェスチャ認識器のバージョン
    int64_t eventNumber;            //!< イベント番号
    int64_t contextNumber;          //!< コンテキスト番号
    int64_t lastEventNumber;        //!< 最後に観測されたイベント番号
    int64_t lastContextNumber;      //!< 最後に観測されたコンテキスト番号
    int32_t lastGestureType;        //!< 最後に観測されたジェスチャの種類
    NN_PADDING4;
};

//!< TouchScreen のコンテキストを表す構造体です。
struct TouchScreenContext final
{
    int32_t count;                      //!< タッチの識別子の数
    NN_PADDING4;
    int32_t ids[TouchStateCountMax];    //!< タッチの識別子
};

//!< アプレットリソースの TouchScreen 用拡張エントリを表す構造体です。
struct TouchScreenAppletResourceEntry final
{
    ::nn::applet::AppletResourceUserId aruid;   //!< ARUID
    GestureContext gestureContext;              //!< Gesture のコンテキスト
    TouchScreenContext touchScreenContext;      //!< TouchScreen のコンテキスト
};

//!< 一時的メモリ領域を表す構造体です。
struct TouchScreenTemp final
{
    //!< ARUID
    ::nn::applet::AppletResourceUserId aruid;

    //!< ジェスチャ認識器のバージョン
    int32_t version;

    NN_PADDING4;

    //!< Gesture の入力状態
    GestureState gestureState;

    //!< TouchScreen の入力状態
    TouchScreenState<TouchStateCountMax> touchScreenState;
};

//!< TouchScreen インターフェイスの解決を担うマネージャを扱うクラスです。
class TouchScreenManager final
{
    NN_DISALLOW_COPY(TouchScreenManager);
    NN_DISALLOW_MOVE(TouchScreenManager);

public:
    //!< サンプリング間隔
    static const ::nn::TimeSpan SamplingInterval;

private:
    //!< 共有リソースについてアクティブ化された回数
    ActivationCount m_CommonActivationCount;

    //!< Gesture がアクティブ化された回数
    ActivationCount m_GestureActivationCount;

    //!< TouchScreen がアクティブ化された回数
    ActivationCount m_TouchScreenActivationCount;

    //!< 起床状態にあるか否か
    bool m_IsAwake;

    //!< サンプリング番号
    int64_t m_SamplingNumber;

    //!< タイマーイベント
    ::nn::os::TimerEventType* m_pTimerEvent;

    //!< ドライバ
    ITouchScreenDriver* m_pDriver;

    //!< アプレットリソースマネージャ
    AppletResourceManager* m_pAppletResourceManager;

    //!< アプレットリソースマネージャのミューテックス
    ::nn::os::SdkRecursiveMutex* m_pAppletResourceManagerMutex;

    //!< Handheld マネージャ
    HandheldManager* m_pHandheldManager;

    //!< InputDetector マネージャ
    InputDetectorManager* m_pInputDetectorManager;

    //!< InputDetector マネージャのミューテックス
    ::nn::os::SdkMutex* m_pInputDetectorManagerMutex;

    //!< 入力状態
    TouchScreenState<TouchStateCountMax> m_State;

    //!< 自動操作が有効か否かを表す値
    bool m_IsAutoPilotEnabled;

    //!< 自動操作状態
    ::nn::hid::debug::TouchScreenAutoPilotState<TouchStateCountMax
        > m_AutoPilotState;

    //!< ジェスチャ認識器
    GestureRecognizer m_GestureRecognizer;

    //!< アプレットリソースの TouchScreen 用拡張エントリ
    TouchScreenAppletResourceEntry
        m_AppletResourceEntries[AppletResourceEntryCountMax];

    //!< 一時的メモリ領域
    TouchScreenTemp m_Temp;

public:
    TouchScreenManager() NN_NOEXCEPT;

    //!< タイマーイベントを設定します。
    void SetTimerEvent(::nn::os::TimerEventType* pTimerEvent) NN_NOEXCEPT;

    //!< ドライバを設定します。
    void SetDriver(ITouchScreenDriver* pDriver) NN_NOEXCEPT;

    //!< アプレットリソースマネージャを設定します。
    void SetAppletResourceManager(
        AppletResourceManager* pManager, ::nn::os::SdkRecursiveMutex* pMutex
        ) NN_NOEXCEPT;

    //!< Handheld マネージャを設定します。
    void SetHandheldManager(HandheldManager* pManager) NN_NOEXCEPT;

    //!< InputDetector マネージャを設定します。
    void SetInputDetectorManager(
        InputDetectorManager* pManager, ::nn::os::SdkMutex* pMutex) NN_NOEXCEPT;

    //!< マネージャを Gesture 向けにアクティブ化します。
    ::nn::Result ActivateForGesture() NN_NOEXCEPT;

    //!< マネージャを Gesture 向けに非アクティブ化します。
    ::nn::Result DeactivateForGesture() NN_NOEXCEPT;

    //!< アプレットリソースの状態を Gesture 向けに保証します。
    ::nn::Result EnsureAppletResourceForGesture(
        ::nn::applet::AppletResourceUserId aruid, int32_t version) NN_NOEXCEPT;

    //!< マネージャを TouchScreen 向けにアクティブ化します。
    ::nn::Result ActivateForTouchScreen() NN_NOEXCEPT;

    //!< マネージャを TouchScreen 向けに非アクティブ化します。
    ::nn::Result DeactivateForTouchScreen() NN_NOEXCEPT;

    //!< アプレットリソースの状態を TouchScreen 向けに保証します。
    ::nn::Result EnsureAppletResourceForTouchScreen(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 自動操作状態を設定します。
    ::nn::Result SetAutoPilotState(
        const ::nn::hid::debug::TouchScreenAutoPilotState<TouchStateCountMax
            >& value) NN_NOEXCEPT;

    //!< 自動操作状態を解除します。
    ::nn::Result UnsetAutoPilotState() NN_NOEXCEPT;

    //!< 入力状態を解決します。
    void Sample() NN_NOEXCEPT;

    //!< デバイスを起床状態に遷移させます。
    ::nn::Result WakeDeviceUp() NN_NOEXCEPT;

    //!< デバイスをスリープ状態に遷移させます。
    ::nn::Result PutDeviceToSleep() NN_NOEXCEPT;

private:
    //!< 入力状態を更新します。
    void Update() NN_NOEXCEPT;

    //!< 共有メモリを処理します。
    void ProcessSharedMemory(
        void (*processor)(
            TouchScreenManager* that,
            GestureSharedMemoryFormat* gestureAddress,
            GestureContext* pGestureContext,
            TouchScreenSharedMemoryFormat* touchScreenAddress,
            TouchScreenContext* pTouchScreenContext,
            ::nn::applet::AppletResourceUserId aruid,
            bool enablesInput,
            bool isGestureActivated,
            bool isTouchScreenActivated) NN_NOEXCEPT) NN_NOEXCEPT;

    //!< 自動操作を無効化します。
    void DisableAutoPilot() NN_NOEXCEPT;

    //!< 共有リソースをアクティブ化します。
    ::nn::Result ActivateCommon() NN_NOEXCEPT;

    //!< 共有リソースをアクティブ化します。
    ::nn::Result DeactivateCommon() NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
