﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/erpt.h>
#include <nn/err/detail/err_ErrorCodeConvert.h>
#include <nn/fs.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/hid_TouchScreen.h>
#include <nn/hid/detail/hid_Log.h>
#include <nn/os/os_Tick.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nnd/ftm/ftm.h>

#include "hid_TouchScreenDriver-os.horizon.ftm.h"

//!< システム設定の情報ログを出力します。
#define NN_HID_TOUCHSCREEN_INFO(...) \
    NN_DETAIL_HID_INFO("[hid::TouchScreen] Information: " __VA_ARGS__)

//!< システム設定の警告ログを出力します。
#define NN_HID_TOUCHSCREEN_WARN(...) \
    NN_DETAIL_HID_WARN("[hid::TouchScreen] Warning: " __VA_ARGS__)

//!< 失敗時にシステム設定の警告ログを出力します。
#define NN_HID_TOUCHSCREEN_WARN_UNLESS_RESULT_SUCCESS(expression) \
    do \
    { \
        auto _nn_detail_hid_result = (expression); \
        if ( _nn_detail_hid_result.IsFailure() ) \
        { \
            NN_HID_TOUCHSCREEN_WARN( \
                    "Failed: %s\n  Module: %d\n  Description: %d\n  InnerValue: 0x%08x\n", \
                NN_MACRO_STRINGIZE(expression), \
                _nn_detail_hid_result.GetModule(), \
                _nn_detail_hid_result.GetDescription(), \
                _nn_detail_hid_result.GetInnerValueForDebug()); \
        } \
    } while ( NN_STATIC_CONDITION(false) )

namespace nn { namespace hid { namespace detail {

namespace {

//!< デバイスをセンシング有効状態に遷移させます。
::nn::Result ActivateDeviceSensing(bool resets) NN_NOEXCEPT;

//!< タッチの識別子が有効か否かを表す値を返します。
bool IsValidTouchId(size_t touchId) NN_NOEXCEPT;

//!< タッチに関するイベントからタッチの状態を作成します。
TouchState MakeTouchState(
    const ::nnd::ftm::TouchEventReport& report) NN_NOEXCEPT;

//!< EventReport を読み込みます。
::nn::Result ReadEventReports(
    bool* pOutIsOverflow, int* pOutReadCount, char* pOutReadData, int readCount
    ) NN_NOEXCEPT;

//!< 識別子の変換表を扱うクラスです。
class TouchIdMapAccessor final
{
    NN_DISALLOW_COPY(TouchIdMapAccessor);
    NN_DISALLOW_MOVE(TouchIdMapAccessor);

private:
    //!< 不正な識別子
    static const size_t InvalidTouchId = static_cast<size_t>(
        TouchStateCountMax);

    //!< 禁止された識別子
    bool m_ForbiddenMap[TouchStateCountMax];

    //!< 識別子の変換表
    size_t (&m_TouchIdMap)[TouchStateCountMax];

public:
    explicit TouchIdMapAccessor(
        size_t (&touchIdMap)[TouchStateCountMax]) NN_NOEXCEPT;

    //!< 指定された識別子の使用を禁止します。
    void Forbid(size_t touchId) NN_NOEXCEPT;

    //!< 識別子の変換表をリセットします。
    void Reset() NN_NOEXCEPT;

    //!< 識別子を変換します。
    size_t Convert(size_t touchId) const NN_NOEXCEPT;

    //!< 識別子を登録します。
    size_t Register(size_t touchId) NN_NOEXCEPT;

    //!< 識別子の登録を解除します。
    size_t Unregister(size_t touchId) NN_NOEXCEPT;
};

//!< ファームウェアのアップデータを扱うクラスです。
class TouchFirmwareUpdater final
{
    NN_DISALLOW_COPY(TouchFirmwareUpdater);
    NN_DISALLOW_MOVE(TouchFirmwareUpdater);

private:
    //!< スクリーンの種類を表す列挙型です
    enum class ScreenType : uint8_t
    {
        Nissa,   //!< 日写製スクリーン
        Gis,     //!< GIS 製スクリーン
        Gis2,    //!< GIS 製 VA スクリーン
        Unknown, //!< 不明なスクリーン
    };

    //!< LCD の種類を表す列挙型です。
    enum class LcdType : uint8_t
    {
        JdiAmorphous, //!< JDI 製 amorphous LCD
        JdiLtps,      //!< JDI 製 LTPS LCD
        Innolux,      //!< Innolux 製 LCD
        Unknown,      //!< 不明な LCD
    };

    //!< モジュールとファームウェアの組み合わせを表す構造体です。
    struct FirmwareCombination final
    {
        //!< スクリーンの種類
        ScreenType screenType;

        //!< LCD の種類
        LcdType lcdType;

        //!< ファームウェアのリリースバージョン
        int version;
    };

    //!< ファームウェアファイルの名前とバージョンの対応を表す構造体です。
    struct FirmwareFileInfo final
    {
        //!< ファイル名
        const char* name;

        //!< ファームウェアのリリースバージョン
        int version;
    };

    //!< 共通ファームウェアバージョン
    static const int CommonFirmwareVersion = 0x00120100;

    //!< 現在のファームウェアバージョン
    int m_CurrentVersion;

    //!< 更新可能なファームウェアバージョン
    int m_UpdatedVersion;

    //!< ファームウェア壊れが発生していたか否か
    bool m_IsFirmwareBroken;

public:
    TouchFirmwareUpdater() NN_NOEXCEPT;

    //!< アップデータをアクティブ化します。
    ::nn::Result Activate() NN_NOEXCEPT;

    //!< アップデータを非アクティブ化します。
    ::nn::Result Deactivate() NN_NOEXCEPT;

    //!< 更新が必要か否かを返します。
    bool IsUpdateRequired() const NN_NOEXCEPT;

    //!< アップデートを実行します。
    ::nn::Result Proceed() const NN_NOEXCEPT;

    //!< 現在のファームウェアバージョンを読み込みます。
    ::nn::Result GetCurrentVersion(int* pOutVersion) const NN_NOEXCEPT;

private:
    //!< 更新可能なファームウェアバージョンを選択します。
    ::nn::Result GetUpdatedVersion(int* pOutVersion) const NN_NOEXCEPT;

    //!< タッチスクリーンの種類を取得します。
    ::nn::Result GetScreenType(ScreenType* pOutScreen) const NN_NOEXCEPT;

    //!< LCD の種類を取得します。
    ::nn::Result GetLcdType(LcdType* pOutLcd) const NN_NOEXCEPT;

    //!< ファームウェアデータをファイルから読み込みます。
    static ::nn::Result ReadFirmwareFile(
        char* buffer, void* pParameter, int64_t offset, size_t size) NN_NOEXCEPT;

    //!< エラーレポートを作成します。
    void CreateErrorReport(::nn::Result result) const NN_NOEXCEPT;
};

} // namespace

TouchScreenDriver::TouchScreenDriver() NN_NOEXCEPT
    : m_ActivationCount()
    , m_IsDeviceAvailable(false)
    , m_IsAwake(false)
    , m_SamplingNumber(0)
    , m_FingerId(0)
{
    TouchIdMapAccessor(m_TouchIdMap).Reset();
}

TouchScreenDriver::~TouchScreenDriver() NN_NOEXCEPT
{
    // 何もしない
}

::nn::Result TouchScreenDriver::Activate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultTouchScreenDriverActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施

        auto needsRollback = true;

        // ライブラリを初期化
        ::nnd::ftm::Initialize();

        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                ::nnd::ftm::Finalize();
            }
        };

        // FW の更新があった場合にアップデートを実施
        TouchFirmwareUpdater updater;
        if (updater.Activate().IsSuccess())
        {
            NN_UTIL_SCOPE_EXIT
            {
                updater.Deactivate();
            };

            if (updater.IsUpdateRequired())
            {
                auto result = updater.Proceed();
                if (result.IsFailure())
                {
                    NN_HID_TOUCHSCREEN_WARN(
                        "TouchFirmwareUpdater::Proceed() failed. (%08x, %03d-%04d)\n",
                        result.GetInnerValueForDebug(),
                        result.GetModule(), result.GetDescription());
                }
                else
                {
                    NN_HID_TOUCHSCREEN_INFO("Firmware update is completed.\n");
                }
            }

            int version;
            if(updater.GetCurrentVersion(&version).IsSuccess())
            {
                NN_HID_TOUCHSCREEN_INFO("Firmware Version %X\n", version);
            }
        }

        if (ActivateDeviceSensing(true).IsSuccess())
        {
            m_IsDeviceInitialized = true;

            m_IsDeviceAvailable = true;

            needsRollback = false;
        }

        m_IsAwake = true;
    }

    // このインスタンスからアクティブ化した回数をインクリメント
    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result TouchScreenDriver::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultTouchScreenDriverDeactivationLowerLimitOver());

    // このインスタンスからアクティブ化した回数をデクリメント
    --m_ActivationCount;

    if (m_ActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施

        // タッチの識別子をリセット
        m_FingerId = 0;

        // 識別子を破棄
        m_TouchIds.Clear();

        if (m_IsDeviceInitialized)
        {
            // デバイスをリセット
            ::nnd::ftm::ResetDevice();

            // ライブラリの終了処理を実行
            ::nnd::ftm::Finalize();
        }

        m_IsAwake = false;

        m_IsDeviceAvailable = false;

        m_IsDeviceInitialized = false;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result TouchScreenDriver::Wake() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());

    if (m_IsDeviceInitialized && !m_IsAwake)
    {
        m_IsDeviceAvailable =
            ActivateDeviceSensing(!m_IsDeviceAvailable).IsSuccess();
    }

    m_IsAwake = true;

    NN_RESULT_SUCCESS;
}

::nn::Result TouchScreenDriver::Sleep() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());

    if (m_IsDeviceInitialized && m_IsAwake)
    {
        auto result = ::nn::Result();

        if (m_IsDeviceAvailable)
        {
            // デバイスの起床状態を解除
            result = ::nnd::ftm::SleepInDevice();

            if (result.IsFailure())
            {
                NN_HID_TOUCHSCREEN_WARN(
                    "nnd::ftm::SleepInDevice() failed. (%08x, %03d-%04d)\n",
                    result.GetInnerValueForDebug(),
                    result.GetModule(), result.GetDescription());

                m_IsDeviceAvailable = false;
            }
        }

        if (!m_IsDeviceAvailable)
        {
            // デバイスをリセット
            result = ::nnd::ftm::ResetDevice();

            if (result.IsSuccess())
            {
                m_IsDeviceAvailable = true;
            }
            else
            {
                NN_HID_TOUCHSCREEN_WARN(
                    "nnd::ftm::ResetDevice() failed. (%08x, %03d-%04d)\n",
                    result.GetInnerValueForDebug(),
                    result.GetModule(), result.GetDescription());
            }
        }
    }

    m_IsAwake = false;

    NN_RESULT_SUCCESS;
}

void TouchScreenDriver::GetState(
    TouchScreenState<TouchStateCountMax>* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());

    // タイムスタンプを取得
    const auto time = ::nn::os::ConvertToTimeSpan(::nn::os::GetSystemTick());

    // イベントレポートを取得
    auto succeeds = false;
    auto isOverflow = false;
    int readCount = 0;
    if (m_IsDeviceAvailable && m_IsAwake)
    {
        if (ReadEventReports(&isOverflow, &readCount, m_ReadData,
                             ::nnd::ftm::GetMaxEventReportCount()).IsSuccess())
        {
            succeeds = true;
        }
        else
        {
            m_IsDeviceAvailable = false;
        }
    }

    // 作業領域を初期化
    m_NewIds.Clear();
    m_OldIds.Clear();

    // イベントレポートを処理
    if (!succeeds || isOverflow)
    {
        // イベントレポートの取りこぼしが発生した場合はタッチを全て破棄

        TouchIdMapAccessor(m_TouchIdMap).Reset();

        m_TouchIds.Clear();
    }
    else
    {
        // イベントレポートをパース
        ::nnd::ftm::ParseEventReports(m_Reports, m_ReadData, readCount);

        // イベントレポートを処理
        this->ProcessReports(m_Reports, readCount);
    }

    // 既存のタッチの経過時間を処理
    const ::nn::TimeSpanType deltaTime = time - m_LastTime;
    for (size_t i = 0; i < m_TouchIds.GetCount(); ++i)
    {
        m_Touches[m_TouchIds[i]].deltaTime = deltaTime;
    }

    // 新たな識別子を処理
    for (size_t i = 0; i < m_NewIds.GetCount(); ++i)
    {
        m_TouchIds.Append(m_NewIds[i]);
        m_Touches[m_NewIds[i]].deltaTime = ::nn::TimeSpanType();
    }

    // タッチスクリーンの入力状態を決定
    *pOutValue = TouchScreenState<TouchStateCountMax>();
    pOutValue->samplingNumber = m_SamplingNumber++;
    pOutValue->count = static_cast<int32_t>(m_TouchIds.GetCount());
    for (int32_t i = 0; i < pOutValue->count; ++i)
    {
        pOutValue->touches[i] = m_Touches[m_TouchIds[i]];
    }

    // タッチの属性をリセット
    for (size_t i = 0; i < m_TouchIds.GetCount(); ++i)
    {
        m_Touches[m_TouchIds[i]].attributes.Reset();
    }

    // 全てのタッチが開放された場合はタッチの識別子をリセット
    if (m_TouchIds.GetCount() == 0)
    {
        m_FingerId = 0;
    }

    // 去った識別子を処理
    for (size_t i = 0; i < m_OldIds.GetCount(); ++i)
    {
        m_TouchIds.Remove(m_OldIds[i]);
    }

    m_LastTime = time;
}

void TouchScreenDriver::DeassertReset() NN_NOEXCEPT
{
    ::nnd::ftm::DeassertReset();
}

void TouchScreenDriver::ProcessReports(
    const ::nnd::ftm::EventReport reports[], int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(reports);

    for (int i = 0; i < count; ++i)
    {
        switch (reports[i].eventId)
        {
        case ::nnd::ftm::EventId::TouchEnter:
            this->ProcessTouchEnterEvent(reports[i].content.touchReport);
            break;

        case ::nnd::ftm::EventId::TouchLeave:
            this->ProcessTouchLeaveEvent(reports[i].content.touchReport);
            break;

        case ::nnd::ftm::EventId::TouchMotion:
            this->ProcessTouchMotionEvent(reports[i].content.touchReport);
            break;

        default:
            break;
        }
    }
}

void TouchScreenDriver::ProcessTouchEnterEvent(
    const ::nnd::ftm::TouchEventReport& report) NN_NOEXCEPT
{
    TouchIdMapAccessor accessor(m_TouchIdMap);

    for (size_t i = 0; i < m_NewIds.GetCount(); ++i)
    {
        accessor.Forbid(m_NewIds[i]);
    }

    for (size_t i = 0; i < m_TouchIds.GetCount(); ++i)
    {
        accessor.Forbid(m_TouchIds[i]);
    }

    const size_t touchId = accessor.Register(report.touchId);

    if (IsValidTouchId(touchId))
    {
        // タッチの状態を更新
        TouchState& touch = m_Touches[touchId];
        touch = MakeTouchState(report);
        touch.attributes.Set<TouchAttribute::Start>();
        touch.fingerId = m_FingerId++;

        // 新たな識別子に追加
        m_NewIds.Append(touchId);
    }
}

void TouchScreenDriver::ProcessTouchLeaveEvent(
    const ::nnd::ftm::TouchEventReport& report) NN_NOEXCEPT
{
    const size_t touchId =
        TouchIdMapAccessor(m_TouchIdMap).Unregister(report.touchId);

    if (IsValidTouchId(touchId))
    {
        if (m_NewIds.Contains(touchId))
        {
            m_NewIds.Remove(touchId);
        }
        else if (!m_OldIds.Contains(touchId))
        {
            // タッチの状態を更新
            TouchState& touch = m_Touches[touchId];
            int32_t fingerId = touch.fingerId;
            touch = MakeTouchState(report);
            touch.attributes.Set<TouchAttribute::End>();
            touch.fingerId = fingerId;

            // 去った識別子に追加
            m_OldIds.Append(touchId);
        }
    }
}

void TouchScreenDriver::ProcessTouchMotionEvent(
    const ::nnd::ftm::TouchEventReport& report) NN_NOEXCEPT
{
    const size_t touchId =
        TouchIdMapAccessor(m_TouchIdMap).Convert(report.touchId);

    if (IsValidTouchId(touchId))
    {
        // タッチの状態を更新
        TouchState& touch = m_Touches[touchId];
        int32_t fingerId = touch.fingerId;
        touch = MakeTouchState(report);
        touch.fingerId = fingerId;
    }
}

namespace {

::nn::Result ActivateDeviceSensing(bool resets) NN_NOEXCEPT
{
    ::nn::Result result = ::nn::ResultSuccess();

    if (resets)
    {
        // デバイスをリセット
        result = ::nnd::ftm::ResetDevice();
    }

    if (result.IsFailure())
    {
        NN_HID_TOUCHSCREEN_WARN(
            "nnd::ftm::ResetDevice() failed. (%08x, %03d-%04d)\n",
            result.GetInnerValueForDebug(),
            result.GetModule(), result.GetDescription());

        NN_RESULT_THROW(result);
    }
    else
    {
        // デバイスのスリープ状態を解除
        result = ::nnd::ftm::SleepOutDevice();

        if (result.IsFailure())
        {
            NN_HID_TOUCHSCREEN_WARN(
                "nnd::ftm::SleepOutDevice() failed. (%08x, %03d-%04d)\n",
                result.GetInnerValueForDebug(),
                result.GetModule(), result.GetDescription());

            NN_RESULT_THROW(result);
        }
        else
        {
            // デバイスのセンシングをアクティブ化
            result = ::nnd::ftm::ActivateSensing();

            if (result.IsFailure())
            {
                NN_HID_TOUCHSCREEN_WARN(
                    "nnd::ftm::ActivateSensing() failed. (%08x, %03d-%04d)\n",
                    result.GetInnerValueForDebug(),
                    result.GetModule(), result.GetDescription());

                const ::nn::Result lastResult = result;

                result = ::nnd::ftm::ResetDevice();

                if (result.IsFailure())
                {
                    NN_HID_TOUCHSCREEN_WARN(
                        "nnd::ftm::ResetDevice() failed. (%08x, %03d-%04d)\n",
                        result.GetInnerValueForDebug(),
                        result.GetModule(), result.GetDescription());

                    NN_RESULT_THROW(result);
                }

                NN_RESULT_THROW(lastResult);
            }
        }
    }

    NN_RESULT_SUCCESS;
}

bool IsValidTouchId(size_t touchId) NN_NOEXCEPT
{
    return (touchId < TouchStateCountMax);
}

TouchState MakeTouchState(
    const ::nnd::ftm::TouchEventReport& report) NN_NOEXCEPT
{
    TouchState state = {};
    state.x = report.x;
    state.y = report.y;
    state.diameterX = report.minor;
    state.diameterY = report.major;
    state.rotationAngle = report.milliDegree / 1000;
    return state;
}

::nn::Result ReadEventReports(
    bool* pOutIsOverflow, int* pOutReadCount, char* pOutReadData, int readCount
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutIsOverflow);
    NN_SDK_REQUIRES_NOT_NULL(pOutReadCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutReadData);
    NN_SDK_REQUIRES_RANGE(
        readCount, 1, ::nnd::ftm::GetMaxEventReportCount() + 1);

    auto result = ::nn::Result();
    int leftCount = 0;
    result = ::nnd::ftm::ReadLeftEventCount(&leftCount);
    if (result.IsFailure())
    {
        NN_HID_TOUCHSCREEN_WARN(
            "nnd::ftm::ReadLeftEventCount() failed. (%08x, %03d-%04d)\n",
            result.GetInnerValueForDebug(),
            result.GetModule(), result.GetDescription());

        NN_RESULT_THROW(result);
    }

    if (leftCount > 0)
    {
        result = ::nnd::ftm::ReadEventReports(
            pOutReadData,
            pOutReadCount,
            pOutIsOverflow,
            ::std::min(leftCount, readCount));

        if (result.IsFailure())
        {
            NN_HID_TOUCHSCREEN_WARN(
                "nnd::ftm::ReadEventReports() failed. (%08x, %03d-%04d)\n",
                result.GetInnerValueForDebug(),
                result.GetModule(), result.GetDescription());

            NN_RESULT_THROW(result);
        }
    }

    NN_RESULT_SUCCESS;
}

TouchIdMapAccessor::TouchIdMapAccessor(
    size_t (&touchIdMap)[TouchStateCountMax]) NN_NOEXCEPT
    : m_TouchIdMap(touchIdMap)
{
    for (bool& forbidden : m_ForbiddenMap)
    {
        forbidden = false;
    }
}

void TouchIdMapAccessor::Forbid(size_t touchId) NN_NOEXCEPT
{
    if (IsValidTouchId(touchId))
    {
        m_ForbiddenMap[touchId] = true;
    }
}

void TouchIdMapAccessor::Reset() NN_NOEXCEPT
{
    for (size_t& touchId : m_TouchIdMap)
    {
        touchId = InvalidTouchId;
    }
}

size_t TouchIdMapAccessor::Convert(size_t touchId) const NN_NOEXCEPT
{
    return IsValidTouchId(touchId) ? m_TouchIdMap[touchId]
                                   : InvalidTouchId;
}

size_t TouchIdMapAccessor::Register(size_t touchId) NN_NOEXCEPT
{
    if (IsValidTouchId(touchId) && !IsValidTouchId(m_TouchIdMap[touchId]))
    {
        for (size_t i = 0; i < TouchStateCountMax; ++i)
        {
            if (m_ForbiddenMap[i])
            {
                continue;
            }

            m_TouchIdMap[touchId] = i;

            return i;
        }
    }

    return InvalidTouchId;
}

size_t TouchIdMapAccessor::Unregister(size_t touchId) NN_NOEXCEPT
{
    if (IsValidTouchId(touchId))
    {
        size_t value = m_TouchIdMap[touchId];

        m_TouchIdMap[touchId] = InvalidTouchId;

        return value;
    }

    return InvalidTouchId;
}


TouchFirmwareUpdater::TouchFirmwareUpdater() NN_NOEXCEPT
    : m_CurrentVersion(0)
    , m_UpdatedVersion(0)
    , m_IsFirmwareBroken(false)
{
}

::nn::Result TouchFirmwareUpdater::Activate() NN_NOEXCEPT
{
    if (::nnd::ftm::ResetDevice().IsFailure())
    {
        // リセットが失敗する場合はファームウェア破壊が考えられる
        m_IsFirmwareBroken = true;

        // ファームウェア破壊情報が入ったエラーレポートを作成する
        this->CreateErrorReport(
                ResultTouchScreenDriverDetectFirmwareCorruption());

        // 共通ファームウェアを書き込んだ後にバージョン取得を試みる
        m_UpdatedVersion = CommonFirmwareVersion;
        NN_RESULT_DO(this->Proceed());
    }

    NN_RESULT_DO(this->GetCurrentVersion(&m_CurrentVersion));

    NN_RESULT_DO(this->GetUpdatedVersion(&m_UpdatedVersion));

    NN_RESULT_SUCCESS;
}

::nn::Result TouchFirmwareUpdater::Deactivate() NN_NOEXCEPT
{
    m_CurrentVersion = 0;
    m_UpdatedVersion = 0;
    m_IsFirmwareBroken = false;

    NN_RESULT_SUCCESS;
}

bool TouchFirmwareUpdater::IsUpdateRequired() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(m_CurrentVersion, 0);
    NN_SDK_REQUIRES_NOT_EQUAL(m_UpdatedVersion, 0);

    return m_CurrentVersion != m_UpdatedVersion;
}

::nn::Result TouchFirmwareUpdater::Proceed() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(m_UpdatedVersion, 0);

    auto result = ::nn::Result();

    const FirmwareFileInfo infos[] =
    {
        {"NTD_4CD_1601.fts256", 0x00100100},
        {"NTD_4CD_1801.fts256", 0x00120100},
        {"NTD_4CD_2602.fts256", 0x001A0300},
        {"NTD_4CD_3801.fts256", 0x00290100},
    };

    // 読み込むファイル名を決定
    const int FileNameLengthMax = 64;
    char fileNameBuffer[FileNameLengthMax];
    for (const auto& info : infos)
    {
        if (m_UpdatedVersion == info.version)
        {
            ::nn::util::SNPrintf(
                fileNameBuffer, FileNameLengthMax, "rom:/ftmFwUpdate/%s", info.name);
            break;
        }
    }

    ::nn::fs::FileHandle fileHandle;
    NN_RESULT_DO(::nn::fs::OpenFile(&fileHandle, fileNameBuffer, ::nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        ::nn::fs::CloseFile(fileHandle);
    };

    int64_t fileSize;
    NN_RESULT_DO(::nn::fs::GetFileSize(&fileSize, fileHandle));

    result = ::nnd::ftm::UpdateFirmware(
        ReadFirmwareFile, &fileHandle, static_cast<size_t>(fileSize));
    if (result.IsFailure())
    {
        NN_HID_TOUCHSCREEN_WARN(
            "nnd::ftm::UpdateFirmware() failed. (%08x, %03d-%04d)\n",
            result.GetInnerValueForDebug(),
            result.GetModule(), result.GetDescription());

        NN_RESULT_THROW(result);
    }

    // ファームウェア破損時は補償領域も破壊されてる可能性があるので AutoTune を行う
    if (m_IsFirmwareBroken)
    {
        result = ::nnd::ftm::RunAutoTune();
        if (result.IsFailure())
        {
            NN_HID_TOUCHSCREEN_WARN(
                    "nnd::ftm::RunAutoTune() failed. (%08x, %03d-%04d)\n",
                    result.GetInnerValueForDebug(),
                    result.GetModule(), result.GetDescription());

            NN_RESULT_THROW(result);
        }
    }

    NN_RESULT_SUCCESS;
}

::nn::Result TouchFirmwareUpdater::GetCurrentVersion(int* pOutVersion) const NN_NOEXCEPT
{
    auto result = ::nn::Result();

    ::nnd::ftm::FirmwareVersion version;
    result = ::nnd::ftm::ReadFirmwareVersion(&version);
    if (result.IsFailure())
    {
        NN_HID_TOUCHSCREEN_WARN(
            "nnd::ftm::ReadFirmwareVersion() failed. (%08x, %03d-%04d)\n",
            result.GetInnerValueForDebug(),
            result.GetModule(), result.GetDescription());

        NN_RESULT_THROW(result);
    }

    *pOutVersion = version.release;

    NN_RESULT_SUCCESS;
}

::nn::Result TouchFirmwareUpdater::GetUpdatedVersion(int* pOutVersion) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(m_CurrentVersion, 0);

    const FirmwareCombination combinations[] =
    {
        {ScreenType::Nissa, LcdType::JdiAmorphous, 0x00120100},
        {ScreenType::Nissa, LcdType::JdiLtps,      0x00120100},
        {ScreenType::Nissa, LcdType::Innolux,      0x00120100},
        {ScreenType::Gis,   LcdType::JdiAmorphous, 0x001A0300},
        {ScreenType::Gis,   LcdType::JdiLtps,      0x001A0300},
        {ScreenType::Gis,   LcdType::Innolux,      0x001A0300},
        {ScreenType::Gis2,  LcdType::JdiAmorphous, 0x00290100},
        {ScreenType::Gis2,  LcdType::JdiLtps,      0x00290100},
        {ScreenType::Gis2,  LcdType::Innolux,      0x00290100},
    };

    ScreenType screen;
    NN_RESULT_DO(this->GetScreenType(&screen));

    // スクリーンの種類が不明な場合は更新しない
    if (screen == ScreenType::Unknown)
    {
        *pOutVersion = m_CurrentVersion;
        NN_RESULT_SUCCESS;
    }

    LcdType lcd;
    NN_RESULT_DO(this->GetLcdType(&lcd));

    // LCD の種類が不明な場合は更新しない
    if (lcd == LcdType::Unknown)
    {
        *pOutVersion = m_CurrentVersion;
        NN_RESULT_SUCCESS;
    }

    // 更新可能なバージョンの決定
    for (const auto& combination : combinations)
    {
        if (screen == combination.screenType && lcd == combination.lcdType)
        {
            *pOutVersion = combination.version;
            NN_RESULT_SUCCESS;
        }
    }

    // 本来到達してはいけない
    // 更新先のバージョン指定漏れ時に不定にしないための対応
    NN_HID_TOUCHSCREEN_WARN(
        "Firmware version can be updated is undefined. (%d, %d)\n",
        static_cast<int>(screen), static_cast<int>(lcd));
    *pOutVersion = CommonFirmwareVersion;

    NN_RESULT_SUCCESS;
}

::nn::Result TouchFirmwareUpdater::GetScreenType(ScreenType* pOutScreen) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutScreen);

    auto result = ::nn::Result();

    ::nnd::ftm::GpioInformEventReport info;
    result = ::nnd::ftm::RequestGpioState(&info);
    if(result.IsFailure())
    {
        NN_HID_TOUCHSCREEN_WARN(
            "nnd::ftm::RequestGpioState() failed. (%08x, %03d-%04d)\n",
            result.GetInnerValueForDebug(),
            result.GetModule(), result.GetDescription());

        *pOutScreen = ScreenType::Unknown;
    }
    else
    {
        if (info.gpio0 == 1 && info.gpio1 == 1 && info.gpio2 == 1)
        {
            *pOutScreen = ScreenType::Nissa;
        }
        else if (info.gpio0 == 0 && info.gpio1 == 1 && info.gpio2 == 1)
        {
            *pOutScreen = ScreenType::Gis;
        }
        else if (info.gpio0 == 1 && info.gpio1 == 0 && info.gpio2 == 1)
        {
            *pOutScreen = ScreenType::Gis2;
        }
        else
        {
            *pOutScreen = ScreenType::Unknown;
        }
    }

    NN_RESULT_SUCCESS;
}

::nn::Result TouchFirmwareUpdater::GetLcdType(LcdType* pOutLcd) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutLcd);

    // TODO : LCD の種類取得
    *pOutLcd = LcdType::JdiAmorphous;

    NN_RESULT_SUCCESS;
}

::nn::Result TouchFirmwareUpdater::ReadFirmwareFile(
    char* buffer, void* pParameter, int64_t offset, size_t size) NN_NOEXCEPT
{

    ::nn::fs::FileHandle* pFileHandle = reinterpret_cast<::nn::fs::FileHandle*>(pParameter);
    NN_RESULT_DO(::nn::fs::ReadFile(*pFileHandle, offset, buffer, size));

    NN_RESULT_SUCCESS;
}

void TouchFirmwareUpdater::CreateErrorReport(::nn::Result result) const NN_NOEXCEPT
{
    auto errorCode =
        ::nn::err::detail::ConvertResultToErrorCode(result);
    char errorCodeString[::nn::err::ErrorCode::StringLengthMax];
    ::nn::util::SNPrintf(
            errorCodeString,
            sizeof(errorCodeString),
            "%04d-%04d",
            errorCode.category,
            errorCode.number);
    ::nn::erpt::Context context(::nn::erpt::CategoryId::ErrorInfo);
    NN_HID_TOUCHSCREEN_WARN_UNLESS_RESULT_SUCCESS(
            context.Add(
                ::nn::erpt::FieldId::ErrorCode,
                errorCodeString,
                static_cast<uint32_t>(sizeof(errorCodeString))));
    NN_HID_TOUCHSCREEN_WARN_UNLESS_RESULT_SUCCESS(
            context.CreateReport(
                ::nn::erpt::ReportType::ReportType_Invisible));
}

} // namespace

}}} // namespace nn::hid::detail
