﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/hid/debug/hid_SleepButton.h>
#include <nn/hid/system/hid_SleepButton.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_TimerEvent.h>

#include "hid_ActivationCount.h"
#include "hid_AppletResourceManager.h"
#include "hid_InputDetectorManager.h"
#include "hid_ISleepButtonDriver.h"
#include "hid_SharedMemoryFormat.h"
#include "hid_SystemButtonNotifier.h"
#include "hid_SynchronizedTimer.h"

namespace nn { namespace hid { namespace detail {

//!< スリープボタンインターフェイスの解決を担うマネージャを扱うクラスです。
class SleepButtonManager final
{
    NN_DISALLOW_COPY(SleepButtonManager);
    NN_DISALLOW_MOVE(SleepButtonManager);

public:
    //!< サンプリング間隔
    static const ::nn::TimeSpan SamplingInterval;

private:
    //!< スリープボタンが押下されているか否か
    ::std::atomic<bool> m_IsSleepButtonDown;

    //!< スリープボタンの入力通知が必要か否か
    ::std::atomic<bool> m_NeedsToSignal;

    //!< スリープボタン入力通知イベント
    ::nn::os::SystemEvent m_SleepButtonEvent;

    //!< アクティブ化された回数
    ActivationCount m_ActivationCount;

    //!< サンプリング番号
    int64_t m_SamplingNumber;

    //!< タイマーイベント
    SynchronizedTimer* m_pTimer;

    //!< ドライバ
    ISleepButtonDriver* m_pDriver;

    //!< アプレットリソースマネージャ
    AppletResourceManager* m_pAppletResourceManager;

    //!< アプレットリソースマネージャのミューテックス
    ::nn::os::SdkRecursiveMutex* m_pAppletResourceManagerMutex;

    //!< InputDetector マネージャ
    InputDetectorManager* m_pInputDetectorManager;

    //!< InputDetector マネージャのミューテックス
    ::nn::os::SdkMutex* m_pInputDetectorManagerMutex;

    //!< 入力状態
    ::nn::hid::system::SleepButtonState m_State;

    //!< 入力状態計算用の一時的メモリ領域
    ::nn::hid::system::SleepButtonState m_TempState;

    //!< ARUID 処理用の一時的メモリ領域
    ::nn::applet::AppletResourceUserId m_TempAruid;

    //!< ドライバによるスリープボタンの入力通知を行う Notifier
    SystemButtonNotifier m_DriverSleepButtonNotifier;

    //!< 自動操作によるスリープボタンの入力通知を行う Notifier
    SystemButtonNotifier m_AutoPilotSleepButtonNotifier;

public:
    SleepButtonManager() NN_NOEXCEPT;

    //!< タイマーイベントを設定します。
    void SetTimer(SynchronizedTimer* timer) NN_NOEXCEPT;

    //!< ドライバを設定します。
    void SetDriver(ISleepButtonDriver* pDriver) NN_NOEXCEPT;

    //!< アプレットリソースマネージャを設定します。
    void SetAppletResourceManager(
        AppletResourceManager* pManager, ::nn::os::SdkRecursiveMutex* pMutex
        ) NN_NOEXCEPT;

    //!< InputDetector マネージャを設定します。
    void SetInputDetectorManager(
        InputDetectorManager* pManager, ::nn::os::SdkMutex* pMutex) NN_NOEXCEPT;

    //!< スリープボタンの入力通知を行う Notifier を作成します。
    SystemButtonNotifier MakeSystemButtonNotifier() NN_NOEXCEPT;

    //!< スリープボタンの入力通知をバインドします。
    ::nn::Result AcquireSleepButtonEventHandle(
        ::nn::os::NativeHandle* pOutHandle,
        const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT;

    //!< スリープボタンの入力通知をシグナル状態にします。
    void SignalSleepButtonEvent() NN_NOEXCEPT;

    //!< マネージャをアクティブ化します。
    ::nn::Result Activate() NN_NOEXCEPT;

    //!< マネージャを非アクティブ化します。
    ::nn::Result Deactivate() NN_NOEXCEPT;

    //!< アプレットリソースの状態を保証します。
    ::nn::Result EnsureAppletResource(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 自動操作状態を設定します。
    ::nn::Result SetAutoPilotState(
        const ::nn::hid::debug::SleepButtonAutoPilotState& value) NN_NOEXCEPT;

    //!< 自動操作状態を解除します。
    ::nn::Result UnsetAutoPilotState() NN_NOEXCEPT;

    //!< 入力状態を解決します。
    void Sample() NN_NOEXCEPT;

private:
    //!< 入力状態を更新します。
    void Update() NN_NOEXCEPT;

    //!< 共有メモリを処理します。
    void ProcessSharedMemory(
        void (*processor)(SleepButtonManager* that,
                          SleepButtonSharedMemoryFormat* address,
                          ::nn::applet::AppletResourceUserId aruid,
                          bool enablesInput) NN_NOEXCEPT) NN_NOEXCEPT;

    //!< 自動操作を無効化します。
    void DisableAutoPilot() NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
