﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/hid_Xpad.h>

#include "hid_NpadStateUtility.h"
#include "hid_SixAxisSensorXcdDriver.h"

namespace {

void GetTypicalCalibrationValue(nn::xcd::SensorCalibrationValue* pOutCalibrationValue) NN_NOEXCEPT
{
    const nn::xcd::SensorCalibrationValue Lsm6ds3TypicalCalibrationValues = {
        { 0,0,0 },                  // accelerometerOrigin
        { 16384, 16384, 16384 },    // accelerometerSensitivity
        { 0,0,0 },                  // gyroscopeOrigin
        { 13371, 13371, 13371 }     // gyroscopeSensitivity
    };

    *pOutCalibrationValue = Lsm6ds3TypicalCalibrationValues; // Lsm6ds3 の typical 値を取得
}

void ConvertCalibrationValue(nn::xcd::SensorCalibrationValue* pOutValue, const nn::hid::system::DeviceTypeSet& deviceType) NN_NOEXCEPT
{
    // Note: 感度は実装面を考慮しない固定値が書かれているため Flip しない
    if (deviceType.Test<nn::hid::system::DeviceType::SwitchProController>())
    {
        ::nn::hid::detail::ConvertFullKeyAccAxis(&pOutValue->accelerometerOrigin);
        ::nn::hid::detail::ConvertFullKeyGyroAxis(&pOutValue->gyroscopeOrigin);
    }
    else if ((deviceType & nn::hid::detail::DeviceTypeMask_JoyLefts).IsAnyOn())
    {
        ::nn::hid::detail::ConvertLeftJoyAccAxis(&pOutValue->accelerometerOrigin);
        ::nn::hid::detail::ConvertLeftJoyGyroAxis(&pOutValue->gyroscopeOrigin);
    }
    else if ((deviceType & nn::hid::detail::DeviceTypeMask_JoyRights).IsAnyOn())
    {
        ::nn::hid::detail::ConvertRightJoyAccAxis(&pOutValue->accelerometerOrigin);
        ::nn::hid::detail::ConvertRightJoyGyroAxis(&pOutValue->gyroscopeOrigin);
    }
}

} // namespace

namespace nn { namespace hid { namespace detail {

Result SixAxisSensorXcdDriver::ActivateSixAxisSensor(SixAxisSensorXcdConfig config) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultGamePadDriverActivationUpperLimitOver()); // TORIAEZU

    // アクティブ化した回数をインクリメント
    ++m_ActivationCount;
    m_Config = config;

    // キャリブレーション処理
    nn::xcd::SensorCalibrationValue cal = {};
    if (m_Config.pPad->GetSensorCalibrationValue(&cal) == true)
    {
        // 取得したキャリブレーション値に対してデバイスの種類に応じた軸反転を行う
        ConvertCalibrationValue(&cal, m_Config.pPad->GetDeviceType());
    }
    else
    {
        GetTypicalCalibrationValue(&cal); // 失敗したときは Lsm6ds3 の Typical 値を用いる
    }

    m_SixAxisSensorProcessor.SetSensorCalibrationValue(cal);
    m_SixAxisSensorProcessor.SetDirection(nn::hid::detail::UnitDirection);
    const nn::util::Float3 InitialAngle = { { { 0.0f, 0.0f, 0.0f } } };
    m_SixAxisSensorProcessor.SetAngle(InitialAngle);

    NN_RESULT_SUCCESS;
}

Result SixAxisSensorXcdDriver::DeactivateSixAxisSensor() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultGamePadDriverDeactivationLowerLimitOver()); // TORIAEZU

    // アクティブ化した回数をデクリメント
    --m_ActivationCount;

    if(m_ActivationCount.IsZero())
    {
        // 何もしない
    }
    m_Config.pPad = nullptr;
    m_SixAxisSensorProcessor.Reset();

    NN_RESULT_SUCCESS;
}

void SixAxisSensorXcdDriver::Sample() NN_NOEXCEPT
{
    if (m_ActivationCount.IsZero() || m_Config.pPad == nullptr)
    {
        return;
    }

    ::nn::xcd::SixAxisSensorState states[nn::xcd::SixAxisSensorSampleCountMax];
    int gotCount = 0;
    gotCount = m_Config.pPad->GetSensorStates(states, NN_ARRAY_SIZE(states));
    auto deviceType = m_Config.pPad->GetDeviceType();
    for (int i = gotCount - 1; i >= 0; i--)
    {
        if (deviceType.Test<system::DeviceType::SwitchProController>())
        {
            ConvertFullKeyAxis(&states[i]);
        }
        else if ((deviceType & DeviceTypeMask_JoyLefts).IsAnyOn())
        {
            ConvertLeftJoyAxis(&states[i]);
        }
        else if ((deviceType & DeviceTypeMask_JoyRights).IsAnyOn())
        {
            ConvertRightJoyAxis(&states[i]);
        }
        m_SixAxisSensorProcessor.Sampling(states[i]);
    }
}

int SixAxisSensorXcdDriver::GetSixAxisSensorStates(SixAxisSensorState* outStates,
                                                   int count) const NN_NOEXCEPT
{
    return m_SixAxisSensorProcessor.GetSixAxisSensorStates(outStates, count);
}

void SixAxisSensorXcdDriver::SetSixAxisSensorSetting(const SixAxisSensorSetting* const pSetting) NN_NOEXCEPT
{
    m_SixAxisSensorProcessor.SetSixAxisSensorSetting(pSetting);
}

bool SixAxisSensorXcdDriver::IsSixAxisSensorAtRest() const NN_NOEXCEPT
{
    return m_SixAxisSensorProcessor.IsSixAxisSensorAtRest();
}

}}} // namespace nn::hid::detail
