﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       SixAxisSensor の処理に特化した API の宣言
 */

#pragma once

#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/util/util_MathTypes.h>
#include <nn/xcd/xcd_Input.h>

#include "hid_Accelerometer.h"
#include "hid_Gyroscope.h"
#include "hid_RingLifo.h"
#include "hid_SixAxisSensorLifo.h"

namespace nn { namespace hid { namespace detail {

//!< 基本姿勢
const nn::hid::DirectionState  UnitDirection = { {{{1.0f, 0.0f, 0.0f}}},
                                                 {{{0.0f, 1.0f, 0.0f}}},
                                                 {{{0.0f, 0.0f, 1.0f}}} };

//!< 6軸センサーの処理を担うクラス
class SixAxisSensorProcessor final
{
    NN_DISALLOW_COPY(SixAxisSensorProcessor);
    NN_DISALLOW_MOVE(SixAxisSensorProcessor);

private:

    SixAxisSensorLifo                       m_Lifo;                          //!< 加工後のセンサー値を格納するLIFO

    Accelerometer                           m_Accelerometer;                 //!< 加速度算出のためのクラス
    Gyroscope                               m_Gyroscope;                     //!< 角速度算出のためのクラス

    nn::util::Float3                        m_Angle;                         //!< 現在の回転角度[deg]
    nn::hid::DirectionState                 m_Direction;                     //!< 現在の姿勢行列
    nn::hid::DirectionState                 m_LastDirection;                 //!< 1サンプル前の姿勢行列
    int64_t                                 m_SamplingNumber;                //!< 現在のサンプリング番号
    float                                   m_SamplingIntervalSeconds;       //!< センサーのサンプリング周期[sec] (※ 通信周期ではない)
    nn::xcd::PadState                       m_LastPadState;                  //!< 前回サンプリングした PadState
    nn::xcd::SixAxisSensorState             m_LastSixAxisSensorState;        //!< 前回サンプリングした SixAxisSensorState

    const SixAxisSensorProcessorSetting*    m_pSixAxisSensorProcessorSetting;//!< 設定値

public:
    SixAxisSensorProcessor()  NN_NOEXCEPT;
    ~SixAxisSensorProcessor() NN_NOEXCEPT {};

    //!< 6軸センサーの加工処理クラスの内部ステートをクリアします。
    void Reset() NN_NOEXCEPT;

    //!< 6軸センサーのパラメータを設定します。
    void SetSixAxisSensorSetting(const SixAxisSensorSetting* const pSetting) NN_NOEXCEPT;

    //!< 6軸センサーのキャリブレーション値を設定します。
    void SetSensorCalibrationValue( const nn::xcd::SensorCalibrationValue& value ) NN_NOEXCEPT;

    //!< 現在の回転角度[deg]を設定します。
    void SetAngle( const  nn::util::Float3& angle ) NN_NOEXCEPT;

    //!< 現在の姿勢を設定します。
    void SetDirection( const nn::hid::DirectionState& direction ) NN_NOEXCEPT;

    //!< SixAxisSensorState を注入して内部ステートを更新します。
    void Sampling(const nn::xcd::SixAxisSensorState& state) NN_NOEXCEPT;

    //!< SixAxisSensorState を注入して内部ステートを更新します。
    void Sampling(const nn::xcd::SixAxisSensorState& state,
                  const nn::TimeSpan& deltaTime) NN_NOEXCEPT;

    //!< SixAxisSensor の入力状態を過去に遡って読み出します。
    int  GetSixAxisSensorStates(SixAxisSensorState* outStates, int count ) const NN_NOEXCEPT;

    bool IsSixAxisSensorAtRest() const NN_NOEXCEPT;

    //!< 現在の姿勢を基本姿勢を用いて更新します。
    void UpdateDirectionByBaseDirection(nn::hid::DirectionState* pOutDirection,
                                        const nn::hid::DirectionState& lastDirection) const NN_NOEXCEPT;

private:
    void Sampling(int sampleDelta,
                  const nn::util::Float3& accelerationHWC,
                  const nn::util::Float3& angularVelocityHWC,
                  float deltaTimeInSec,
                  const nn::hid::SixAxisSensorAttributeSet& attributes) NN_NOEXCEPT;

    //!< DEPRECATED
    void Sampling(const nn::xcd::SensorState& accelerationCount,
                  const nn::xcd::SensorState& angularVelocityCount,
                  const nn::TimeSpan& deltaTime,
                  const nn::hid::SixAxisSensorAttributeSet& attributes) NN_NOEXCEPT;

    //!< 現在の回転角度[deg] を更新します。
    void UpdateAngle(const nn::util::Float3& angularVelocity, float deltaTimeInSec) NN_NOEXCEPT;
    //!< 現在の回転角度[deg] を更新します。
    void UpdateAngle(const nn::util::Float3& angularVelocity) NN_NOEXCEPT;
    //!< 現在の姿勢値を更新します。
    void UpdateDirection(const nn::util::Float3& acceleration,
                         const nn::util::Float3& angularVelocity,
                         float deltaTimeInSec) NN_NOEXCEPT;
    //!< 現在の姿勢値を更新します。
    void UpdateDirection(const nn::util::Float3& acceleration,
                         const nn::util::Float3& angularVelocity) NN_NOEXCEPT;


    //!< パケットロスト時の補間処理をします。
    void Interpolate(const nn::xcd::SixAxisSensorState& state,
                     const int64_t& lostSampleCount) NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
