﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/hid/detail/hid_Log.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd.h>

#include "hid_SixAxisSensorMathUtility.h"
#include "hid_SixAxisSensorProcessor.h"

namespace
{

template <typename T>
T Clamp(T value, T min, T max)
{
    if (value < min)
    {
        value = min;
    }
    else if (max < value)
    {
        value = max;
    }

    return value;
}

void Clamp(nn::hid::SixAxisSensorState* pOutState)
{
    for(auto& acceleration : pOutState->acceleration.v)
    {
        acceleration = Clamp(acceleration,
                             -nn::hid::AccelerometerMax,
                             ::nn::hid::AccelerometerMax);
    }

    for(auto& angularVelocity : pOutState->angularVelocity.v)
    {
        angularVelocity = Clamp(angularVelocity,
                          -nn::hid::AngularVelocityMax,
                          nn::hid::AngularVelocityMax);
    }
}

} // namespace

namespace nn { namespace hid { namespace detail {

SixAxisSensorProcessor::SixAxisSensorProcessor() NN_NOEXCEPT
    : m_Lifo()
    , m_Accelerometer()
    , m_Gyroscope()
    , m_Angle()
    , m_Direction(UnitDirection)
    , m_LastDirection(UnitDirection)
    , m_SamplingNumber(0)
    , m_SamplingIntervalSeconds(0.005f) // デフォルト 5ms
    , m_LastPadState()
    , m_LastSixAxisSensorState()
{
    // 何もしない
}

void SixAxisSensorProcessor::Reset() NN_NOEXCEPT
{
    m_Lifo.Clear();
    m_LastSixAxisSensorState.sampleNumber = 0;
}

void SixAxisSensorProcessor::SetSixAxisSensorSetting(const SixAxisSensorSetting* const pSetting) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSetting);

    m_pSixAxisSensorProcessorSetting = &pSetting->sixAxisSensorProcessorSetting;
    m_Accelerometer.SetAccelerometerSetting(&pSetting->accelerometerSetting);
    m_Gyroscope.SetGyroscopeSetting(&pSetting->gyroscopeSetting);
}

//!< 6軸センサーのキャリブレーションを実行します。
void SixAxisSensorProcessor::SetSensorCalibrationValue( const nn::xcd::SensorCalibrationValue& value ) NN_NOEXCEPT
{
    m_Accelerometer.SetCalibrationValue(value.accelerometerOrigin, value.accelerometerSensitivity );
    m_Gyroscope.SetCalibrationValue(value.gyroscopeOrigin, value.gyroscopeSensitivity);
}

void SixAxisSensorProcessor::SetAngle( const ::nn::util::Float3& angle ) NN_NOEXCEPT
{
    m_Angle = angle;
}

void SixAxisSensorProcessor::SetDirection( const nn::hid::DirectionState& direction ) NN_NOEXCEPT
{
    m_Direction = direction;
}

bool SixAxisSensorProcessor::IsSixAxisSensorAtRest() const NN_NOEXCEPT
{
    return m_Gyroscope.IsAtRest();
}

void SixAxisSensorProcessor::UpdateDirectionByBaseDirection(nn::hid::DirectionState* pOutDirection,
                                               const nn::hid::DirectionState& lastDirection) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutDirection);
    NN_SDK_REQUIRES_NOT_NULL(m_pSixAxisSensorProcessorSetting);

    float f1;
    float f2;

    //----- 各方向を動かす
    f2 = ApproximateDirectionAxis(&pOutDirection->x,
                                  lastDirection.x,
                                  m_pSixAxisSensorProcessorSetting->baseDirection.x,
                                  m_pSixAxisSensorProcessorSetting->baseDirectionRevisePower);

    f1 = ApproximateDirectionAxis(&pOutDirection->y,
                                  lastDirection.y,
                                  m_pSixAxisSensorProcessorSetting->baseDirection.y,
                                  m_pSixAxisSensorProcessorSetting->baseDirectionRevisePower);

    if ( f1 > f2 )
    {
        f2 = f1 ;
    }

    f1 = ApproximateDirectionAxis(&pOutDirection->z,
                                  lastDirection.z,
                                  m_pSixAxisSensorProcessorSetting->baseDirection.z,
                                  m_pSixAxisSensorProcessorSetting->baseDirectionRevisePower);
    if ( f1 > f2 )
    {
        f2 = f1 ;
    }

    //----- 正規化する
    nn::hid::detail::NormalizeDirectionState( pOutDirection, 2.999f ) ;

    // f2 が補正の強さ
}

void SixAxisSensorProcessor::Sampling(nn::xcd::SixAxisSensorState const& state) NN_NOEXCEPT
{
    const int64_t LostSamples = state.sampleNumber - m_LastSixAxisSensorState.sampleNumber - 1;

    // xcd のサンプリング番号がクリアされた場合のワークアラウンド
    // 200 サンプル以上遡った場合は強制的に更新する
    const bool ForceUpdate = (LostSamples <= -200);

    // 過去の入力状態が指定された場合はスキップ
    if (ForceUpdate == false && LostSamples < 0)
    {
        return;
    }

    // パケットロスト時の補間処理を行います。
    if (m_LastSixAxisSensorState.sampleNumber != 0 && LostSamples > 0)
    {
        Interpolate(state, LostSamples);
    }


    // 新しくサンプリングされた生値を元に内部ステートを更新
    nn::hid::SixAxisSensorAttributeSet attributes;
    attributes.Reset();
    attributes.Set<nn::hid::SixAxisSensorAttribute::IsConnected>();

    Sampling(int(LostSamples),
             state.accelerometer,
             state.gyroscope,
             state.deltaTimeInSec,
             attributes);

    m_LastSixAxisSensorState = state;
}

void SixAxisSensorProcessor::Sampling(const nn::xcd::SixAxisSensorState& state,
                                      const nn::TimeSpan& deltaTime) NN_NOEXCEPT
{
    m_SamplingIntervalSeconds = static_cast<float>(deltaTime.GetMicroSeconds()) / 1000.f / 1000.f;

    // 新しくサンプリングされた生値を元に内部ステートを更新
    nn::hid::SixAxisSensorAttributeSet attributes;
    attributes.Reset();
    attributes.Set<nn::hid::SixAxisSensorAttribute::IsConnected>();

    Sampling(
        0,
        state.accelerometer,
        state.gyroscope,
        m_SamplingIntervalSeconds,
        attributes
    );
}

void SixAxisSensorProcessor::Sampling(int sampleDelta,
                                      const nn::util::Float3& accelerationCount,
                                      const nn::util::Float3& angularVelocityCount,
                                      float deltaTimeInSec,
                                      const nn::hid::SixAxisSensorAttributeSet& attributes) NN_NOEXCEPT
{
    NN_UNUSED(sampleDelta);
    NN_SDK_REQUIRES_NOT_NULL(m_pSixAxisSensorProcessorSetting);

    nn::hid::SixAxisSensorState sensorState = {};

    // 加速度値[G] の生成
    m_Accelerometer.GetState(&sensorState.acceleration, accelerationCount);


    // 角速度値[dps] の生成
    m_Gyroscope.GetState(&sensorState.angularVelocity,
        angularVelocityCount,
        attributes);

    // 回転角の更新
    UpdateAngle(sensorState.angularVelocity, deltaTimeInSec);
    // 姿勢値の更新
    UpdateDirection(sensorState.acceleration,
                    sensorState.angularVelocity,
                    deltaTimeInSec);

    // 6軸センサー値をクランプ
    Clamp(&sensorState);

    // 内挿補間でないときにのみ状態更新
    if (attributes.Test<nn::hid::SixAxisSensorAttribute::IsInterpolated>() == false)
    {
        sensorState.attributes = attributes;
        sensorState.angle = m_Angle;
        sensorState.direction = m_Direction;
        sensorState.samplingNumber = m_SamplingNumber;
        //m_OriginalSamplingNumber += sampleDelta;
        //sensorState.deltaTime = m_IsLegacy ? 0.005 : deltaTime;
        sensorState.deltaTime = ::nn::TimeSpan::FromNanoSeconds(static_cast<int64_t>(deltaTimeInSec * 1e9f));
        m_SamplingNumber++;

        // サンプリング無効時は入力状態に更新
        if (m_pSixAxisSensorProcessorSetting->IsSamplingEnabled() == false)
        {
            sensorState.acceleration = nn::util::Float3{ { { 0.f, 0.f, -1.f } } };
            sensorState.angularVelocity = nn::util::Float3{ { { 0.f, 0.f, 0.f } } };
            sensorState.angle = nn::util::Float3{ { { 0.f, 0.f, 0.f } } };
            sensorState.direction = UnitDirection;
        }

        m_Lifo.Append(sensorState); // LIFO に格納
    }
}

void SixAxisSensorProcessor::UpdateAngle(const nn::util::Float3& angularVelocity, float deltaTimeInSecond) NN_NOEXCEPT
{
    m_Angle.x += angularVelocity.x * deltaTimeInSecond;
    m_Angle.y += angularVelocity.y * deltaTimeInSecond;
    m_Angle.z += angularVelocity.z * deltaTimeInSecond;
}

void SixAxisSensorProcessor::UpdateAngle(const nn::util::Float3& angularVelocity) NN_NOEXCEPT
{
    UpdateAngle(angularVelocity, m_SamplingIntervalSeconds);
}

void SixAxisSensorProcessor::UpdateDirection(const nn::util::Float3& acceleration,
                                             const nn::util::Float3& angularVelocity,
                                             float deltaTimeInSecond) NN_NOEXCEPT
{
    m_LastDirection = m_Direction;

    // 角速度による姿勢値の更新
    m_Gyroscope.UpdateDirection(&m_Direction,
                                angularVelocity,
                                deltaTimeInSecond);

    // 基本姿勢による姿勢値の補正
    if (m_pSixAxisSensorProcessorSetting->isBaseDirReviseEnabled)
    {
        UpdateDirectionByBaseDirection(&m_Direction,
                                       m_LastDirection);
    }

    // 加速度による姿勢値の補正
    m_Accelerometer.UpdateDirection(&m_Direction,
                                    acceleration);
}

void SixAxisSensorProcessor::UpdateDirection(const nn::util::Float3& acceleration,
                                             const nn::util::Float3& angularVelocity) NN_NOEXCEPT
{
    UpdateDirection(acceleration, angularVelocity, m_SamplingIntervalSeconds);
}

void SixAxisSensorProcessor::Interpolate(const nn::xcd::SixAxisSensorState& state,
                                         const int64_t& lostSampleCount) NN_NOEXCEPT
{
    // パケットロストが無い場合は何もしない
    if (lostSampleCount <= 0)
    {
        return;
    }

    const auto ClampedLostSampleCount = (lostSampleCount > SixAxisSensorStateCountMax) ? SixAxisSensorStateCountMax
                                                                                       : lostSampleCount;

    // 線形補間のための増分を計算します。
    nn::xcd::SensorState deltaAccelerometer;
    deltaAccelerometer.x = static_cast<int16_t>((state.accelerometer.x - m_LastSixAxisSensorState.accelerometer.x) / (ClampedLostSampleCount + 1));
    deltaAccelerometer.y = static_cast<int16_t>((state.accelerometer.y - m_LastSixAxisSensorState.accelerometer.y) / (ClampedLostSampleCount + 1));
    deltaAccelerometer.z = static_cast<int16_t>((state.accelerometer.z - m_LastSixAxisSensorState.accelerometer.z) / (ClampedLostSampleCount + 1));

    nn::xcd::SensorState deltaGyroscope;
    deltaGyroscope.x = static_cast<int16_t>((state.gyroscope.x - m_LastSixAxisSensorState.gyroscope.x) / (ClampedLostSampleCount + 1));
    deltaGyroscope.y = static_cast<int16_t>((state.gyroscope.y - m_LastSixAxisSensorState.gyroscope.y) / (ClampedLostSampleCount + 1));
    deltaGyroscope.z = static_cast<int16_t>((state.gyroscope.z - m_LastSixAxisSensorState.gyroscope.z) / (ClampedLostSampleCount + 1));

    // attributes を生成
    nn::hid::SixAxisSensorAttributeSet attributes;
    attributes.Reset();
    attributes.Set<nn::hid::SixAxisSensorAttribute::IsConnected>();
    attributes.Set<nn::hid::SixAxisSensorAttribute::IsInterpolated>();

    nn::util::Float3 accelerometer = m_LastSixAxisSensorState.accelerometer;
    nn::util::Float3 gyroscope = m_LastSixAxisSensorState.gyroscope;
    for (int i = 0; i < ClampedLostSampleCount; i++)
    {
        accelerometer.x += deltaAccelerometer.x;
        accelerometer.y += deltaAccelerometer.y;
        accelerometer.z += deltaAccelerometer.z;

        gyroscope.x += deltaGyroscope.x;
        gyroscope.y += deltaGyroscope.y;
        gyroscope.z += deltaGyroscope.z;
        Sampling(0,
                 accelerometer,
                 gyroscope,
                 m_SamplingIntervalSeconds,
                 attributes);
    }
}

int SixAxisSensorProcessor::GetSixAxisSensorStates(SixAxisSensorState* pOutStates, int count ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);

    auto ret = m_Lifo.Read(pOutStates, count);
    return ret;
}

}}} // namespace nn::hid::detail //NOLINT(impl/function_size)
