﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <new>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_MemoryFence.h>
#include <nn/os/os_SharedMemory.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace hid { namespace detail {

//!< リング LIFO 用に共有メモリを初期化します。
::nn::Result InitializeSharedMemoryForRingLifo(
    ::nn::os::SharedMemoryType* sharedMemory, size_t lifoSize) NN_NOEXCEPT;

//!< リング LIFO 用に共有メモリをアタッチします。
void AttachSharedMemoryForRingLifo(::nn::os::SharedMemoryType* sharedMemory,
                             size_t lifoSize,
                             ::nn::os::NativeHandle handle,
                             bool managed) NN_NOEXCEPT;

//!< リング LIFO 用に初期化された共有メモリを開放します。
void FinalizeSharedMemoryForRingLifo(::nn::os::SharedMemoryType* sharedMemory
                                     ) NN_NOEXCEPT;

//!< リング LIFO を共有メモリ上で保持するためのクラスです。
template<typename T>
class RingLifoHolder final
{
    NN_DISALLOW_COPY(RingLifoHolder);
    NN_DISALLOW_MOVE(RingLifoHolder);

private:
    //!< 新規に構築されたか否かを表す値
    bool m_IsCreated;

    //!< 共通メモリオブジェクト
    ::nn::os::SharedMemoryType m_SharedMemory;

    //!< リング LIFO
    T* m_pLifo;

public:
    RingLifoHolder() NN_NOEXCEPT
        : m_IsCreated(false)
        , m_pLifo(nullptr)
    {
        // 何もしない
    }

    ~RingLifoHolder() NN_NOEXCEPT { /* 何もしない */ }

    //!< 共有メモリ上にリング LIFO を構築します。
    ::nn::Result Initialize() NN_NOEXCEPT
    {
        NN_RESULT_DO(
            InitializeSharedMemoryForRingLifo(&m_SharedMemory, sizeof(T)));

        m_IsCreated = true;

        m_pLifo = reinterpret_cast<T*>(
            ::nn::os::GetSharedMemoryAddress(&m_SharedMemory));

        if (m_pLifo != nullptr)
        {
            // リング LIFO の構築
            new(m_pLifo) T();

            // リング LIFO の構築の完了を保証
            ::nn::os::FenceMemoryStoreLoad();
        }

        NN_RESULT_SUCCESS;
    }

    //!< 共有メモリ上のリング LIFO をアタッチします。
    void Attach(::nn::os::NativeHandle handle, bool managed) NN_NOEXCEPT
    {
        AttachSharedMemoryForRingLifo(
            &m_SharedMemory, sizeof(T), handle, managed);

        m_pLifo = reinterpret_cast<T*>(
            ::nn::os::GetSharedMemoryAddress(&m_SharedMemory));
    }

    //!< 共有メモリ上に構築したリング LIFO を破棄します。
    void Finalize() NN_NOEXCEPT
    {
        if (m_IsCreated && (m_pLifo != nullptr))
        {
            // リング LIFO の破棄
            m_pLifo->~T();
            m_pLifo = nullptr;

            // 共有メモリに対するアクセスの完了を保証
            ::nn::os::FenceMemoryAnyStore();
        }

        m_IsCreated = false;

        FinalizeSharedMemoryForRingLifo(&m_SharedMemory);
    }

    //!< リング LIFO を返します。
    T* GetLifo() const NN_NOEXCEPT
    {
        return m_pLifo;
    }

    //!< 共有メモリのサイズを返します。
    size_t GetSharedMemorySize() const NN_NOEXCEPT
    {
        return ::nn::os::GetSharedMemorySize(&m_SharedMemory);
    }

    //!< 共有メモリのサイズを返します。
    ::nn::os::NativeHandle GetSharedMemoryHandle() const NN_NOEXCEPT
    {
        return ::nn::os::GetSharedMemoryHandle(&m_SharedMemory);
    }
};

}}} // namespace nn::hid::detail
