﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <atomic>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/result/result_HandlingUtility.h>

#include "hid_AtomicStorage.h"

namespace nn { namespace hid { namespace detail {

//!< リング FIFO を扱うためのクラスです。
template<typename FIFOT,
         int FIFOEntryCount,
         typename AtomicStorageT = AtomicStorage<FIFOT>>
class RingFIFO
{
    NN_STATIC_ASSERT(FIFOEntryCount >= 1);
    NN_DISALLOW_COPY(RingFIFO);
    NN_DISALLOW_MOVE(RingFIFO);

protected:
    //!< リングバッファ上の末尾位置
    ::std::atomic<int64_t> m_Tail;

    //!< リングバッファ上の要素数
    ::std::atomic<int64_t> m_Count;

    //!< リングバッファ
    AtomicStorageT m_Buffer[FIFOEntryCount + 1];

public:
    RingFIFO() NN_NOEXCEPT
        : m_Tail(0)
        , m_Count(0)
    {
        // 何もしない
    }

    virtual ~RingFIFO() NN_NOEXCEPT { /* 何もしない */ }

public:
    //!< リング FIFO 上の要素の型
    typedef FIFOT ValueType;

public:
    //!< FIFO を空にします。
    void Clear() NN_NOEXCEPT
    {
        m_Count = 0;
    }

    //!< FIFO に要素を追加します。
    Result Push(const FIFOT& value) NN_NOEXCEPT
    {
        NN_RESULT_DO(this->PushImpl(value));
        NN_RESULT_SUCCESS;
    }

    //!< FIFO が空か否か判定します。任意のタイミングでスレッドセーフです。
    bool IsEmpty() const NN_NOEXCEPT
    {
        return (m_Count == 0);
    }

    //!< FIFO から要素を取り出します。任意のタイミングでスレッドセーフです。
    Result Pop(FIFOT* buffer) NN_NOEXCEPT
    {
        NN_RESULT_DO(this->PopImpl(buffer));
        NN_RESULT_SUCCESS;
    }

protected:
    //!< 基準点からの相対位置を絶対位置に変換します。
    static int CalculateIndex(int index, int bias, int bufferCount) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(index, 0, bufferCount);
        NN_SDK_REQUIRES_RANGE(bias, -bufferCount, bufferCount);
        return (index + bufferCount + bias) % bufferCount;
    }

    static int CalculateNextIndex(int index, int bufferCount) NN_NOEXCEPT
    {
        if(index >= (bufferCount - 1))
        {
            return 0;
        }

        return index + 1;
    }

    //!< Push() の内部実装
    template<typename T>
    Result PushImpl(const T& value) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_Count < FIFOEntryCount, ResultFifoFull());

        // 次の末尾位置を計算
        int nextIndex = RingFIFO::CalculateNextIndex(static_cast<int>(m_Tail), NN_ARRAY_SIZE(m_Buffer));

        // 末尾位置に要素を書き込み
        m_Buffer[nextIndex].Set(value);

        // 末尾位置を更新
        m_Tail = nextIndex;

        // リングバッファ上の要素数を更新
        if (m_Count < FIFOEntryCount)
        {
            ++m_Count;
        }

        NN_RESULT_SUCCESS;
    }

    //!< Pop() の内部実装
    template<typename T>
    Result PopImpl(T* buffer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(buffer);
        NN_RESULT_THROW_UNLESS(m_Count > 0, ResultFifoEmpty());

        while (NN_STATIC_CONDITION(true))
        {
            // 読み出し開始時点の末尾位置を待避
            const int tail = static_cast<int>(m_Tail);
            const int bias = m_Count - 1;

            // 古い値から順に要素の読み出し
            const int index = RingFIFO::CalculateIndex(
                tail, -bias, NN_ARRAY_SIZE(m_Buffer));

            if (m_Buffer[index].Get(buffer))
            {
                // 読み出しに成功したらカウントを 1つ減らす
                --m_Count;
                break;
            }

            //　読み出しができなかった場合は、再度繰り返す
        }

        NN_RESULT_SUCCESS;
    }
};


}}} // namespace nn::hid::detail
