﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/result/result_HandlingUtility.h>

#include "hid_ActivationCount.h"
#include "hid_PlayReportManager.h"

namespace nn { namespace hid { namespace detail {

#define NOTIFY_IMMEDIATE

#ifndef NOTIFY_IMMEDIATE
const ::nn::TimeSpan PlayReportManager::SamplingInterval =
    ::nn::TimeSpan::FromSeconds(60);
#endif

PlayReportManager::PlayReportManager() NN_NOEXCEPT :
    m_pTimer(nullptr),
    m_ActivationCount(),
    m_UsageUpdateEvent(::nn::os::EventClearMode_ManualClear, true),
    m_UpdateFlags(),
    m_Mutex(false)
{
    // 何もしない
}

::nn::Result PlayReportManager::Activate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultGamePadDriverActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
#ifndef NOTIFY_IMMEDIATE
        // タイマーイベントをアクティブ化
        m_pTimer->Enable(SamplingInterval);
#endif
    }

    // アクティブ化した回数をインクリメント
    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result PlayReportManager::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultGamePadDriverDeactivationLowerLimitOver());

    // アクティブ化した回数をデクリメント
    --m_ActivationCount;

    if(m_ActivationCount.IsZero())
    {
        // タイマーイベントを非アクティブ化
        m_pTimer->Disable();
    }

    NN_RESULT_SUCCESS;
}

void PlayReportManager::SetTimer(SynchronizedTimer* timer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(timer);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pTimer = timer;
}

void PlayReportManager::CheckForUpdates() NN_NOEXCEPT
{
    if (m_UpdateFlags.Test<PlayReportUpdateFlag::Usage>())
    {
        m_UsageUpdateEvent.Signal();
    }

    m_UpdateFlags.Reset();
}

Result PlayReportManager::AcquireUsageUpdateEventHandle(::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    *pOutValue = m_UsageUpdateEvent.GetReadableHandle();

    NN_RESULT_SUCCESS;
}

int PlayReportManager::GetControllerUsages(system::PlayReportControllerUsage* pOutValues, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValues);
    NN_SDK_REQUIRES_GREATER(count, 0);

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);
    int returnCount = std::min(m_UsagesCount, count);
    for (int i = 0; i < returnCount; i++)
    {
        pOutValues[i] = m_Usages[i];
    }
    return returnCount;
}

void PlayReportManager::UpdateDeviceInUse(uint8_t controllerNumber, system::PlayReportControllerUsage* pValues, int count) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);
    int index = 0;

    // ControllerNumber の一致する Usage をすべて削除
    while(index < m_UsagesCount)
    {
        if (m_Usages[index].controllerNumber == controllerNumber)
        {
            // データを削除して前詰めする
            for (int i = index; i < m_UsagesCount - 1; ++i)
            {
                m_Usages[i] = m_Usages[i + 1];
            }

            m_Usages[m_UsagesCount - 1] = system::PlayReportControllerUsage();
            --m_UsagesCount;
        }
        else
        {
            index++;
        }
    }

    // Usage をリストの末尾に追加する
    for (int i = 0; i < count; i++)
    {
        if (m_UsagesCount < system::PlayReportControllerUsageCountMax)
        {
            m_Usages[m_UsagesCount] = pValues[i];
            m_UsagesCount++;
        }
    }

    // データの更新フラグをセット
    m_UpdateFlags.Set<PlayReportUpdateFlag::Usage>();

#ifdef NOTIFY_IMMEDIATE
    this->CheckForUpdates();
#endif
}

bool PlayReportManager::GetIndexOfDevice(int* pOutIndex, uint8_t controllerNumber, system::PlayReportDeviceType device) NN_NOEXCEPT
{
    for (int i = 0; i < m_UsagesCount; ++i)
    {
        if (m_Usages[i].controllerNumber == controllerNumber &&
            m_Usages[i].deviceType == device)
        {
            *pOutIndex = i;
            return true;
        }
    }

    return false;
}

}}} // namespace nn::hid::detail
