﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/hid/hid_ResultPalma.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_BitPack.h>
#include <nn/xcd/xcd.h>
#include "hid_PalmaManager.h"

namespace nn { namespace hid { namespace detail {

namespace
{
    // Activity Entry のヘッダー情報を除いた実データのサイズ
    const int ActivityEntryDataFieldSize = 5;

    void GetPackedActivityEntryData(uint8_t* outBuffer, uint8_t waveSet, uint16_t waveIndex, uint16_t ledIndex)
    {
        NN_SDK_REQUIRES_NOT_NULL(outBuffer);

        *(reinterpret_cast<uint16_t*>(&outBuffer[0])) = ledIndex;
        *(reinterpret_cast<uint16_t*>(&outBuffer[2])) = waveIndex;
        outBuffer[4] = waveSet;
    }

    ::nn::xcd::BleDeviceDatabaseType ConvertToDatabaseType(PalmaWaveSet waveSet)
    {
        switch (waveSet)
        {
            case PalmaWaveSet_Small:
                return ::nn::xcd::BleDeviceDatabaseType_WaveSmall;
            case PalmaWaveSet_Medium:
                return ::nn::xcd::BleDeviceDatabaseType_WaveMedium;
            case PalmaWaveSet_Large:
                return ::nn::xcd::BleDeviceDatabaseType_WaveLarge;
            default:
                NN_UNEXPECTED_DEFAULT;
        }
    }

    void ConvertToPalmaOperationInfoWriteDatabaseEntry(PalmaOperationType* pOutType, PalmaOperationInfo* pOutValue, const ::nn::xcd::BleDeviceOperationResult& operationResult)
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutType);
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);

        auto databaseType = operationResult.individual.writeDatabaseEntry.type;
        auto index = operationResult.individual.writeDatabaseEntry.index;
        switch (databaseType)
        {
            case ::nn::xcd::BleDeviceDatabaseType_Activity:
                *pOutType = ::nn::hid::PalmaOperationType_WriteActivityEntry;
                pOutValue->individual.writeActivityEntry.index = index;
                break;
            case ::nn::xcd::BleDeviceDatabaseType_Led:
                *pOutType = ::nn::hid::PalmaOperationType_WriteRgbLedPatternEntry;
                pOutValue->individual.writeRgbLedPatternEntry.index = index;
                break;
            case ::nn::xcd::BleDeviceDatabaseType_WaveSmall:
                *pOutType = ::nn::hid::PalmaOperationType_WriteWaveEntry;
                pOutValue->individual.writeWaveEntry.waveSet = ::nn::hid::PalmaWaveSet_Small;
                pOutValue->individual.writeWaveEntry.index = index;
                break;
            case ::nn::xcd::BleDeviceDatabaseType_WaveMedium:
                *pOutType = ::nn::hid::PalmaOperationType_WriteWaveEntry;
                pOutValue->individual.writeWaveEntry.waveSet = ::nn::hid::PalmaWaveSet_Medium;
                pOutValue->individual.writeWaveEntry.index = index;
                break;
            case ::nn::xcd::BleDeviceDatabaseType_WaveLarge:
                *pOutType = ::nn::hid::PalmaOperationType_WriteWaveEntry;
                pOutValue->individual.writeWaveEntry.waveSet = ::nn::hid::PalmaWaveSet_Large;
                pOutValue->individual.writeWaveEntry.index = index;
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
        }
    }

    void ConvertToPalmaOperationInfo(PalmaOperationType* pOutType, PalmaOperationInfo* pOutValue, const ::nn::xcd::BleDeviceOperationResult& operationResult)
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutType);
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);
        memset(pOutValue->individual.raw, 0, sizeof(pOutValue->individual.raw));

        switch (operationResult.type)
        {
            case ::nn::xcd::BleDeviceOperationType_PlayDeviceActivity:
            {
                *pOutType = ::nn::hid::PalmaOperationType_PlayActivity;
                pOutValue->individual.playActivity.index = operationResult.individual.activityIndex.index;
                break;
            }
            case ::nn::xcd::BleDeviceOperationType_WriteDatabaseEntry:
            {
                ConvertToPalmaOperationInfoWriteDatabaseEntry(pOutType, pOutValue, operationResult);
                break;
            }
            case ::nn::xcd::BleDeviceOperationType_GetDatabaseIdentificationVersion:
            {
                *pOutType = ::nn::hid::PalmaOperationType_ReadDataBaseIdentificationVersion;
                pOutValue->individual.readDataBaseIdentificationVersion.version = operationResult.individual.databaseIdentificationVersion.version;
                break;
            }
            case ::nn::xcd::BleDeviceOperationType_SetDatabaseIdentificationVersion:
            {
                *pOutType = ::nn::hid::PalmaOperationType_WriteDataBaseIdentificationVersion;
                pOutValue->individual.writeDataBaseIdentificationVersion.version = operationResult.individual.databaseIdentificationVersion.version;
                break;
            }
            case ::nn::xcd::BleDeviceOperationType_SetFrModeType:
            {
                *pOutType = ::nn::hid::PalmaOperationType_SetFrModeType;
                pOutValue->individual.setFrModeType.frModeType = static_cast<PalmaFrModeType>(operationResult.individual.frModeType.type);
                break;
            }
            case ::nn::xcd::BleDeviceOperationType_GetStepCount:
            {
                *pOutType = ::nn::hid::PalmaOperationType_ReadStep;
                pOutValue->individual.readStep.step = operationResult.individual.stepCount.step;
                break;
            }
            case ::nn::xcd::BleDeviceOperationType_EnableStep:
            {
                *pOutType = ::nn::hid::PalmaOperationType_EnableStep;
                pOutValue->individual.enablePalmaStep.isEnabled = operationResult.individual.stepCounterState.isEnabled;
                break;
            }
            case ::nn::xcd::BleDeviceOperationType_ReadApplicationSection:
            {
                *pOutType = ::nn::hid::PalmaOperationType_ReadApplicationSection;
                auto* pInfo = &pOutValue->individual.readApplicationSection;
                pInfo->address = operationResult.individual.readApplicationSection.address;
                pInfo->size = operationResult.individual.readApplicationSection.size;
                NN_SDK_REQUIRES_LESS_EQUAL(static_cast<size_t>(pInfo->size), PalmaApplicationSectionAccessUnitSizeMax);
                memcpy(pInfo->buffer.raw, operationResult.individual.readApplicationSection.data, pInfo->size);
                break;
            }
            case ::nn::xcd::BleDeviceOperationType_WriteApplicationSection:
            {
                *pOutType = ::nn::hid::PalmaOperationType_WriteApplicationSection;
                pOutValue->individual.writeApplicationSection.address = operationResult.individual.writeApplicationSection.address;
                pOutValue->individual.writeApplicationSection.size = operationResult.individual.writeApplicationSection.size;
                break;
            }
            case ::nn::xcd::BleDeviceOperationType_ReadContentUniqueCode:
            {
                *pOutType = ::nn::hid::PalmaOperationType_ReadUniqueCode;
                auto* pInfo = &pOutValue->individual.readUniqueCode;
                memcpy(pInfo->uniqueCode, operationResult.individual.contentUniqueCode.uniqueCode, sizeof(pInfo->uniqueCode));
                memcpy(pInfo->contentId, operationResult.individual.contentUniqueCode.contentId, sizeof(pInfo->contentId));
                break;
            }
            case ::nn::xcd::BleDeviceOperationType_SuspendFeature:
            {
                *pOutType = ::nn::hid::PalmaOperationType_SuspendFeature;
                PalmaFeatureSet palmaFeatureSet;
                palmaFeatureSet.Reset();
                auto suspendBitFlag = nn::util::BitPack32();
                suspendBitFlag.SetMaskedBits(0xFFFFFFFF, operationResult.individual.suspendFeature.flagSet);
                for (int i = 0; i < palmaFeatureSet.GetCount(); ++i)
                {
                    palmaFeatureSet.Set(i, suspendBitFlag.GetBit(i));
                }

                pOutValue->individual.suspendPalmaFeature.suspendFeatureSet = palmaFeatureSet;
                break;
            }
            case ::nn::xcd::BleDeviceOperationType_ReadPlayLog:
            {
                *pOutType = ::nn::hid::PalmaOperationType_ReadPlayLog;
                auto* pInfo = &pOutValue->individual.readPalmaPlayLog;
                pInfo->index = operationResult.individual.playLog.index;
                pInfo->size = operationResult.individual.playLog.size;
                memcpy(pInfo->raw, operationResult.individual.playLog.raw, sizeof(pInfo->raw));
                break;
            }
            case ::nn::xcd::BleDeviceOperationType_ResetPlayLog:
            {
                *pOutType = ::nn::hid::PalmaOperationType_ResetPlayLog;
                auto* pInfo = &pOutValue->individual.resetPalmaPlayLog;
                pInfo->index = operationResult.individual.playLog.index;
                break;
            }
            case ::nn::xcd::BleDeviceOperationType_ResetStepCount:
                *pOutType = ::nn::hid::PalmaOperationType_ResetStep;
                break;
            case ::nn::xcd::BleDeviceOperationType_SetContentUniqueCodeInvalid:
                *pOutType = ::nn::hid::PalmaOperationType_SetUniqueCodeInvalid;
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
        }
    } // NOLINT(readability/fn_size)

    ::nn::Result ParseBleDeviceOperationResultType(::nn::xcd::BleDeviceOperationResultType resultType)
    {
        switch (resultType)
        {
            case ::nn::xcd::BleDeviceOperationResultType_Success:
                NN_RESULT_SUCCESS;
            case ::nn::xcd::BleDeviceOperationResultType_Busy:
                return ResultPalmaBusy();
            case ::nn::xcd::BleDeviceOperationResultType_UniqueCodeInvalid:
                return ResultPalmaUniqueCodeInvalid();
            case ::nn::xcd::BleDeviceOperationResultType_OperationNotSupported:
                return ResultPalmaOperationNotSupported();
            case ::nn::xcd::BleDeviceOperationResultType_ResponseTimeout:
                return ResultPalmaOperationTimeout();
            default:
                NN_UNEXPECTED_DEFAULT;
        }
    }

} // namespace

PalmaManager::PalmaManager() NN_NOEXCEPT
    : m_OperationCompletedEvent(::nn::os::EventClearMode_ManualClear, true)
    , m_NpadId(0)
    , m_Handle(::nn::xcd::BleInvalidConnectionHandle)
    , m_pMappedAddress(nullptr)
    , m_TransferMemorySize(0)
    , m_TransferMemoryHolderAruid(::nn::applet::AppletResourceUserId::GetInvalidId())
    , m_OperationResult(::nn::ResultSuccess())
{
    // 何もしない
}

PalmaManager::~PalmaManager() NN_NOEXCEPT
{
    // 何もしない
}

::nn::Result PalmaManager::Initialize() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Handle != ::nn::xcd::BleInvalidConnectionHandle, nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_TRY(::nn::xcd::RegisterBleDeviceOperationEvent(m_OperationCompletedEvent.GetBase(), m_Handle))
        NN_RESULT_CATCH(::nn::xcd::ResultNotConnected)
        {
            NN_RESULT_THROW(ResultPalmaInvalidHandle());
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaManager::AcquireOperationCompleteEventHandle(::nn::os::NativeHandle* outHandle) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Handle != ::nn::xcd::BleInvalidConnectionHandle, nn::hid::ResultPalmaInvalidHandle());

    *outHandle = m_OperationCompletedEvent.GetReadableHandle();

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaManager::GetNpadId(nn::hid::NpadIdType* pOutNpadId) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsValidHandle(), nn::hid::ResultPalmaInvalidHandle());

    *pOutNpadId = m_NpadId;

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaManager::SetXcdHandle(::nn::xcd::BleConnectionHandle handle) NN_NOEXCEPT
{
    ::nn::xcd::RegisterBleDeviceOperationEvent(m_OperationCompletedEvent.GetBase(), handle);
    m_Handle = handle;
    NN_RESULT_SUCCESS;
}

void PalmaManager::InvalidateXcdHandle() NN_NOEXCEPT
{
    ResetTransferMemory(m_TransferMemoryHolderAruid);
    m_Handle = ::nn::xcd::BleInvalidConnectionHandle;
}

::nn::Result PalmaManager::GetXcdHandle(::nn::xcd::BleConnectionHandle* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Handle != ::nn::xcd::BleInvalidConnectionHandle, nn::hid::ResultPalmaInvalidHandle());
    *pOutValue = m_Handle;

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaManager::GetTransferMemoryType(::nn::os::TransferMemoryType** pTransferMemory) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Handle != ::nn::xcd::BleInvalidConnectionHandle, nn::hid::ResultPalmaInvalidHandle());
    *pTransferMemory = &m_TransferMemory;

    NN_RESULT_SUCCESS;
}

void PalmaManager::SetTransferMemory(void* pMappedAddress, uint64_t size, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    m_pMappedAddress = pMappedAddress;
    m_TransferMemorySize = size;
    m_TransferMemoryHolderAruid = aruid;
}

void PalmaManager::ResetTransferMemory(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    if (IsTransferMemoryMapped() && aruid == m_TransferMemoryHolderAruid)
    {
        ::nn::xcd::CancelWriteDatabaseEntry(m_Handle);
        ::nn::os::UnmapTransferMemory(&m_TransferMemory);
        ::nn::os::DestroyTransferMemory(&m_TransferMemory);
        m_pMappedAddress = nullptr;
        m_TransferMemoryHolderAruid = ::nn::applet::AppletResourceUserId::GetInvalidId();
    }
}

::nn::Result PalmaManager::GetPalmaOperationInfo(PalmaOperationType* pOutValue, void* pOutBuffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Handle != ::nn::xcd::BleInvalidConnectionHandle, nn::hid::ResultPalmaInvalidHandle());

    ::nn::xcd::BleDeviceOperationResult operationResult;

    NN_RESULT_TRY(::nn::xcd::GetBleDeviceOperationResult(&operationResult, m_Handle))
        NN_RESULT_CATCH(::nn::xcd::ResultNotConnected)
        {
            NN_RESULT_THROW(ResultPalmaInvalidHandle());
        }
        NN_RESULT_CATCH(::nn::xcd::ResultBluetoothNoOperationResult)
        {
            NN_RESULT_THROW(ResultPalmaNoOperationInfo());
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT("Unexpected result: %08x\n", (NN_RESULT_CURRENT_RESULT).GetInnerValueForDebug());
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;

    if (operationResult.type == ::nn::xcd::BleDeviceOperationType_WriteDatabaseEntry)
    {
        ResetTransferMemory(m_TransferMemoryHolderAruid);
    }

    PalmaOperationInfo palmaOperationInfo;
    ConvertToPalmaOperationInfo(pOutValue, &palmaOperationInfo, operationResult);
    NN_SDK_REQUIRES_EQUAL(bufferSize, sizeof(palmaOperationInfo.individual));
    memcpy(pOutBuffer, &palmaOperationInfo.individual, bufferSize);

    m_OperationResult = ParseBleDeviceOperationResultType(operationResult.result);

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaManager::GetPalmaOperationResult() NN_NOEXCEPT
{
    return m_OperationResult;
}

::nn::Result PalmaManager::PlayPalmaActivity(uint16_t number) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Handle != ::nn::xcd::BleInvalidConnectionHandle, nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_TRY(::nn::xcd::PlayDeviceActivity(number, m_Handle))
        NN_RESULT_CATCH(::nn::xcd::ResultNotConnected)
        {
            NN_RESULT_THROW(nn::hid::ResultPalmaNotConnected());
        }
        NN_RESULT_CATCH(::nn::xcd::ResultBluetoothLeBusy)
        {
            NN_RESULT_THROW(nn::hid::ResultPalmaBusy());
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT("Unexpected result: %08x\n", (NN_RESULT_CURRENT_RESULT).GetInnerValueForDebug());
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaManager::SetFrModeType(uint64_t type) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Handle != ::nn::xcd::BleInvalidConnectionHandle, nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_TRY(::nn::xcd::SetFrModeType(static_cast<uint32_t>(type), m_Handle))
        NN_RESULT_CATCH(::nn::xcd::ResultNotConnected)
        {
            NN_RESULT_THROW(nn::hid::ResultPalmaNotConnected());
        }
        NN_RESULT_CATCH(::nn::xcd::ResultBluetoothLeBusy)
        {
            NN_RESULT_THROW(nn::hid::ResultPalmaBusy());
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT("Unexpected result: %08x\n", (NN_RESULT_CURRENT_RESULT).GetInnerValueForDebug());
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaManager::ReadStep() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Handle != ::nn::xcd::BleInvalidConnectionHandle, nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_TRY(::nn::xcd::GetStepCount(m_Handle))
        NN_RESULT_CATCH(::nn::xcd::ResultNotConnected)
        {
            NN_RESULT_THROW(nn::hid::ResultPalmaNotConnected());
        }
        NN_RESULT_CATCH(::nn::xcd::ResultBluetoothLeBusy)
        {
            NN_RESULT_THROW(nn::hid::ResultPalmaBusy());
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT("Unexpected result: %08x\n", (NN_RESULT_CURRENT_RESULT).GetInnerValueForDebug());
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaManager::EnableStepCounter(bool isEnabled) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Handle != ::nn::xcd::BleInvalidConnectionHandle, nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_TRY(::nn::xcd::EnableStepCounter(isEnabled, m_Handle))
        NN_RESULT_CATCH(::nn::xcd::ResultNotConnected)
        {
            NN_RESULT_THROW(nn::hid::ResultPalmaNotConnected());
        }
        NN_RESULT_CATCH(::nn::xcd::ResultBluetoothLeBusy)
        {
            NN_RESULT_THROW(nn::hid::ResultPalmaBusy());
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT("Unexpected result: %08x\n", (NN_RESULT_CURRENT_RESULT).GetInnerValueForDebug());
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaManager::ResetStep() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Handle != ::nn::xcd::BleInvalidConnectionHandle, nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_TRY(::nn::xcd::ResetStepCount(m_Handle))
        NN_RESULT_CATCH(::nn::xcd::ResultNotConnected)
        {
            NN_RESULT_THROW(nn::hid::ResultPalmaNotConnected());
        }
        NN_RESULT_CATCH(::nn::xcd::ResultBluetoothLeBusy)
        {
            NN_RESULT_THROW(nn::hid::ResultPalmaBusy());
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT("Unexpected result: %08x\n", (NN_RESULT_CURRENT_RESULT).GetInnerValueForDebug());
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaManager::ReadApplicationSection(uint64_t address, uint64_t size) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Handle != ::nn::xcd::BleInvalidConnectionHandle, nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_TRY(::nn::xcd::ReadApplicationSection(static_cast<int32_t>(address), static_cast<int32_t>(size), m_Handle))
        NN_RESULT_CATCH(::nn::xcd::ResultNotConnected)
        {
            NN_RESULT_THROW(nn::hid::ResultPalmaNotConnected());
        }
        NN_RESULT_CATCH(::nn::xcd::ResultBluetoothLeBusy)
        {
            NN_RESULT_THROW(nn::hid::ResultPalmaBusy());
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT("Unexpected result: %08x\n", (NN_RESULT_CURRENT_RESULT).GetInnerValueForDebug());
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaManager::WriteApplicationSection(uint64_t address, uint64_t size, const PalmaApplicationSectionAccessBuffer& buffer) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Handle != ::nn::xcd::BleInvalidConnectionHandle, nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_TRY(::nn::xcd::WriteApplicationSection(static_cast<int32_t>(address), reinterpret_cast<const void*>(buffer.raw), static_cast<int32_t>(size), m_Handle))
        NN_RESULT_CATCH(::nn::xcd::ResultNotConnected)
        {
            NN_RESULT_THROW(nn::hid::ResultPalmaNotConnected());
        }
        NN_RESULT_CATCH(::nn::xcd::ResultBluetoothLeBusy)
        {
            NN_RESULT_THROW(nn::hid::ResultPalmaBusy());
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT("Unexpected result: %08x\n", (NN_RESULT_CURRENT_RESULT).GetInnerValueForDebug());
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaManager::ReadPalmaUniqueCode() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Handle != ::nn::xcd::BleInvalidConnectionHandle, nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_TRY(::nn::xcd::ReadContentUniqueCode(m_Handle))
        NN_RESULT_CATCH(::nn::xcd::ResultNotConnected)
        {
            NN_RESULT_THROW(nn::hid::ResultPalmaNotConnected());
        }
        NN_RESULT_CATCH(::nn::xcd::ResultBluetoothLeBusy)
        {
            NN_RESULT_THROW(nn::hid::ResultPalmaBusy());
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT("Unexpected result: %08x\n", (NN_RESULT_CURRENT_RESULT).GetInnerValueForDebug());
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaManager::SetPalmaUniqueCodeInvalid() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Handle != ::nn::xcd::BleInvalidConnectionHandle, nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_TRY(::nn::xcd::SetContentUniqueCodeInvalid(m_Handle))
        NN_RESULT_CATCH(::nn::xcd::ResultNotConnected)
        {
            NN_RESULT_THROW(nn::hid::ResultPalmaNotConnected());
        }
        NN_RESULT_CATCH(::nn::xcd::ResultBluetoothLeBusy)
        {
            NN_RESULT_THROW(nn::hid::ResultPalmaBusy());
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT("Unexpected result: %08x\n", (NN_RESULT_CURRENT_RESULT).GetInnerValueForDebug());
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaManager::WritePalmaActivityEntry(uint64_t index, uint64_t ledIndex, uint64_t waveSet, uint64_t waveIndex) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Handle != ::nn::xcd::BleInvalidConnectionHandle, nn::hid::ResultPalmaInvalidHandle());

    ::nn::xcd::BleDeviceDatabaseEntryConfig config;
    config.type = ::nn::xcd::BleDeviceDatabaseType_Activity;
    config.index = static_cast<uint16_t>(index);
    uint8_t rawData[ActivityEntryDataFieldSize];
    GetPackedActivityEntryData(rawData, static_cast<uint8_t>(waveSet), static_cast<uint16_t>(waveIndex), static_cast<uint16_t>(ledIndex));

    NN_RESULT_TRY(::nn::xcd::WriteDatabaseEntry(config, ActivityEntryDataFieldSize, reinterpret_cast<const void*>(rawData), m_Handle))
        NN_RESULT_CATCH(::nn::xcd::ResultNotConnected)
        {
            NN_RESULT_THROW(nn::hid::ResultPalmaNotConnected());
        }
        NN_RESULT_CATCH(::nn::xcd::ResultBluetoothLeBusy)
        {
            NN_RESULT_THROW(nn::hid::ResultPalmaBusy());
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT("Unexpected result: %08x\n", (NN_RESULT_CURRENT_RESULT).GetInnerValueForDebug());
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaManager::WritePalmaRgbLedPatternEntry(uint64_t index, const char* buffer, size_t size) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Handle != ::nn::xcd::BleInvalidConnectionHandle, nn::hid::ResultPalmaInvalidHandle());

    ::nn::xcd::BleDeviceDatabaseEntryConfig config;
    config.type = ::nn::xcd::BleDeviceDatabaseType_Led;
    config.index = static_cast<uint16_t>(index);

    NN_RESULT_TRY(::nn::xcd::WriteDatabaseEntry(config, size, reinterpret_cast<const void*>(buffer), m_Handle))
        NN_RESULT_CATCH(::nn::xcd::ResultNotConnected)
        {
            NN_RESULT_THROW(nn::hid::ResultPalmaNotConnected());
        }
        NN_RESULT_CATCH(::nn::xcd::ResultBluetoothLeBusy)
        {
            NN_RESULT_THROW(nn::hid::ResultPalmaBusy());
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT("Unexpected result: %08x\n", (NN_RESULT_CURRENT_RESULT).GetInnerValueForDebug());
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaManager::WritePalmaWaveEntry(uint64_t waveSet, uint64_t index, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pMappedAddress);
    NN_RESULT_THROW_UNLESS(m_Handle != ::nn::xcd::BleInvalidConnectionHandle, nn::hid::ResultPalmaInvalidHandle());

    ::nn::xcd::BleDeviceDatabaseEntryConfig config;
    config.type = ConvertToDatabaseType(static_cast<PalmaWaveSet>(waveSet));
    config.index = static_cast<uint16_t>(index);

    NN_RESULT_TRY(::nn::xcd::WriteDatabaseEntry(config, size, m_pMappedAddress, m_Handle))
        NN_RESULT_CATCH(::nn::xcd::ResultNotConnected)
        {
            NN_RESULT_THROW(nn::hid::ResultPalmaNotConnected());
        }
        NN_RESULT_CATCH(::nn::xcd::ResultBluetoothLeBusy)
        {
            NN_RESULT_THROW(nn::hid::ResultPalmaBusy());
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT("Unexpected result: %08x\n", (NN_RESULT_CURRENT_RESULT).GetInnerValueForDebug());
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaManager::SetPalmaDataBaseIdentificationVersion(int32_t version) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Handle != ::nn::xcd::BleInvalidConnectionHandle, nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_TRY(::nn::xcd::SetDatabaseIdentificationVersion(version, m_Handle))
        NN_RESULT_CATCH(::nn::xcd::ResultNotConnected)
        {
            NN_RESULT_THROW(nn::hid::ResultPalmaNotConnected());
        }
        NN_RESULT_CATCH(::nn::xcd::ResultBluetoothLeBusy)
        {
            NN_RESULT_THROW(nn::hid::ResultPalmaBusy());
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT("Unexpected result: %08x\n", (NN_RESULT_CURRENT_RESULT).GetInnerValueForDebug());
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaManager::GetPalmaDataBaseIdentificationVersion() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Handle != ::nn::xcd::BleInvalidConnectionHandle, nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_TRY(::nn::xcd::GetDatabaseIdentificationVersion(m_Handle))
        NN_RESULT_CATCH(::nn::xcd::ResultNotConnected)
        {
            NN_RESULT_THROW(nn::hid::ResultPalmaNotConnected());
        }
        NN_RESULT_CATCH(::nn::xcd::ResultBluetoothLeBusy)
        {
            NN_RESULT_THROW(nn::hid::ResultPalmaBusy());
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT("Unexpected result: %08x\n", (NN_RESULT_CURRENT_RESULT).GetInnerValueForDebug());
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaManager::SuspendPalmaFeature(PalmaFeatureSet suspendFeatureSet) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Handle != ::nn::xcd::BleInvalidConnectionHandle, nn::hid::ResultPalmaInvalidHandle());

    uint32_t flagSet = 0;
    for (int i = 0; i < suspendFeatureSet.GetCount(); ++i)
    {
        if (suspendFeatureSet[i])
        {
            flagSet |= (1 << i);
        }
    }

    NN_RESULT_TRY(::nn::xcd::SuspendFeature(flagSet, m_Handle))
        NN_RESULT_CATCH(::nn::xcd::ResultNotConnected)
        {
            NN_RESULT_THROW(nn::hid::ResultPalmaNotConnected());
        }
        NN_RESULT_CATCH(::nn::xcd::ResultBluetoothLeBusy)
        {
            NN_RESULT_THROW(nn::hid::ResultPalmaBusy());
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT("Unexpected result: %08x\n", (NN_RESULT_CURRENT_RESULT).GetInnerValueForDebug());
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaManager::ReadPalmaPlayLog(uint16_t index) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Handle != ::nn::xcd::BleInvalidConnectionHandle, nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_TRY(::nn::xcd::ReadDevicePlayLog(index, m_Handle))
        NN_RESULT_CATCH(::nn::xcd::ResultNotConnected)
        {
            NN_RESULT_THROW(nn::hid::ResultPalmaNotConnected());
        }
        NN_RESULT_CATCH(::nn::xcd::ResultBluetoothLeBusy)
        {
            NN_RESULT_THROW(nn::hid::ResultPalmaBusy());
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT("Unexpected result: %08x\n", (NN_RESULT_CURRENT_RESULT).GetInnerValueForDebug());
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaManager::ResetPalmaPlayLog(uint16_t index) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Handle != ::nn::xcd::BleInvalidConnectionHandle, nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_TRY(::nn::xcd::ResetDevicePlayLog(index, m_Handle))
        NN_RESULT_CATCH(::nn::xcd::ResultNotConnected)
        {
            NN_RESULT_THROW(nn::hid::ResultPalmaNotConnected());
        }
        NN_RESULT_CATCH(::nn::xcd::ResultBluetoothLeBusy)
        {
            NN_RESULT_THROW(nn::hid::ResultPalmaBusy());
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT("Unexpected result: %08x\n", (NN_RESULT_CURRENT_RESULT).GetInnerValueForDebug());
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}


::nn::Result PalmaManager::PairPalma() NN_NOEXCEPT
{
    NN_RESULT_DO(::nn::xcd::PairPalma(m_Handle));
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
