﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>

#include "hid_NpadUniquePadConverter.h"
#include "hid_UniquePadUtility.h"

namespace nn { namespace hid { namespace detail {

namespace
{
int GetNpadControllerNumber(NpadIdType& id)
{
    switch (id)
    {
    case NpadId::No1:
    case NpadId::No2:
    case NpadId::No3:
    case NpadId::No4:
    case NpadId::No5:
    case NpadId::No6:
    case NpadId::No7:
    case NpadId::No8:
        return id + 1;
    default:
        return 0;
    }
}

}

void NpadUniquePadConverter::SetUniquePadResourceManager(UniquePadResourceManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    m_pUniquePadResourceManager = pManager;
}

int NpadUniquePadConverter::GetUniquePads(system::UniquePadId* pOutValues, int count, NpadIdType& npadId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValues);
    NN_SDK_REQUIRES_GREATER(count, 0);

    int returnCount = 0;

    system::UniquePadId uniquePadIds[system::UniquePadIdCountMax];
    auto outIdCount = this->GetUniquePadIds(uniquePadIds, NN_ARRAY_SIZE(uniquePadIds));

    for (int i = 0; i < outIdCount; i++)
    {
        int number;
        if (m_pUniquePadResourceManager->GetControllerNumber(&number, uniquePadIds[i]).IsSuccess())
        {
            if (m_pUniquePadResourceManager->IsConnected(uniquePadIds[i]) &&
                number == GetNpadControllerNumber(npadId))
            {
                pOutValues[returnCount] = uniquePadIds[i];
                returnCount++;

                if (returnCount == count)
                {
                    break;
                }
            }
        }
    }

    return returnCount;
}

bool NpadUniquePadConverter::GetUniquePads(system::UniquePadId* pOutValue, NpadIdType& id, system::NpadDeviceTypeSet& deviceType) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_EQUAL(deviceType.CountPopulation(), 1);

    system::UniquePadId uniquePadIds[system::UniquePadIdCountMax];
    auto outIdCount = this->GetUniquePadIds(uniquePadIds, NN_ARRAY_SIZE(uniquePadIds));

    for (int i = 0; i < outIdCount; i++)
    {
        int number;
        system::UniquePadType type;
        if (m_pUniquePadResourceManager->GetControllerNumber(&number, uniquePadIds[i]).IsSuccess() &&
            m_pUniquePadResourceManager->GetType(&type, uniquePadIds[i]).IsSuccess())
        {
            if (m_pUniquePadResourceManager->IsConnected(uniquePadIds[i]) &&
                number == GetNpadControllerNumber(id) &&
                type == ConvertToUniquePadType(deviceType))
            {
                *pOutValue = uniquePadIds[i];
                return true;
            }
        }
    }

    return false;
}

int NpadUniquePadConverter::GetUniquePadIds(system::UniquePadId* pOutValues, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValues);
    NN_SDK_REQUIRES_GREATER(count, 0);

    int outIdCount;
    if (m_pUniquePadResourceManager->GetIds(&outIdCount, pOutValues, count).IsFailure())
    {
        return 0;
    }

    return outIdCount;
}

}}} // namespace nn::hid::detail
