﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>
#include <type_traits>
#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>
#include <nn/hid/hid_NpadColor.h>
#include <nn/hid/hid_NpadCommonTypes.h>
#include <nn/hid/hid_NpadFullKey.h>
#include <nn/hid/hid_NpadHandheld.h>
#include <nn/hid/hid_NpadJoyCommon.h>
#include <nn/hid/hid_NpadJoyDual.h>
#include <nn/hid/hid_NpadJoyLeft.h>
#include <nn/hid/hid_NpadJoyRight.h>
#include <nn/hid/hid_NpadPalma.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/hid/hid_Result.h>
#include <nn/hid/hid_ResultController.h>
#include <nn/hid/system/hid_Irsensor.h>
#include <nn/hid/system/hid_Nfc.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/os/os_Mutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nn/xcd/xcd_Device.h>

#include "hid_RingLifo.h"

namespace nn { namespace hid { namespace detail {

class NpadFullKeyLifo final : public RingLifo<NpadFullKeyState, NpadStateCountMax>
{
public:
    NpadFullKeyLifo() NN_NOEXCEPT { /* 何もしない */ }
};

#ifdef NN_BUILD_CONFIG_TOOLCHAIN_CLANG
NN_STATIC_ASSERT(::std::is_standard_layout<NpadFullKeyLifo>::value);
#endif

class NpadHandheldLifo final : public RingLifo<NpadHandheldState, NpadStateCountMax>
{
public:
    NpadHandheldLifo() NN_NOEXCEPT { /* 何もしない */ }
};

#ifdef NN_BUILD_CONFIG_TOOLCHAIN_CLANG
NN_STATIC_ASSERT(::std::is_standard_layout<NpadHandheldLifo>::value);
#endif

class NpadJoyDualLifo final : public RingLifo<NpadJoyDualState, NpadStateCountMax>
{
public:
    NpadJoyDualLifo() NN_NOEXCEPT { /* 何もしない */ }
};

#ifdef NN_BUILD_CONFIG_TOOLCHAIN_CLANG
NN_STATIC_ASSERT(::std::is_standard_layout<NpadJoyDualLifo>::value);
#endif

class NpadJoyLeftLifo final : public RingLifo<NpadJoyLeftState, NpadStateCountMax>
{
public:
    NpadJoyLeftLifo() NN_NOEXCEPT { /* 何もしない */ }
};

#ifdef NN_BUILD_CONFIG_TOOLCHAIN_CLANG
NN_STATIC_ASSERT(::std::is_standard_layout<NpadJoyLeftLifo>::value);
#endif

class NpadJoyRightLifo final : public RingLifo<NpadJoyRightState, NpadStateCountMax>
{
public:
    NpadJoyRightLifo() NN_NOEXCEPT { /* 何もしない */ }
};

#ifdef NN_BUILD_CONFIG_TOOLCHAIN_CLANG
NN_STATIC_ASSERT(::std::is_standard_layout<NpadJoyRightLifo>::value);
#endif

class NpadPalmaLifo final : public RingLifo<NpadPalmaState, NpadStateCountMax>
{
public:
    NpadPalmaLifo() NN_NOEXCEPT { /* 何もしない */ }
};

#ifdef NN_BUILD_CONFIG_TOOLCHAIN_CLANG
NN_STATIC_ASSERT(::std::is_standard_layout<NpadPalmaLifo>::value);
#endif

class NpadSixAxisSensorLifo final : public RingLifo<SixAxisSensorState, SixAxisSensorStateCountMax>
{
public:
    NpadSixAxisSensorLifo() NN_NOEXCEPT { /* 何もしない */ }
};

#ifdef NN_BUILD_CONFIG_TOOLCHAIN_CLANG
NN_STATIC_ASSERT(::std::is_standard_layout<NpadSixAxisSensorLifo>::value);
#endif

class NpadSystemLifo final : public RingLifo<system::NpadSystemState, NpadStateCountMax>
{
public:
    NpadSystemLifo() NN_NOEXCEPT { /* 何もしない */ }
};

#ifdef NN_BUILD_CONFIG_TOOLCHAIN_CLANG
NN_STATIC_ASSERT(::std::is_standard_layout<NpadSystemLifo>::value);
#endif

class NpadSystemExtLifo final : public RingLifo<system::NpadSystemExtState, NpadStateCountMax>
{
public:
    NpadSystemExtLifo() NN_NOEXCEPT { /* 何もしない */ }
};

#ifdef NN_BUILD_CONFIG_TOOLCHAIN_CLANG
NN_STATIC_ASSERT(::std::is_standard_layout<NpadSystemExtLifo>::value);
#endif

struct NpadGcTriggerState
{
    int64_t samplingNumber;
    int32_t triggerL;
    int32_t triggerR;
};

class NpadGcTriggerLifo final : public RingLifo<NpadGcTriggerState, NpadStateCountMax>
{
public:
    NpadGcTriggerLifo() NN_NOEXCEPT { /* 何もしない */ }
};

#ifdef NN_BUILD_CONFIG_TOOLCHAIN_CLANG
NN_STATIC_ASSERT(::std::is_standard_layout<NpadGcTriggerLifo>::value);
#endif

enum ColorAttribute : uint32_t
{
    ColorAttribute_Ok,            //!< 色情報が正常に取得できる
    ColorAttribute_ReadError,     //!< 色情報の読み出しができない状態
    ColorAttribute_NoController,  //!< コントローラーが接続されていない
};


struct NpadFullKeyColorState
{
    ColorAttribute attribute;     //!< 色情報の状態
    NpadControllerColor fullKey;  //!< フルキーコンの色情報
};

struct NpadJoyColorState
{
    ColorAttribute attribute;     //!< 色情報の状態
    NpadControllerColor left;     //!< 左コンの色情報
    NpadControllerColor right;    //!< 右コンの色情報
};

/**
 * @brief       本体機能用の内部状態
 */
struct NpadSystemProperties
{
    typedef ::nn::util::BitFlagSet<64, NpadSystemProperties>::Flag<0>  IsChargingStandard;             //!< 標準的なコントローラーの充電状態
    typedef ::nn::util::BitFlagSet<64, NpadSystemProperties>::Flag<1>  IsChargingLeft;                 //!< 左コンの充電状態
    typedef ::nn::util::BitFlagSet<64, NpadSystemProperties>::Flag<2>  IsChargingRight;                //!< 右コンの充電状態
    typedef ::nn::util::BitFlagSet<64, NpadSystemProperties>::Flag<3>  IsPoweredStandard;              //!< 標準的なコントローラーの給電状態
    typedef ::nn::util::BitFlagSet<64, NpadSystemProperties>::Flag<4>  IsPoewredLeft;                  //!< 左コンの給電状態
    typedef ::nn::util::BitFlagSet<64, NpadSystemProperties>::Flag<5>  IsPoweredRight;                 //!< 右コンの給電状態
    typedef ::nn::util::BitFlagSet<64, NpadSystemProperties>::Flag<6>  Reserved1;                      //!< Optimization requires this.
    typedef ::nn::util::BitFlagSet<64, NpadSystemProperties>::Flag<7>  Reserved2;                      //!< Optimization requires this.

    typedef ::nn::util::BitFlagSet<64, NpadSystemProperties>::Flag<9>  IsUnSupportedButtonPressed;     //!< NpadStyleSystem の非サポートボタンの押下状態
    typedef ::nn::util::BitFlagSet<64, NpadSystemProperties>::Flag<10> IsUnSupportedButtonPressedExt;  //!< NpadStyleSystemExt の非サポートボタンの押下状態
    typedef ::nn::util::BitFlagSet<64, NpadSystemProperties>::Flag<11> IsAbxyOriented;                 //!< ABXY 表記に対応しているかどうか
    typedef ::nn::util::BitFlagSet<64, NpadSystemProperties>::Flag<12> IsSlSrOriented;                 //!< SLSR 表記に対応しているかどうか
    typedef ::nn::util::BitFlagSet<64, NpadSystemProperties>::Flag<13> IsPlusAvailable;                //!< Plus ボタンをもったコントローラーか
    typedef ::nn::util::BitFlagSet<64, NpadSystemProperties>::Flag<14> IsMinusAvailable;               //!< Minus ボタンをもったコントローラーか
};

/**
 * @brief       Nfc の Xcd のデバイスハンドルを表す構造体
 */
struct NfcXcdDeviceHandleStateImpl
{
    ::nn::xcd::DeviceHandle handle;         //!< Nfc のデバイスハンドル
    bool isAvailable;                       //!< Nfc のが存在するかどうか
    bool isActivated;                       //!< Nfc の有効無効状態
    int64_t samplingNumber; // RingLifo 互換
};

class NfcXcdDeviceHandleState final : public RingLifo<NfcXcdDeviceHandleStateImpl, 1>
{
public:
    NfcXcdDeviceHandleState() NN_NOEXCEPT { /* 何もしない */ }
};

#ifdef NN_BUILD_CONFIG_TOOLCHAIN_CLANG
NN_STATIC_ASSERT(::std::is_standard_layout<NfcXcdDeviceHandleState>::value);
#endif

/**
 * @brief       Npad の内部状態を表す Bit 集合です
 */
typedef ::nn::util::BitFlagSet<64, NpadSystemProperties> NpadSystemPropertiesSet;

/**
 * @brief       システムボタンの内部状態
 */
struct NpadSystemButtonProperties
{
    typedef ::nn::util::BitFlagSet<16, NpadSystemButtonProperties>::Flag<0>  IsHomeButtonEnabled; //!< ホームボタンが有効かどうか
};

/**
 * @brief       システムボタンの内部状態を表す Bit 集合です
 */
typedef ::nn::util::BitFlagSet<16, NpadSystemButtonProperties> NpadSystemButtonPropertiesSet;

class NpadInternalState final
{
    NN_DISALLOW_COPY(NpadInternalState);
    NN_DISALLOW_MOVE(NpadInternalState);

private:
    std::atomic<NpadStyleSet> m_StyleSet;
    std::atomic<int32_t> m_JoyAssignmentMode;
    NpadFullKeyColorState m_FullKeyColor;
    NpadJoyColorState m_JoyColor;

    NpadFullKeyLifo m_FullKeyLifo;
    NpadHandheldLifo m_HandheldLifo;
    NpadJoyDualLifo m_JoyDualLifo;
    NpadJoyLeftLifo m_JoyLeftLifo;
    NpadJoyRightLifo m_JoyRightLifo;
    NpadPalmaLifo m_PalmaLifo;
    NpadSystemExtLifo m_SystemExtLifo;

    NpadSixAxisSensorLifo m_FullKeySixAxisSensorLifo;
    NpadSixAxisSensorLifo m_HandheldSixAxisSensorLifo;
    NpadSixAxisSensorLifo m_JoyDualLeftSixAxisSensorLifo;
    NpadSixAxisSensorLifo m_JoyDualRightSixAxisSensorLifo;
    NpadSixAxisSensorLifo m_JoyLeftSixAxisSensorLifo;
    NpadSixAxisSensorLifo m_JoyRightSixAxisSensorLifo;

    std::atomic<system::NpadDeviceTypeSet> m_DeviceType;
    std::atomic<NpadSystemPropertiesSet> m_SystemProperties;
    std::atomic<NpadSystemButtonPropertiesSet> m_SystemButtonProperties;
    std::atomic<int32_t> m_BatteryLevels[3];

    NfcXcdDeviceHandleState m_NfcXcdDeviceHandle;  // NotUsed

    nn::os::Mutex m_Mutex;  // NotUsed

#ifdef NN_BUILD_CONFIG_ADDRESS_32
    nn::Bit8 m_Padding[8];
#endif

    NpadGcTriggerLifo m_GcTriggerLifo;

public:
    enum BatteryIndex
    {
        BatteryIndex_Standard = 0,
        BatteryIndex_Left,
        BatteryIndex_Right,
    };

    NpadInternalState() NN_NOEXCEPT: m_StyleSet()
              , m_JoyAssignmentMode()
              , m_FullKeyColor()
              , m_JoyColor()
              , m_FullKeyLifo()
              , m_HandheldLifo()
              , m_JoyDualLifo()
              , m_JoyLeftLifo()
              , m_JoyRightLifo()
              , m_PalmaLifo()
              , m_SystemExtLifo()
              , m_FullKeySixAxisSensorLifo()
              , m_HandheldSixAxisSensorLifo()
              , m_JoyDualLeftSixAxisSensorLifo()
              , m_JoyDualRightSixAxisSensorLifo()
              , m_JoyLeftSixAxisSensorLifo()
              , m_JoyRightSixAxisSensorLifo()
              , m_DeviceType()
              , m_SystemProperties()
              , m_SystemButtonProperties()
              , m_NfcXcdDeviceHandle()
              , m_Mutex(false)
#ifdef NN_BUILD_CONFIG_ADDRESS_32
              , m_Padding()
#endif
              , m_GcTriggerLifo()
    {
        NpadSystemPropertiesSet clearedBits;
        clearedBits.Reset();
        m_SystemProperties = clearedBits;

        NpadSystemButtonPropertiesSet systemButtonBits;
        systemButtonBits.Reset();
        m_SystemButtonProperties = systemButtonBits;

        for (auto& battery : m_BatteryLevels)
        {
            battery = static_cast<int32_t>(system::NpadBatteryLevel_None);
        }

        ClearNfcXcdDeviceHandle();
    }
     ~NpadInternalState() NN_NOEXCEPT {}

    NpadStyleSet GetNpadStyleSet() NN_NOEXCEPT { return m_StyleSet; }
    void SetNpadStyleSet(NpadStyleSet style) NN_NOEXCEPT { m_StyleSet = style; }
    NpadJoyAssignmentMode GetNpadJoyAssignmentMode() NN_NOEXCEPT
    {
        switch (m_JoyAssignmentMode)
        {
        case NpadJoyAssignmentMode_Dual:
            return NpadJoyAssignmentMode_Dual;
        case NpadJoyAssignmentMode_Single:
            return NpadJoyAssignmentMode_Single;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
    void SetNpadJoyAssignmentMode(NpadJoyAssignmentMode assignment) NN_NOEXCEPT{ m_JoyAssignmentMode = static_cast<int32_t>(assignment);  }

    NpadFullKeyLifo* GetFullKeyLifo() NN_NOEXCEPT { return &m_FullKeyLifo; }
    NpadHandheldLifo* GetHandheldLifo() NN_NOEXCEPT { return &m_HandheldLifo; }
    NpadJoyDualLifo* GetJoyDualLifo() NN_NOEXCEPT { return &m_JoyDualLifo; }
    NpadJoyLeftLifo* GetJoyLeftLifo() NN_NOEXCEPT { return &m_JoyLeftLifo; }
    NpadJoyRightLifo* GetJoyRightLifo() NN_NOEXCEPT { return &m_JoyRightLifo; }
    NpadSystemExtLifo* GetSystemExtLifo() NN_NOEXCEPT { return &m_SystemExtLifo; }
    NpadGcTriggerLifo* GetGcTriggerLifo() NN_NOEXCEPT { return &m_GcTriggerLifo; }
    NpadPalmaLifo* GetPalmaLifo() NN_NOEXCEPT { return &m_PalmaLifo; }
    void GetLifo(NpadFullKeyLifo** pOutValue) NN_NOEXCEPT { *pOutValue =  &m_FullKeyLifo; }
    void GetLifo(NpadHandheldLifo** pOutValue) NN_NOEXCEPT { *pOutValue = &m_HandheldLifo; }
    void GetLifo(NpadJoyDualLifo** pOutValue) NN_NOEXCEPT { *pOutValue = &m_JoyDualLifo; }
    void GetLifo(NpadJoyLeftLifo** pOutValue) NN_NOEXCEPT { *pOutValue = &m_JoyLeftLifo; }
    void GetLifo(NpadJoyRightLifo** pOutValue) NN_NOEXCEPT { *pOutValue = &m_JoyRightLifo; }
    void GetLifo(NpadPalmaLifo** pOutValue) NN_NOEXCEPT { *pOutValue = &m_PalmaLifo; }
    void GetLifo(NpadSystemExtLifo** pOutValue) NN_NOEXCEPT { *pOutValue = &m_SystemExtLifo; }
    NpadSixAxisSensorLifo* GetFullKeySixAxisSensorLifo() NN_NOEXCEPT { return &m_FullKeySixAxisSensorLifo; }
    NpadSixAxisSensorLifo* GetHandheldSixAxisSensorLifo() NN_NOEXCEPT { return &m_HandheldSixAxisSensorLifo; }
    NpadSixAxisSensorLifo* GetJoyDualLeftSixAxisSensorLifo() NN_NOEXCEPT { return &m_JoyDualLeftSixAxisSensorLifo; }
    NpadSixAxisSensorLifo* GetJoyDualRightSixAxisSensorLifo() NN_NOEXCEPT { return &m_JoyDualRightSixAxisSensorLifo; }
    NpadSixAxisSensorLifo* GetJoyLeftSixAxisSensorLifo() NN_NOEXCEPT { return &m_JoyLeftSixAxisSensorLifo; }
    NpadSixAxisSensorLifo* GetJoyRightSixAxisSensorLifo() NN_NOEXCEPT { return &m_JoyRightSixAxisSensorLifo; }

    void SetNpadFullKeyColor(const NpadFullKeyColorState& color) NN_NOEXCEPT {
        m_FullKeyColor = color;
    }
    void SetNpadJoyColor(const NpadJoyColorState& color) NN_NOEXCEPT {
        m_JoyColor = color; }

    ::nn::Result GetNpadControllerColor(NpadControllerColor* pOutValue) NN_NOEXCEPT
    {
        NpadFullKeyColorState color = m_FullKeyColor;
        switch (color.attribute)
        {
        case ColorAttribute_Ok:
            *pOutValue = color.fullKey;
            break;
        case ColorAttribute_NoController:
            NN_RESULT_THROW(ResultNpadControllerNotConnected());
        case ColorAttribute_ReadError:
            NN_RESULT_THROW(ResultNpadColorNotAvailable());
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        NN_RESULT_SUCCESS;
    }

    ::nn::Result GetNpadControllerColor(NpadControllerColor* pOutLeftColor, NpadControllerColor* pOutRightColor) NN_NOEXCEPT
    {
        NpadJoyColorState color = m_JoyColor;

        switch (color.attribute)
        {
        case ColorAttribute_Ok:
            *pOutLeftColor = color.left;
            *pOutRightColor = color.right;
            break;
        case ColorAttribute_NoController:
            NN_RESULT_THROW(ResultNpadControllerNotConnected());
        case ColorAttribute_ReadError:
            NN_RESULT_THROW(ResultNpadColorNotAvailable());
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        NN_RESULT_SUCCESS;
    }

    void Clear() NN_NOEXCEPT
    {
        m_FullKeyLifo.Clear();
        m_HandheldLifo.Clear();
        m_JoyDualLifo.Clear();
        m_JoyLeftLifo.Clear();
        m_JoyRightLifo.Clear();
        m_PalmaLifo.Clear();
        m_SystemExtLifo.Clear();
        m_GcTriggerLifo.Clear();
        m_FullKeySixAxisSensorLifo.Clear();
        m_HandheldSixAxisSensorLifo.Clear();
        m_JoyDualLeftSixAxisSensorLifo.Clear();
        m_JoyDualRightSixAxisSensorLifo.Clear();
        m_JoyLeftSixAxisSensorLifo.Clear();
        m_JoyRightSixAxisSensorLifo.Clear();

        m_StyleSet = NpadStyleSet();
        m_JoyAssignmentMode = NpadJoyAssignmentMode_Dual;
        m_FullKeyColor = NpadFullKeyColorState();
        m_JoyColor = NpadJoyColorState();
        NpadSystemPropertiesSet clearedBits;
        clearedBits.Reset();
        m_SystemProperties = clearedBits;

        NpadSystemButtonPropertiesSet systemButtonBits;
        systemButtonBits.Reset();
        m_SystemButtonProperties = systemButtonBits;

        m_DeviceType = system::NpadDeviceTypeSet();

        for (auto& battery : m_BatteryLevels)
        {
            battery = static_cast<int32_t>(system::NpadBatteryLevel_None);
        }
    }

    system::NpadDeviceTypeSet GetNpadDeviceTypeSet() NN_NOEXCEPT { return m_DeviceType; }
    void SetNpadDeviceTypeSet(system::NpadDeviceTypeSet deviceType) NN_NOEXCEPT { m_DeviceType = deviceType; }

    bool IsAbxyButtonOriented() NN_NOEXCEPT
    {
        return m_SystemProperties.load().Test<NpadSystemProperties::IsAbxyOriented>();
    }

    bool IsSlSrButtonOriented() NN_NOEXCEPT
    {
        return m_SystemProperties.load().Test<NpadSystemProperties::IsSlSrOriented>();
    }

    void SetButtonOriented(bool isAbxyOriented, bool isSlSrOriented) NN_NOEXCEPT
    {
        // Because the compiler cannot deduce that this type is actually just a uint64_t,
        // cast it to the right size, but error check with a compile-time assert.
        NN_STATIC_ASSERT(sizeof(m_SystemProperties) == sizeof(uint64_t));
        std::atomic<uint64_t> &SystemProperites = (std::atomic<uint64_t> &)m_SystemProperties;

        if(isAbxyOriented != isSlSrOriented)
        {
            uint64_t tempAbxy  = (1 << NpadSystemProperties::IsAbxyOriented::Index);
            uint64_t tempSls = (1 << NpadSystemProperties::IsSlSrOriented::Index);

            uint64_t expected;
            uint64_t newValue;

            if(isAbxyOriented)
            {
                do
                {
                    expected = SystemProperites.load();
                    newValue = (expected | tempAbxy) & ~tempSls;
                }
                while (! SystemProperites.compare_exchange_weak(expected, newValue));

                // Experiment to see if this removes a branch to return in asm code.
                return;
            }
            else
            {
                do
                {
                    expected =  SystemProperites.load();
                    newValue = (expected | tempSls) & ~tempAbxy;
                }
                while (! SystemProperites.compare_exchange_weak(expected, newValue));

                // Experiment to see if this removes a branch to return in asm code.
                return;
            }
        }
        else
        {
            uint64_t temp  = (1 << NpadSystemProperties::IsAbxyOriented::Index) |
                             (1 << NpadSystemProperties::IsSlSrOriented::Index);

            if(isAbxyOriented)
            {
                SystemProperites.fetch_or(temp);
            }
            else
            {
                SystemProperites.fetch_and(~temp);
            }
        }
    }

    bool IsUnsuportedButtonPressedOnNpadSystem() NN_NOEXCEPT
    {
        return m_SystemProperties.load().Test<NpadSystemProperties::IsUnSupportedButtonPressed>();
    }

    void SetUnsupportedButtonStateOnNpadSystem(bool isPressed) NN_NOEXCEPT
    {
        // As above, compile-time verify size before typecast.
        NN_STATIC_ASSERT(sizeof(m_SystemProperties) == sizeof(uint64_t));
        std::atomic<uint64_t> &SystemProperites = (std::atomic<uint64_t> &)m_SystemProperties;

        uint64_t temp  = 1 << NpadSystemProperties::IsUnSupportedButtonPressed::Index;
        if(isPressed)
        {
            SystemProperites.fetch_or(temp);
        }
        else
        {
            SystemProperites.fetch_and(~temp);
        }
    }

    void SetUnsupportedButtonStateOnNpadSystemExt(bool isPressed) NN_NOEXCEPT
    {
        // As above, compile-time verify size before typecast.
        NN_STATIC_ASSERT(sizeof(m_SystemProperties) == sizeof(uint64_t));
        std::atomic<uint64_t> &SystemProperites = (std::atomic<uint64_t> &)m_SystemProperties;

        uint64_t temp  = 1 << NpadSystemProperties::IsUnSupportedButtonPressedExt::Index;
        if(isPressed)
        {
            SystemProperites.fetch_or(temp);
        }
        else
        {
            SystemProperites.fetch_and(~temp);
        }
    }

    bool IsUnsuportedButtonPressedOnNpadSystemExt() NN_NOEXCEPT
    {
        return m_SystemProperties.load().Test<NpadSystemProperties::IsUnSupportedButtonPressedExt>();
    }

    bool IsPlusAvailable() NN_NOEXCEPT
    {
        return m_SystemProperties.load().Test<NpadSystemProperties::IsPlusAvailable>();
    }

    void SetIsPlusAvailable(bool isAvailable) NN_NOEXCEPT
    {
        this->SetSystemProperties(NpadSystemProperties::IsPlusAvailable::Index, isAvailable);
    }

    bool IsMinusAvailable() NN_NOEXCEPT
    {
        return m_SystemProperties.load().Test<NpadSystemProperties::IsMinusAvailable>();
    }

    void SetIsMinusAvailable(bool isAvailable) NN_NOEXCEPT
    {
        this->SetSystemProperties(NpadSystemProperties::IsMinusAvailable::Index, isAvailable);
    }

    void SetSystemProperties(int index, bool flag) NN_NOEXCEPT
    {
        // As above, compile-time verify size before typecast.
        NN_STATIC_ASSERT(sizeof(m_SystemProperties) == sizeof(uint64_t));
        std::atomic<uint64_t> &SystemProperites = (std::atomic<uint64_t> &)m_SystemProperties;

        uint64_t temp  = 1ll << index;

        if(flag)
        {
            SystemProperites.fetch_or(temp);
        }
        else
        {
            SystemProperites.fetch_and(~temp);
        }
    }

    system::NpadPowerInfo GetNpadPowerInfo(BatteryIndex index) NN_NOEXCEPT
    {
        system::NpadPowerInfo powerInfo;

        switch (m_BatteryLevels[index])
        {
        case system::NpadBatteryLevel_CriticalLow:
            powerInfo.batteryLevel = system::NpadBatteryLevel_CriticalLow;
            break;
        case system::NpadBatteryLevel_Low:
            powerInfo.batteryLevel = system::NpadBatteryLevel_Low;
            break;
        case system::NpadBatteryLevel_Midium:
            powerInfo.batteryLevel = system::NpadBatteryLevel_Midium;
            break;
        case system::NpadBatteryLevel_High:
            powerInfo.batteryLevel = system::NpadBatteryLevel_High;
            break;
        case system::NpadBatteryLevel_None:
            powerInfo.batteryLevel = system::NpadBatteryLevel_None;
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
        powerInfo.isCharging = m_SystemProperties.load().Test(NpadSystemProperties::IsChargingStandard::Index + index);
        powerInfo.isPowered = m_SystemProperties.load().Test(NpadSystemProperties::IsPoweredStandard::Index + index);

        return powerInfo;
    }

    void SetHomeButtonEnabled(bool enabled) NN_NOEXCEPT
    {
        // As above, compile-time verify size before typecast.
        NN_STATIC_ASSERT(sizeof(m_SystemButtonProperties) == sizeof(uint32_t));
        std::atomic<uint32_t> &SystemButtonProperites = (std::atomic<uint32_t> &)m_SystemButtonProperties;

        uint32_t temp  = 1 << NpadSystemButtonProperties::IsHomeButtonEnabled::Index;

        // m_SystemButtonProperties internally is a uint32_t value.
        if(enabled)
        {
            SystemButtonProperites.fetch_or(temp);
        }
        else
        {
            SystemButtonProperites.fetch_and(~temp);
        }
    }

    bool IsHomeButtonEnabled() const NN_NOEXCEPT
    {
        return m_SystemButtonProperties.load().Test<NpadSystemButtonProperties::IsHomeButtonEnabled>();
    }

    // Now unused, but left for compatibility.
    void SetNpadPowerInfo(system::NpadPowerInfo powerInfo, BatteryIndex index) NN_NOEXCEPT
    {
        m_BatteryLevels[index] = static_cast<int32_t>(powerInfo.batteryLevel);
        SetSystemProperties(NpadSystemProperties::IsChargingStandard::Index + index, powerInfo.isCharging);
        SetSystemProperties(NpadSystemProperties::IsPoweredStandard::Index + index, powerInfo.isPowered);
    }

    // This function requires top performance due to the number of times it is called per second.
    void SetNpadPowerInfo(const system::NpadPowerInfo &powerInfoStandard,
                          const system::NpadPowerInfo &powerInfoLeft,
                          const system::NpadPowerInfo &powerInfoRight) NN_NOEXCEPT
    {
        uint8_t isChargingPoweredMask = 0;

        // Build a mask from the powerInfoStandard.
        if(powerInfoStandard.isCharging)
        {
            isChargingPoweredMask |= 1 << (0 + 0);
        }

        if(powerInfoStandard.isPowered)
        {
            isChargingPoweredMask |= 1 << (0 + 3);
        }

        // Build a mask from the powerInfoLeft.
        if(powerInfoLeft.isCharging)
        {
            isChargingPoweredMask |= 1 << (1 + 0);
        }

        if(powerInfoLeft.isPowered)
        {
            isChargingPoweredMask |= 1 << (1 + 3);
        }

        // Build a mask from the powerInfoRight.
        if(powerInfoRight.isCharging)
        {
            isChargingPoweredMask |= 1 << (2 + 0);
        }

        if(powerInfoRight.isPowered)
        {
            isChargingPoweredMask |= 1 << (2 + 3);
        }

        // Use the fact that the mask is the only thing in the low 8-bits
        // To do a direct store instead of 6 or/and combos with locks.
        // Note - we do not care what was in the values so a CAS is not needed and there are
        // also external locks to help as well.
        ((std::atomic<uint8_t> &)m_SystemProperties).store(isChargingPoweredMask);

        // Only 2 or 3 bits per battery level are used, but we must write them all.
        m_BatteryLevels[0] = powerInfoStandard.batteryLevel;
        m_BatteryLevels[1] = powerInfoLeft.batteryLevel;
        m_BatteryLevels[2] = powerInfoRight.batteryLevel;
    }

    Result GetNfcXcdDeviceHandle(nn::xcd::DeviceHandle* pOutHandle) NN_NOEXCEPT
    {
        NfcXcdDeviceHandleStateImpl handle;
        m_NfcXcdDeviceHandle.Read(&handle, 1);

        if (handle.isAvailable == false)
        {
            NN_RESULT_THROW(system::ResultNoNfcDeviceFoundOnNpad());
        }

        *pOutHandle = handle.handle;
        NN_RESULT_SUCCESS;
    }
    void SetNfcXcdDeviceHandle(nn::xcd::DeviceHandle xcdDeviceHandle) NN_NOEXCEPT
    {
        NfcXcdDeviceHandleStateImpl handle;
        m_NfcXcdDeviceHandle.Read(&handle, 1);
        handle.handle = xcdDeviceHandle;
        handle.samplingNumber++;

        m_NfcXcdDeviceHandle.Append(handle);
    }
    void ClearNfcXcdDeviceHandle() NN_NOEXCEPT
    {
        m_NfcXcdDeviceHandle.Append(NfcXcdDeviceHandleStateImpl());
    }

    bool IsNfcActivated() NN_NOEXCEPT
    {
        NfcXcdDeviceHandleStateImpl handle;
        m_NfcXcdDeviceHandle.Read(&handle, 1);

        return (handle.isActivated && handle.isAvailable);
    }

    bool IsNfcAvailable() NN_NOEXCEPT
    {
        NfcXcdDeviceHandleStateImpl handle;
        m_NfcXcdDeviceHandle.Read(&handle, 1);

        return handle.isAvailable;
    }

    void SetNfcState(bool isAvailable, bool isActivated) NN_NOEXCEPT
    {
        NfcXcdDeviceHandleStateImpl handle;
        m_NfcXcdDeviceHandle.Read(&handle, 1);
        handle.isAvailable = isAvailable;
        handle.isActivated = isActivated;
        handle.samplingNumber++;

        m_NfcXcdDeviceHandle.Append(handle);
    }
};

#ifdef NN_BUILD_CONFIG_TOOLCHAIN_CLANG
NN_STATIC_ASSERT(::std::is_standard_layout<NpadInternalState>::value);
#endif

}}} // namespace nn::hid::detail
