﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoyCommon.h>
#include <nn/hid/system/hid_NpadCommon.h>

#include "hid_ActivationCount.h"
#include "hid_NpadAbstractedPadHolder.h"
#include "hid_NpadCommonResourceHolder.h"
#include "hid_NpadDeviceManager.h"
#include "hid_SystemButtonNotifier.h"

namespace nn { namespace hid { namespace detail {

struct NpadCommonStateImpl
{
    int64_t samplingNumber;             //!< Npad の入力状態が更新される度に増加する値です。
    NpadButtonSet buttons;              //!< Npad のデジタルボタンの状態です。
    AnalogStickState analogStickL;      //!< Npad の左アナログスティックの状態です。
    AnalogStickState analogStickR;      //!< Npad の右アナログスティックの状態です。
    NpadAttributesSet attributes;       //!< Npad の状態を表す属性値です。
    bool isUpdated;                     //!< 入力値が更新されているかどうかを示します
};

//!< ボタンやアナログスティックなどの入力状態の更新を担うクラスです。
class NpadInputStateUpdater final
{
    NN_DISALLOW_COPY(NpadInputStateUpdater);
    NN_DISALLOW_MOVE(NpadInputStateUpdater);

private:
    //!< Npad が共通で扱う hid のサービスを保持するクラス
    NpadCommonResourceHolder* m_pCommonResourceHolder;

    //!< Npad 毎に割り当てられる AbstractedPad を保持するクラス
    NpadAbstractedPadHolder* m_pAbstractedPadHolder;

    //!< Npad の状態管理を担うマネージャー
    NpadDeviceManager* m_pDeviceManager;

    //!< アクティブ化された回数
    ActivationCount m_NpadActivationCount;

    //!< 撮影ボタンの入力通知を行う Notifier
    SystemButtonNotifier m_CaptureButtonNotifier;

    //!< ホームボタンの入力通知を行う Notifier
    SystemButtonNotifier m_HomeButtonNotifier;

    //!< 最新の Gc トリガーの入力状態
    NpadGcTriggerState m_GcTriggerState;

    //!< 最新の Home ボタンの入力状態
    bool m_IsHomeButtonDown;

    //!< Home 入力を無視するかどうか
    bool m_IgnoreHomeButton;

    //!< 最新の撮影ボタンの入力状態
    bool m_IsCaptureButtonDown;

    //!< NpadSystem における非サポートボタンの押下状態
    bool m_UnsupportedButtonStateForNpadSystem;

    //!< NpadSystemExt における非サポートボタンの押下状態
    bool m_UnsupportedButtonStateForNpadSystemExt;

    //!< 据置モード中の携帯機コントローラーのボタンの押下状態
    bool m_IsHandheldButtonPressedOnConsoleMode;

    //!< 入力変化通知のために用いる統合された入力状態
    AbstractedPadState m_IntegratedStateForInputDetect;

    //!< Npad のスタイル毎の入力状態
    NpadCommonStateImpl m_CommonStates[NpadStyleCountMax];

public:
    NpadInputStateUpdater() NN_NOEXCEPT;

    ~NpadInputStateUpdater() NN_NOEXCEPT;

    //!< NpadCommonResourceHolder を設定します。
    void SetNpadCommonResourceHolder(NpadCommonResourceHolder* pHolder) NN_NOEXCEPT;

    //!< NpadAbstractedPadHolder を設定します。
    void SetNpadAbstractedPadHolder(NpadAbstractedPadHolder* pHolder) NN_NOEXCEPT;

    //!< NpadDeviceManager を設定します。
    void SetNpadDeviceManager(NpadDeviceManager* pHolder) NN_NOEXCEPT;

    //!< 撮影ボタンの入力通知を行う Notifier を設定します。
    void SetCaptureButtonNotifier(SystemButtonNotifier value) NN_NOEXCEPT;

    //!< ホームボタンの入力通知を行う Notifier を設定します。
    void SetHomeButtonNotifier(SystemButtonNotifier value) NN_NOEXCEPT;

    //!< マネージャをアクティブ化します。
    ::nn::Result Activate() NN_NOEXCEPT;

    //!< マネージャを非アクティブ化します。
    ::nn::Result Deactivate() NN_NOEXCEPT;

    //!< Npad のアプレットリソースの状態を保証します。
    ::nn::Result EnsureNpadAppletResource(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 入力状態を更新します。
    void UpdateInputStates(bool force) NN_NOEXCEPT;

    //!< 共有メモリの状態を更新します
    void UpdateSharedMemory() NN_NOEXCEPT;

    //!< 共有メモリの状態を強制的に更新します
    void UpdateSharedMemoryForced() NN_NOEXCEPT;

    //!< 特定の Aruid に対して共有メモリ内部の状態を更新します
    void UpdateSharedMemoryForTarget(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< Home ボタンの有効/無効状態を切り替えます。内部で共有メモリの状態も更新します
    ::nn::Result SetHomeButtonEnabledWithSharedMemoryUpdate(bool enabled,
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 本体機能で扱える操作系が入力状態にあるか
    bool IsNpadSystemExtInputEnable() NN_NOEXCEPT;

    //!< 据置モード中の携帯機コントローラーの操作が行われたかどうか
    bool IsHandheldButtonPressedOnConsoleMode() NN_NOEXCEPT;

private:
    //!< 入力状態を更新する
    void UpdateStates(bool forceUpdate) NN_NOEXCEPT;

    //!< 共有メモリを処理します。
    void ProcessSharedMemory(
        void (*processor)(NpadInputStateUpdater* that,
                          NpadSharedMemoryEntry* address,
                          ::nn::applet::AppletResourceUserId aruid,
                          bool enablesInput) NN_NOEXCEPT) NN_NOEXCEPT;

    //!< 指定した Aruid のものについてのみ共有メモリを処理します。
    void ProcessSharedMemoryForTargetAruid(
        ::nn::applet::AppletResourceUserId aruid,
        void (*processor)(NpadInputStateUpdater* that,
                          NpadSharedMemoryEntry* address,
                          ::nn::applet::AppletResourceUserId aruid,
                          bool enablesInput) NN_NOEXCEPT) NN_NOEXCEPT;

    //!< 共有メモリの内容を更新します
    void UpdateSharedMemoryForInputs(NpadSharedMemoryEntry* address,
                            ::nn::applet::AppletResourceUserId aruid,
                            bool enablesInput,
                            bool updateOnlyWithEmpty,
                            bool forceUpdate) NN_NOEXCEPT;

    //!< 共有メモリをスタイル毎に更新します。
    template <typename TState, typename TLifo>
    void UpdateSharedMemoryForNpadState(NpadStyleSet currentStyle,
                                        const int& styleIndex,
                                        ::nn::applet::AppletResourceUserId aruid,
                                        NpadInternalState* pLifo,
                                        const NpadCommonStateImpl& state,
                                        bool enablesInput,
                                        bool updateOnlyWithEmpty,
                                        bool forceUpdate) NN_NOEXCEPT;

    //!< 非サポートボタンの入力状態を更新する
    void UpdateSharedMemoryForUnsupporetedButton(NpadInternalState* pInternalState,
                                                 bool enablesInput) NN_NOEXCEPT;


    //!< HomeButton の有効無効状態を更新する
    void UpdateSharedMemoryForHomeButtonEnabled(NpadInternalState* pInternalState,
                                                ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< システムボタンの状態を Notifier に通知する
    void NotifySystemButtons() NN_NOEXCEPT;

};

}}} // namespace nn::hid::detail
