﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/hid/hid_Mouse.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/debug/hid_Mouse.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_TypedStorage.h>

#include "hid_AppletResourceManager.h"
#include "hid_IMouseDriver.h"
#include "hid_MouseLifo.h"
#include "hid_MouseManager.h"
#include "hid_MouseStateUtility.h"
#include "hid_Point.h"
#include "hid_Rectangle.h"
#include "hid_SharedMemoryFormat.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< Mouse の入力状態を計算します。
void CalculateMouseState(MouseState* pOutValue,
                         const MouseState& state,
                         bool enablesInput,
                         MouseButtonSet& buttonMask) NN_NOEXCEPT;

} // namespace

const ::nn::TimeSpan MouseManager::SamplingInterval =
    ::nn::TimeSpan::FromMilliSeconds(8);

MouseManager::MouseManager() NN_NOEXCEPT
    : m_SamplingNumber(0)
    , m_pTimer(nullptr)
    , m_pDriver(nullptr)
    , m_pAppletResourceManager(nullptr)
    , m_pAppletResourceManagerMutex(nullptr)
    , m_State()
    , m_TempState()
    , m_TempAruid()
    , m_IsAutoPilotEnabled(false)
    , m_IsAutoPilotStateUpdated(false)
    , m_AutoPilotState()
{
    for (MouseAppletResourceEntry& extraEntry : m_AppletResourceEntries)
    {
        extraEntry = MouseAppletResourceEntry();
    }
}

void MouseManager::SetTimer(SynchronizedTimer* timer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(timer);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pTimer = timer;
}

void MouseManager::SetDriver(IMouseDriver* pDriver) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDriver);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pDriver = pDriver;
}

void MouseManager::SetAppletResourceManager(
    AppletResourceManager* pManager, ::nn::os::SdkRecursiveMutex* pMutex
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    NN_SDK_REQUIRES_NOT_NULL(pMutex);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pAppletResourceManager = pManager;
    m_pAppletResourceManagerMutex = pMutex;
}

void MouseManager::SetInputDetectorManager(
    InputDetectorManager* pManager, ::nn::os::SdkMutex* pMutex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    NN_SDK_REQUIRES_NOT_NULL(pMutex);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pInputDetectorManager = pManager;
    m_pInputDetectorManagerMutex = pMutex;
}

::nn::Result MouseManager::Activate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pTimer);
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultMouseActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施

        // ドライバをアクティブ化
        NN_RESULT_DO(m_pDriver->Activate());

        // LIFO を初期化
        this->ProcessSharedMemory([] (MouseManager* that,
                                      MouseSharedMemoryFormat* address,
                                      ::nn::applet::AppletResourceUserId aruid,
                                      bool enablesInput,
                                      MouseButtonSet& buttonMask) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(that);
            NN_SDK_REQUIRES_NOT_NULL(address);
            NN_UNUSED(that);
            NN_UNUSED(aruid);
            NN_UNUSED(enablesInput);
            NN_UNUSED(buttonMask);
            ::nn::util::Get(address->lifo).Clear();
        });

        // タイマーイベントをアクティブ化
        m_pTimer->Enable(SamplingInterval);

        m_State = MouseState();

        // 入力状態を更新
        this->Update();
    }

    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result MouseManager::Deactivate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pTimer);
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultMouseDeactivationLowerLimitOver());

    --m_ActivationCount;

    if (m_ActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施

        // 自動操作を無効化
        this->DisableAutoPilot();

        // タイマーイベントを非アクティブ化
        m_pTimer->Disable();

        // ドライバを非アクティブ化
        NN_RESULT_DO(m_pDriver->Deactivate());
    }

    NN_RESULT_SUCCESS;
}

::nn::Result MouseManager::EnsureAppletResource(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    m_TempAruid = aruid;

    this->ProcessSharedMemory([] (MouseManager* that,
                                  MouseSharedMemoryFormat* address,
                                  ::nn::applet::AppletResourceUserId haystack,
                                  bool enablesInput,
                                  MouseButtonSet& buttonMask) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(that);
        NN_SDK_REQUIRES_NOT_NULL(address);

        MouseState* pOutValue = &that->m_TempState;

        const MouseState& state = that->m_State;

        const ::nn::applet::AppletResourceUserId& needle = that->m_TempAruid;

        MouseLifo& lifo = ::nn::util::Get(address->lifo);

        if (lifo.IsEmpty() && needle == haystack)
        {
            CalculateMouseState(pOutValue, state, enablesInput, buttonMask);

            lifo.Append(*pOutValue);
        }
    });

    NN_RESULT_SUCCESS;
}

::nn::Result MouseManager::SetAutoPilotState(
    const ::nn::hid::debug::MouseAutoPilotState& value) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultMouseNotInitialized());

    m_IsAutoPilotEnabled = true;

    m_IsAutoPilotStateUpdated = true;

    m_AutoPilotState = value;

    Point point = { value.x, value.y };

    point = MouseRectangle.Clamp(point);

    m_AutoPilotState.x = point.x;
    m_AutoPilotState.y = point.y;

    NN_RESULT_SUCCESS;
}

::nn::Result MouseManager::UnsetAutoPilotState() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultMouseNotInitialized());
    this->DisableAutoPilot();
    NN_RESULT_SUCCESS;
}

void MouseManager::Sample() NN_NOEXCEPT
{
    if (!m_ActivationCount.IsZero())
    {
        this->Update();

        this->ProcessSharedMemory([] (MouseManager* that,
                                      MouseSharedMemoryFormat* address,
                                      ::nn::applet::AppletResourceUserId aruid,
                                      bool enablesInput,
                                      MouseButtonSet& buttonMask) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(that);
            NN_SDK_REQUIRES_NOT_NULL(address);
            NN_UNUSED(aruid);

            MouseState* pOutValue = &that->m_TempState;

            const MouseState& state = that->m_State;

            CalculateMouseState(pOutValue, state, enablesInput, buttonMask);

            ::nn::util::Get(address->lifo).Append(*pOutValue);
        });
    }
}

void MouseManager::Update() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);
    NN_SDK_REQUIRES_NOT_NULL(m_pInputDetectorManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pInputDetectorManagerMutex);

    // 入力変化検知用に状態を保存
    m_TempState = m_State;

    // 入力状態を取得
    m_pDriver->GetState(&m_State);

    // サンプリング番号を設定
    m_State.samplingNumber = m_SamplingNumber++;

    if (m_IsAutoPilotEnabled)
    {
        // 自動操作が有効な場合は自動操作状態をマージ

        m_State.x = m_AutoPilotState.x;
        m_State.y = m_AutoPilotState.y;
        m_State.deltaX = m_AutoPilotState.deltaX;
        m_State.deltaY = m_AutoPilotState.deltaY;
        m_State.wheelDelta = m_AutoPilotState.wheelDelta;
        m_State.buttons |= m_AutoPilotState.buttons;
        m_State.attributes |= m_AutoPilotState.attributes;
        m_State.attributes.Set<MouseAttribute::IsConnected>(
            m_State.deltaX != 0 ||
            m_State.deltaY != 0 ||
            m_State.wheelDelta != 0 ||
            m_State.buttons.IsAnyOn() ||
            m_AutoPilotState.attributes.Test<MouseAttribute::IsConnected>());
        m_State.attributes.Set<MouseAttribute::Transferable>(
            !m_State.attributes.Test<MouseAttribute::IsConnected>() ||
            m_AutoPilotState.attributes.Test<MouseAttribute::Transferable>());

        if (m_IsAutoPilotStateUpdated)
        {
            // 差分値が複数回参照されないよう消去
            m_AutoPilotState.deltaX = 0;
            m_AutoPilotState.deltaY = 0;
            m_AutoPilotState.wheelDelta = 0;
            m_IsAutoPilotStateUpdated = false;
        }
    }

    if (m_State.attributes.Test<MouseAttribute::Transferable>() ||
        m_TempState.attributes.Test<MouseAttribute::Transferable>())
    {
        // 上位システムによるカーソルの補足を跨いだ場合は差分値は無効

        m_State.deltaX = 0;
        m_State.deltaY = 0;
    }
    else
    {
        // 座標の差分値に整合性が取れていることを保証

        const int32_t minX = MouseRectangle.x;
        const int32_t maxX = MouseRectangle.x + MouseRectangle.width - 1;
        const int32_t deltaX = m_State.x - m_TempState.x;

        if ((m_State.deltaX > deltaX &&
             m_State.x != maxX && m_TempState.x != minX) ||
            (m_State.deltaX < deltaX &&
             m_State.x != minX && m_TempState.x != maxX))
        {
            m_State.deltaX = deltaX;
        }

        const int32_t minY = MouseRectangle.y;
        const int32_t maxY = MouseRectangle.y + MouseRectangle.height - 1;
        const int32_t deltaY = m_State.y - m_TempState.y;

        if ((m_State.deltaY > deltaY &&
             m_State.y != maxY && m_TempState.y != minY) ||
            (m_State.deltaY < deltaY &&
             m_State.y != minY && m_TempState.y != maxY))
        {
            m_State.deltaY = deltaY;
        }
    }

    ::std::lock_guard<decltype(*m_pInputDetectorManagerMutex)
                      > locker(*m_pInputDetectorManagerMutex);

    if (!m_State.attributes.Test<MouseAttribute::Transferable>() &&
        (m_State.deltaX != 0 ||
         m_State.deltaY != 0 ||
         m_State.wheelDelta != 0 ||
         (m_State.buttons ^ m_TempState.buttons).IsAnyOn()))
    {
        m_pInputDetectorManager->Notify(
            ::nn::hid::system::InputSourceId::Mouse::Mask);
    }
}

void MouseManager::ProcessSharedMemory(
    void (*processor)(MouseManager* that,
                      MouseSharedMemoryFormat* address,
                      ::nn::applet::AppletResourceUserId aruid,
                      bool enablesInput,
                      MouseButtonSet& buttonMask) NN_NOEXCEPT) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(processor);
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceManagerMutex);

    ::std::lock_guard<decltype(*m_pAppletResourceManagerMutex)
                      > locker(*m_pAppletResourceManagerMutex);

    const AppletResourceEntry (&entries)[AppletResourceEntryCountMax] =
        m_pAppletResourceManager->GetAppletResourceEntries();

    for (size_t i = 0; i < AppletResourceEntryCountMax; ++i)
    {
        const AppletResourceEntry& entry = entries[i];

        MouseAppletResourceEntry& extraEntry = m_AppletResourceEntries[i];

        if (!entry.flags.Test<AppletResourceFlag::IsAvailable>())
        {
            extraEntry = MouseAppletResourceEntry();
        }
        else
        {
            NN_SDK_ASSERT_NOT_NULL(entry.address);

            if (entry.aruid != extraEntry.aruid)
            {
                extraEntry = MouseAppletResourceEntry();

                extraEntry.aruid = entry.aruid;
            }

            processor(this,
                      &entry.address->mouse,
                      entry.aruid,
                      entry.flags.Test<AppletResourceFlag::EnablesInput>(),
                      extraEntry.buttonMask);
        }
    }
}

void MouseManager::DisableAutoPilot() NN_NOEXCEPT
{
    m_IsAutoPilotEnabled = false;
    m_IsAutoPilotStateUpdated = false;
    m_AutoPilotState = ::nn::hid::debug::MouseAutoPilotState();
}

namespace {

void CalculateMouseState(MouseState* pOutValue,
                         const MouseState& state,
                         bool enablesInput,
                         MouseButtonSet& buttonMask) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    *pOutValue = state;

    if (enablesInput)
    {
        buttonMask |= ~state.buttons;

        pOutValue->buttons &= buttonMask;
    }
    else
    {
        buttonMask.Reset();

        pOutValue->deltaX = 0;
        pOutValue->deltaY = 0;
        pOutValue->wheelDelta = 0;
        pOutValue->buttons.Reset();
        pOutValue->attributes.Set<MouseAttribute::Transferable>();
    }
}

} // namespace

}}} // namespace nn::hid::detail
