﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/xcd/xcd_Tera.h>
#include <nn/xcd/xcd_Device.h>
#include <nn/xcd/xcd_Irsensor.h>
#include <nn/xcd/xcd_TeraFirmware.h>
#include <nn/irsensor/irsensor_PointingProcessorTypesPrivate.h>
#include <nn/irsensor/irsensor_TeraPluginProcessorTypesPrivate.h>
#include <nn/irsensor/irsensor_IIrSensorDriver.h>
#include <nn/irsensor/irsensor_IrCameraTypes.h>
#include <nn/irsensor/irsensor_PackedTypes.h>

namespace nn { namespace hid { namespace detail {

class IrSensorXcdDriver final : public ::nn::irsensor::IIrSensorDriver
{
    NN_DISALLOW_COPY(IrSensorXcdDriver);
    NN_DISALLOW_MOVE(IrSensorXcdDriver);

private:
    struct MomentProcessorInfo final
    {
        typedef ::nn::irsensor::PackedMomentProcessorConfig IrSensorConfig;
        typedef ::nn::irsensor::MomentProcessorState IrSensorState;
        typedef ::nn::xcd::IrMomentProcessorState* XcdBuffer;
        typedef ::nn::xcd::IrMomentProcessorState XcdState;
        static const ::nn::xcd::IrProcessorType XcdType = ::nn::xcd::IrProcessorType::Moment;
        static const int XcdStateCountMax = ::nn::xcd::IrMomentProcessorStateCountMax;
    };

    struct ClusteringProcessorInfo final
    {
        typedef ::nn::irsensor::PackedClusteringProcessorConfig IrSensorConfig;
        typedef ::nn::irsensor::ClusteringProcessorState IrSensorState;
        typedef ::nn::xcd::IrClusteringProcessorState* XcdBuffer;
        typedef ::nn::xcd::IrClusteringProcessorState XcdState;
        static const ::nn::xcd::IrProcessorType XcdType = ::nn::xcd::IrProcessorType::Clustering;
        static const int XcdStateCountMax = ::nn::xcd::IrClusteringProcessorStateCountMax;
    };

    struct ImageTransferProcessorInfo final
    {
        typedef ::nn::irsensor::PackedImageTransferProcessorExConfig IrSensorConfig;
        typedef ::nn::irsensor::ImageTransferProcessorState IrSensorState;
        typedef ::nn::xcd::IrImageTransferProcessorState* XcdBuffer;
        typedef ::nn::xcd::IrImageTransferProcessorState XcdState;
        static const ::nn::xcd::IrProcessorType XcdType = ::nn::xcd::IrProcessorType::ImageTransfer;
        static const int XcdStateCountMax = 1;
    };

    struct PointingProcessorInfo final
    {
        typedef ::nn::irsensor::PackedPointingProcessorConfig IrSensorConfig;
        typedef ::nn::irsensor::PointingProcessorMarkerState IrSensorState;
        typedef ::nn::xcd::IrDpdProcessorState* XcdBuffer;
        typedef ::nn::xcd::IrDpdProcessorState XcdState;
        static const ::nn::xcd::IrProcessorType XcdType = ::nn::xcd::IrProcessorType::Dpd;
        static const int XcdStateCountMax = ::nn::xcd::IrDpdProcessorStateCountMax;
    };

    struct TeraPluginProcessorInfo final
    {
        typedef ::nn::irsensor::PackedTeraPluginProcessorConfig IrSensorConfig;
        typedef ::nn::irsensor::TeraPluginProcessorState IrSensorState;
        typedef ::nn::xcd::IrTeraPluginProcessorState* XcdBuffer;
        typedef ::nn::xcd::IrTeraPluginProcessorState XcdState;
        static const ::nn::xcd::IrProcessorType XcdType = ::nn::xcd::IrProcessorType::TeraPlugin;
        static const int XcdStateCountMax = 1;
    };

    // 内部的には、Moment モードとして動作させる
    struct IrLedProcessorInfo final
    {
        typedef ::nn::irsensor::PackedIrLedProcessorConfig IrSensorConfig;
        typedef ::nn::irsensor::MomentProcessorState IrSensorState;
        typedef ::nn::xcd::IrMomentProcessorState* XcdBuffer;
        typedef ::nn::xcd::IrMomentProcessorState XcdState;
        static const ::nn::xcd::IrProcessorType XcdType = ::nn::xcd::IrProcessorType::Moment;
        static const int XcdStateCountMax = ::nn::xcd::IrMomentProcessorStateCountMax;
    };

    union XcdState
    {
        ::nn::xcd::IrMomentProcessorState moment;
        ::nn::xcd::IrClusteringProcessorState clustering;
        ::nn::xcd::IrImageTransferProcessorState imageTransfer;
        ::nn::xcd::IrDpdProcessorState dpd[nn::xcd::IrDpdProcessorStateCountMax];
        ::nn::xcd::IrTeraPluginProcessorState teraPlugin;
    };

private:
    // 環境ノイズが小さいと判定する閾値
    static const float AmbientNoiseThresholdLow;
    // 環境ノイズが大きいと判定する閾値
    static const float AmbientNoiseThresholdHigh;
    // FunctionLevel と McuVersion の対応テーブル
    static const nn::xcd::McuVersionData RequiredVersionTableForFunctionLevel[];

private:
    ::nn::xcd::DeviceHandle m_Handle;

    ::nn::os::SystemEventType m_SamplingSuccessEvent;

    ::nn::os::SystemEventType m_CommandRequestSuccessEvent;

    ::nn::xcd::IrCommonData m_CommonDataBuffer;

    XcdState m_MainDataBuffer;

    ::nn::irsensor::IrCameraStatus m_IrCameraStatus;

    int64_t m_SamplingNumber;

    ::nn::TimeSpanType m_TimeStamp;

    int m_SamplingTrialCount;

    bool m_IsSubtractionEnabled;

    ::nn::xcd::IrMcuVersion m_RequiredMcuVersion;

    void* m_pLatestTransferMemoryAddress;

    ::nn::xcd::IrTeraPluginParameter m_PluginParam;

public:
    IrSensorXcdDriver() NN_NOEXCEPT;

    virtual ~IrSensorXcdDriver() NN_NOEXCEPT NN_OVERRIDE;

    ::nn::Result Initialize() NN_NOEXCEPT;

    void Finalize() NN_NOEXCEPT;

    virtual ::nn::Result CheckVersion(
        const ::nn::irsensor::PackedMcuVersion& requiredVersion,
        const ::nn::irsensor::PackedFunctionLevel& functionLevel
        ) NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result RequestVersion() NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result StopImageProcessor() NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result SuspendImageProcessor() NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result StopSampling() NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result RunImageProcessor(
        const ::nn::irsensor::PackedMomentProcessorConfig& config,
        const ::nn::irsensor::PackedFunctionLevel& functionLevel
        ) NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result RunImageProcessor(
        const ::nn::irsensor::PackedClusteringProcessorConfig& config,
        const ::nn::irsensor::PackedFunctionLevel& functionLevel
        ) NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result RunImageProcessor(
        const ::nn::irsensor::PackedImageTransferProcessorExConfig& config,
        const ::nn::irsensor::PackedFunctionLevel& functionLevel,
        void* buffer) NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result RunImageProcessor(
        const ::nn::irsensor::PackedPointingProcessorConfig& config,
        const ::nn::irsensor::PackedFunctionLevel& functionLevel
        ) NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result RunImageProcessor(
        const ::nn::irsensor::PackedTeraPluginProcessorConfig& config,
        const ::nn::irsensor::PackedFunctionLevel& functionLevel
        ) NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result RunImageProcessor(
        const ::nn::irsensor::PackedIrLedProcessorConfig& config,
        const ::nn::irsensor::PackedFunctionLevel& functionLevel
        ) NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result ResumeImageProcessor(
        const ::nn::irsensor::PackedMomentProcessorConfig& config,
        const ::nn::irsensor::PackedFunctionLevel& functionLevel
        ) NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result ResumeImageProcessor(
        const ::nn::irsensor::PackedClusteringProcessorConfig& config,
        const ::nn::irsensor::PackedFunctionLevel& functionLevel
        ) NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result ResumeImageProcessor(
        const ::nn::irsensor::PackedImageTransferProcessorExConfig& config,
        const ::nn::irsensor::PackedFunctionLevel& functionLevel,
        void* buffer) NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result ResumeImageProcessor(
        const ::nn::irsensor::PackedPointingProcessorConfig& config,
        const ::nn::irsensor::PackedFunctionLevel& functionLevel
        ) NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result ResumeImageProcessor(
        const ::nn::irsensor::PackedTeraPluginProcessorConfig& config,
        const ::nn::irsensor::PackedFunctionLevel& functionLevel
        ) NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result ResumeImageProcessor(
        const ::nn::irsensor::PackedIrLedProcessorConfig& config,
        const ::nn::irsensor::PackedFunctionLevel& functionLevel
        ) NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result GetImageProcessorStates(
        ::nn::irsensor::MomentProcessorState* pOutStates,
        int* pOutCount, int countMax) NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result GetImageProcessorStates(
        ::nn::irsensor::ClusteringProcessorState* pOutStates,
        int* pOutCount, int countMax) NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result GetImageProcessorState(
        ::nn::irsensor::ImageTransferProcessorState* pOutState,
        void* pOutImage) NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result GetImageProcessorStates(
        ::nn::irsensor::PointingProcessorMarkerState* pOutStates,
        int* pOutCount, int countMax) NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result GetImageProcessorStates(
        ::nn::irsensor::TeraPluginProcessorState* pOutStates,
        int* pOutCount, int countMax) NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result SetIrCameraStatus(
        ::nn::irsensor::IrCameraStatus status) NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result SetXcdDeviceHandle(
        ::nn::xcd::DeviceHandle handle) NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result SetLatestTransferMemoryAddress(
        void* pLatestTransferMemoryAddress) NN_NOEXCEPT NN_OVERRIDE;

    void Activate(::nn::xcd::DeviceHandle handle) NN_NOEXCEPT;

    void ConnectUnsupportedDevice() NN_NOEXCEPT;

    void Deactivate() NN_NOEXCEPT;

private:
    ::nn::Result LockMcu() NN_NOEXCEPT;

    ::nn::Result UnlockMcu() NN_NOEXCEPT;

    ::nn::Result SetMcuState(::nn::xcd::McuState state) NN_NOEXCEPT;

    ::nn::Result SetMode(::nn::xcd::IrProcessorType mode, int modeOffset, ::nn::xcd::IrCommandType type,
        const ::nn::irsensor::PackedFunctionLevel& functionLevel) NN_NOEXCEPT;

    ::nn::Result WriteRegisters(
        const ::nn::xcd::IrWriteRegisterBlock* blocks,
        int count) NN_NOEXCEPT;

    ::nn::Result WriteRegistersEx(
        const ::nn::xcd::IrWriteRegisterBlock* blocks,
        int count) NN_NOEXCEPT;

    ::nn::Result SetConfig(
        const ::nn::irsensor::PackedMomentProcessorConfig& config,
        const ::nn::xcd::IrCommandType& type
        ) NN_NOEXCEPT;

    ::nn::Result SetConfig(
        const ::nn::irsensor::PackedClusteringProcessorConfig& config,
        const ::nn::xcd::IrCommandType& type
        ) NN_NOEXCEPT;

    ::nn::Result SetConfig(
        const ::nn::irsensor::PackedImageTransferProcessorExConfig& config,
        const ::nn::xcd::IrCommandType& type
        ) NN_NOEXCEPT;

    ::nn::Result SetConfig(
        const ::nn::irsensor::PackedPointingProcessorConfig& config,
        const ::nn::xcd::IrCommandType& type
        ) NN_NOEXCEPT;

    ::nn::Result SetConfig(
        const ::nn::irsensor::PackedTeraPluginProcessorConfig& config,
        const ::nn::xcd::IrCommandType& type
        ) NN_NOEXCEPT;

    ::nn::Result SetConfig(
        const ::nn::irsensor::PackedIrLedProcessorConfig& config,
        const ::nn::xcd::IrCommandType& type
        ) NN_NOEXCEPT;

    ::nn::Result GetMcuVersion(
        ::nn::xcd::McuVersionData* pOutVersion
        ) NN_NOEXCEPT;

    ::nn::Result CheckVersion(
        ::nn::xcd::IrProcessorType mode,
        const ::nn::irsensor::PackedFunctionLevel& functionLevel
        ) NN_NOEXCEPT;

    ::nn::Result CheckMcuValidity(
        ) NN_NOEXCEPT;

    ::nn::Result CheckBatteryStatus(
        ) NN_NOEXCEPT;

    ::nn::Result StartSampling(
        ::nn::xcd::IrProcessorType mode,
        void* pLatestTranfserMemoryAddress,
        ::nn::xcd::DeviceHandle handle) NN_NOEXCEPT;

    template <typename Info>
    ::nn::Result RunImageProcessor(
        const typename Info::IrSensorConfig& config,
        const ::nn::irsensor::PackedFunctionLevel& functionLevel,
        void* buffer,
        int mode) NN_NOEXCEPT;

    template <typename Info>
    ::nn::Result ResumeImageProcessor(
        const typename Info::IrSensorConfig& config,
        const ::nn::irsensor::PackedFunctionLevel& functionLevel,
        void* buffer,
        int mode) NN_NOEXCEPT;

    template <typename Info>
    ::nn::Result GetImageProcessorStates(
        typename Info::IrSensorState* pOutStates,
        int* pOutCount, int countMax) NN_NOEXCEPT;

    void ConvertImageProcessorState(
        ::nn::irsensor::MomentProcessorState* pOutState,
        const ::nn::xcd::IrCommonData& commonData,
        const ::nn::xcd::IrMomentProcessorState& state) NN_NOEXCEPT;

    void ConvertImageProcessorState(
        ::nn::irsensor::ClusteringProcessorState* pOutState,
        const ::nn::xcd::IrCommonData& commonData,
        const ::nn::xcd::IrClusteringProcessorState& state) NN_NOEXCEPT;

    void ConvertImageProcessorState(
        ::nn::irsensor::ImageTransferProcessorState* pOutState,
        const ::nn::xcd::IrCommonData& commonData,
        const ::nn::xcd::IrImageTransferProcessorState& state) NN_NOEXCEPT;

    void ConvertImageProcessorState(
        ::nn::irsensor::PointingProcessorMarkerState* pOutState,
        const ::nn::xcd::IrCommonData& commonData,
        const ::nn::xcd::IrDpdProcessorState& state) NN_NOEXCEPT;

    void ConvertImageProcessorState(
        ::nn::irsensor::TeraPluginProcessorState* pOutState,
        const ::nn::xcd::IrCommonData& commonData,
        const ::nn::xcd::IrTeraPluginProcessorState& state) NN_NOEXCEPT;

    ::nn::Result GetXcdImageProcessorStates(
        ::nn::irsensor::MomentProcessorState* pOutStates,
        int* pOutCount,
        int countMax,
        ::nn::xcd::DeviceHandle handle) NN_NOEXCEPT;

    ::nn::Result GetXcdImageProcessorStates(
        ::nn::irsensor::ClusteringProcessorState* pOutStates,
        int* pOutCount,
        int countMax,
        ::nn::xcd::DeviceHandle handle) NN_NOEXCEPT;

    ::nn::Result GetXcdImageProcessorState(
        ::nn::irsensor::ImageTransferProcessorState* pOutState,
        void* pOutImage,
        ::nn::xcd::DeviceHandle handle) NN_NOEXCEPT;

    ::nn::Result GetXcdImageProcessorStates(
        ::nn::irsensor::PointingProcessorMarkerState* pOutStates,
        int* pOutCount,
        int countMax,
        ::nn::xcd::DeviceHandle handle) NN_NOEXCEPT;

    ::nn::Result GetXcdImageProcessorStates(
        ::nn::irsensor::TeraPluginProcessorState* pOutStates,
        int* pOutCount,
        int countMax,
        ::nn::xcd::DeviceHandle handle) NN_NOEXCEPT;

    ::nn::irsensor::IrCameraAmbientNoiseLevel ComputeNoiseLevel(
        const ::nn::xcd::IrCommonData& commonData) NN_NOEXCEPT;

    ::nn::irsensor::Rect CalculateImageTransferWoi(
        const ::nn::irsensor::PackedImageTransferProcessorExConfig& config) NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
