﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/hid/hid_IHidDebugServer.sfdl.h>
#include <nn/hid/hid_IHidSystemServer.sfdl.h>
#include <nn/hid/debug/hid_InputDetection.h>
#include <nn/hid/system/hid_InputDetection.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_NativeHandle.h>

#include "hid_HidDebugServer.h"
#include "hid_HidSystemServer.h"
#include "hid_InputDetectorId.h"
#include "hid_InputDetectorImpl.h"
#include "hid_InputDetectorLifo.h"
#include "hid_SharedMemoryAccessor.h"
#include "hid_SharedMemoryFormat.h"
#include "hid_StaticObject.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< InputDetector のセッションを扱うためのクラスです。
class InputDetectorSharedMemoryAccessor final : public SharedMemoryAccessor
{
private:
    //!< InputDetector の LIFO
    ::std::atomic<InputDetectorLifo*> m_pLifo[::nn::hid::system::InputSourceIdCountMax];

public:
    explicit InputDetectorSharedMemoryAccessor() NN_NOEXCEPT
    {
        this->SetResultActivationUpperLimitOver(
            ::nn::hid::system::ResultInputDetectorActivationUpperLimitOver());
        this->SetResultDeactivationLowerLimitOver(
            ::nn::hid::system::ResultInputDetectorDeactivationLowerLimitOver());
        for (auto& pLifo : m_pLifo)
        {
            pLifo = nullptr;
        }
    }

    virtual ~InputDetectorSharedMemoryAccessor() NN_NOEXCEPT NN_OVERRIDE { /* 何もしない */ }

    //!< InputDetector の LIFO を返します。
    InputDetectorLifo* GetLifo(int index) const NN_NOEXCEPT
    {
        return m_pLifo[index];
    }

protected:
    virtual ::nn::Result Attach(SharedMemoryFormat* ptr) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(ptr);
        ::nn::sf::SharedPointer<IHidSystemServer> pHidSystemServer;
        NN_RESULT_DO(CreateHidSystemServerProxy(&pHidSystemServer));
        NN_RESULT_DO(pHidSystemServer->ActivateInputDetector(
            ::nn::applet::AppletResourceUserId::GetInvalidId()));
        for (int i = 0; i < ::nn::hid::system::InputSourceIdCountMax; i++)
        {
            m_pLifo[i] = &::nn::util::Get(ptr->inputDetector.entries[i].lifo);
        }
        NN_RESULT_SUCCESS;
    }

    virtual ::nn::Result Detach() NN_NOEXCEPT NN_OVERRIDE
    {
        ::nn::sf::SharedPointer<IHidDebugServer> pHidDebugServer;
        NN_RESULT_DO(CreateHidDebugServerProxy(&pHidDebugServer));
        NN_RESULT_DO(pHidDebugServer->DeactivateInputDetector());
        for (auto& pLifo : m_pLifo)
        {
            pLifo = nullptr;
        }
        NN_RESULT_SUCCESS;
    }
};

//!< InputDetector の共有メモリアクセッサを返します。
InputDetectorSharedMemoryAccessor& GetInputDetectorSharedMemoryAccessor() NN_NOEXCEPT
{
    return StaticObject<InputDetectorSharedMemoryAccessor>::Get();
}

} // namespace

::nn::Result InitializeInputDetector() NN_NOEXCEPT
{
    InputDetectorSharedMemoryAccessor& accessor = GetInputDetectorSharedMemoryAccessor();
    NN_RESULT_DO(accessor.SetAppletResourceUserId(
        ::nn::applet::AppletResourceUserId::GetInvalidId()));
    NN_RESULT_DO(accessor.Activate());
    NN_RESULT_SUCCESS;
}

::nn::Result FinalizeInputDetector() NN_NOEXCEPT
{
    NN_RESULT_DO(GetInputDetectorSharedMemoryAccessor().Deactivate());
    NN_RESULT_SUCCESS;
}

::nn::Result GetInputSourceState(::nn::hid::system::InputSourceState* pOutValue,
                                 const ::nn::hid::system::InputSourceIdSet& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_GREATER_EQUAL(id.CountPopulation(), 1);
    NN_RESULT_THROW_UNLESS(id.CountPopulation() >= 1,
                           ::nn::hid::system::ResultInputDetectorInvalidIdCount());
    NN_RESULT_DO(VerifyValidInputSourceIdSet(id));

    InputDetectorSharedMemoryAccessor& accessor = GetInputDetectorSharedMemoryAccessor();

    // 指定された InputSourceId の中で最大のタイムスタンプを返す
    InputDetectorState maxTimeStamp = {};
    for (int i = 0; i < ::nn::hid::system::InputSourceIdCountMax; i++)
    {
        if (id.Test(i))
        {
            auto pLifo = accessor.GetLifo(i);
            NN_RESULT_THROW_UNLESS(pLifo != nullptr,
                                   ::nn::hid::system::ResultInputDetectorNotInitialized());
            InputDetectorState tmpState = {};
            const int gotCount = pLifo->Read(&tmpState, 1);
            NN_UNUSED(gotCount);
            NN_SDK_REQUIRES_EQUAL(gotCount, 1);
            if (tmpState.inputSourceState.timestamp > maxTimeStamp.inputSourceState.timestamp)
            {
                maxTimeStamp = tmpState;
            }
        }
    }
    *pOutValue = maxTimeStamp.inputSourceState;
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_RESULT_SUCCESS;
}

::nn::Result NotifyInputDetector(const ::nn::hid::system::InputSourceIdSet& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(id.CountPopulation(), 1);
    NN_RESULT_THROW_UNLESS(id.CountPopulation() >= 1,
                           ::nn::hid::system::ResultInputDetectorInvalidIdCount());
    NN_RESULT_DO(VerifyValidInputSourceIdSet(id));

    ::nn::sf::SharedPointer<IHidSystemServer> pHidSystemServer;
    NN_RESULT_DO(CreateHidSystemServerProxy(&pHidSystemServer));
    NN_RESULT_DO(pHidSystemServer->NotifyInputDetector(id));
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
