﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_IHidDebugServer.sfdl.h>
#include <nn/hid/hid_ServiceName.h>
#include <nn/lmem/lmem_Common.h>
#include <nn/os/os_Mutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_HipcClientProxyByName.h>
#include <nn/sf/sf_ISharedObject.h>

#include "hid_HidDebugServer.h"
#include "hid_LockableMutexType.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< IPC 用ヒープメモリを扱うクラスです。
typedef ::nn::sf::ExpHeapStaticAllocator<16 * 1024, HidDebugServer> Allocator;

}

::nn::Result CreateHidDebugServerProxy(
    ::nn::sf::SharedPointer<IHidDebugServer>* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    NN_FUNCTION_LOCAL_STATIC(
        ::nn::sf::SharedPointer<IHidDebugServer>, s_SharedPointer);

    static LockableMutexType s_Mutex = { NN_OS_MUTEX_INITIALIZER(false) };

    NN_FUNCTION_LOCAL_STATIC(
        ::std::atomic<bool>, s_IsAllocatorInitialized, (false));

    if (!s_IsAllocatorInitialized)
    {
        ::std::lock_guard<decltype(s_Mutex)> locker(s_Mutex);

        if (!s_IsAllocatorInitialized)
        {
            Allocator::Initialize(::nn::lmem::CreationOption_NoOption);

            s_IsAllocatorInitialized = true;
        }
    }

    NN_FUNCTION_LOCAL_STATIC(
        ::std::atomic<bool>, s_IsSharedPointerInitialized, (false));

    if (!s_IsSharedPointerInitialized)
    {
        ::std::lock_guard<decltype(s_Mutex)> locker(s_Mutex);

        if (!s_IsSharedPointerInitialized)
        {
            NN_RESULT_DO((::nn::sf::CreateHipcProxyByName<
                IHidDebugServer, Allocator::Policy>)(&s_SharedPointer,
                                                     HidDebugServiceName));

            s_IsSharedPointerInitialized = true;
        }
    }

    *pOutValue = s_SharedPointer;

    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
