﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_IHidDebugServer.sfdl.h>
#include <nn/hid/hid_NpadJoyCommon.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/system/hid_Result.system.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/os/os_MemoryHeap.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_NativeHandle.h>

#include "hid_HidDebugServer.h"
#include "hid_FirmwareUpdateDebugImpl.h"

namespace nn { namespace hid { namespace debug { namespace detail {

::nn::Result InitializeFirmwareUpdate() NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidDebugServer>();
    ::nn::hid::detail::CreateHidDebugServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->ActivateFirmwareUpdate());
    NN_RESULT_DO(pProxy->DiscardFirmwareInfoCacheForRevert());
    NN_RESULT_SUCCESS;
}

::nn::Result StartFirmwareUpdate(system::UniquePadId id) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidDebugServer>();
    ::nn::hid::detail::CreateHidDebugServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->StartFirmwareUpdate(id));
    NN_RESULT_SUCCESS;
}

::nn::Result StartFirmwareUpdateForRevert(system::UniquePadId id) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidDebugServer>();
    ::nn::hid::detail::CreateHidDebugServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->StartFirmwareUpdateForRevert(id));
    NN_RESULT_SUCCESS;
}

::nn::Result GetFirmwareVersion(debug::FirmwareVersion* pOutValue,
                                NpadIdType id,
                                system::NpadDeviceTypeSet deviceType) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidDebugServer>();
    ::nn::hid::detail::CreateHidDebugServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->GetFirmwareVersion(pOutValue, id, deviceType));
    NN_RESULT_SUCCESS;
}

::nn::Result GetAvailableFirmwareVersionForRevert(system::FirmwareVersion* pOutValue,
                                                  system::UniquePadId id) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidDebugServer>();
    ::nn::hid::detail::CreateHidDebugServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->GetAvailableFirmwareVersionForRevert(pOutValue, id));
    NN_RESULT_SUCCESS;
}

::nn::Result GetDestinationFirmwareVersion(debug::FirmwareVersion* pOutValue,
                                           NpadIdType id,
                                           system::NpadDeviceTypeSet deviceType) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidDebugServer>();
    ::nn::hid::detail::CreateHidDebugServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->GetDestinationFirmwareVersion(pOutValue, id, deviceType));
    NN_RESULT_SUCCESS;
}

::nn::Result GetFirmwareUpdateStage(debug::FirmwareUpdateStage* pOutStage, uint8_t* pOutProgress) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidDebugServer>();
    ::nn::hid::detail::CreateHidDebugServerProxy(&pProxy);

    auto outProgress = int64_t();
    auto outStage = int64_t();

    NN_RESULT_DO(pProxy->GetFirmwareUpdateStage(::nn::sf::Out<int64_t>(&outStage),
                                                ::nn::sf::Out<int64_t>(&outProgress)));
    *pOutProgress = static_cast<uint8_t>(outProgress);

    switch (outStage)
    {
    case debug::FirmwareUpdateStage_Preparing:
        *pOutStage = debug::FirmwareUpdateStage_Preparing;
        break;
    case debug::FirmwareUpdateStage_BluetoothDownload:
        *pOutStage = debug::FirmwareUpdateStage_BluetoothDownload;
        break;
    case debug::FirmwareUpdateStage_BluetoothVerify:
        *pOutStage = debug::FirmwareUpdateStage_BluetoothVerify;
        break;
    case debug::FirmwareUpdateStage_BluetoothCommit:
        *pOutStage = debug::FirmwareUpdateStage_BluetoothCommit;
        break;
    case debug::FirmwareUpdateStage_Reboot1:
        *pOutStage = debug::FirmwareUpdateStage_Reboot1;
        break;
    case debug::FirmwareUpdateStage_McuVersionCheck:
        *pOutStage = debug::FirmwareUpdateStage_McuVersionCheck;
        break;
    case debug::FirmwareUpdateStage_McuBoot:
        *pOutStage = debug::FirmwareUpdateStage_McuBoot;
        break;
    case debug::FirmwareUpdateStage_McuErasing:
        *pOutStage = debug::FirmwareUpdateStage_McuErasing;
        break;
    case debug::FirmwareUpdateStage_McuWriting:
        *pOutStage = debug::FirmwareUpdateStage_McuWriting;
        break;
    case debug::FirmwareUpdateStage_McuReboot:
        *pOutStage = debug::FirmwareUpdateStage_McuReboot;
        break;
    case debug::FirmwareUpdateStage_McuVerify:
        *pOutStage = debug::FirmwareUpdateStage_McuVerify;
        break;
    case debug::FirmwareUpdateStage_Reboot2:
        *pOutStage = debug::FirmwareUpdateStage_Reboot2;
        break;
    case debug::FirmwareUpdateStage_Completed:
        *pOutStage = debug::FirmwareUpdateStage_Completed;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result IsFirmwareUpdatingDevice(bool* pOutIsUpdating,
                                      system::UniquePadId id) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidDebugServer>();
    ::nn::hid::detail::CreateHidDebugServerProxy(&pProxy);

    auto outIsUpdating = bool();
    NN_RESULT_DO(pProxy->IsFirmwareUpdatingDevice(&outIsUpdating, id));
    *pOutIsUpdating = outIsUpdating;

    NN_RESULT_SUCCESS;
}

::nn::Result StartFirmwareUpdateIndividual(
    system::FirmwareUpdateDeviceHandle* pOutDeviceHandle,
    system::UniquePadId id,
    FirmwareUpdateTargetChip target,
    char* pImageData,
    size_t imageSize) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidDebugServer>();
    ::nn::hid::detail::CreateHidDebugServerProxy(&pProxy);

    // TransferMemory の生成
    nn::os::TransferMemoryType imageMemory;
    NN_RESULT_DO(
        nn::os::CreateTransferMemory(
            &imageMemory,
            pImageData,
            nn::util::align_up(imageSize, nn::os::MemoryPageSize),
            nn::os::MemoryPermission_ReadOnly
        )
    );

    // ハンドルの管理をプロセス側に任せる
    auto imageHandle = nn::os::DetachTransferMemory(&imageMemory);
    nn::os::DestroyTransferMemory(&imageMemory);

    system::FirmwareUpdateDeviceHandle deviceHandle;
    NN_RESULT_DO(
        pProxy->StartFirmwareUpdateIndividual(
            &deviceHandle,
            id,
            static_cast<int64_t>(target),
            nn::sf::NativeHandle(imageHandle, true),
            static_cast<uint64_t>(imageSize)
        )
    );
    *pOutDeviceHandle = deviceHandle;

    NN_RESULT_SUCCESS;
}

}}}} // namespace nn::hid::debug::detail
