﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstring>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/xcd/xcd.h>
#include <nn/xcd/xcd_AttachmentDevice.h>

#include "hid_ExternalBusXcdDriver.h"

namespace
{
inline nn::Result ConvertXcdResultToHidResult(nn::Result xcdResult)
{
    if (xcdResult.IsFailure())
    {
        if (nn::xcd::ResultAttachmentDeviceNotAttached::Includes(xcdResult))
        {
            return nn::hid::ResultExternalBusDeviceNotAttached();
        }
        else if (nn::xcd::ResultAttachmentDeviceAttachmentReadyTimeout::Includes(xcdResult))
        {
            return nn::hid::ResultExternalBusDeviceReadyTimeout();
        }
        else if (nn::xcd::ResultAttachmentDeviceInquiryTimeout::Includes(xcdResult))
        {
            return nn::hid::ResultExternalBusDeviceInquiryTimeout();
        }
        else if (nn::xcd::ResultAttachmentDeviceNotReady::Includes(xcdResult))
        {
            return nn::hid::ResultExternalBusDeviceNotEnabled();
        }
        else if (nn::xcd::ResultNotConnected::Includes(xcdResult))
        {
            return nn::hid::ResultExternalBusDeviceInvalidHandle();
        }
        else if (nn::xcd::ResultAttachmentDeviceTimeout::Includes(xcdResult))
        {
            return nn::hid::ResultExternalBusDeviceTimeout();
        }
        else if (nn::xcd::ResultMcuBusy::Includes(xcdResult) ||
                 nn::xcd::ResultInvalidMcuState::Includes(xcdResult))
        {
            // バージョンチェックが動作中の場合やステート遷移要求が出ている場合は DeviceReadyTimeout として返す
            return nn::hid::ResultExternalBusDeviceReadyTimeout();
        }
        else
        {
            return xcdResult;
        }
    }
    NN_RESULT_SUCCESS;
}
}

namespace nn { namespace hid { namespace detail {

nn::Result ExternalBusXcdDriver::Initialize() NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

void ExternalBusXcdDriver::Finalize() NN_NOEXCEPT
{
    return;
}

nn::Result ExternalBusXcdDriver::SetXcdDeviceHandle(::nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    m_Handle = handle;
    NN_RESULT_SUCCESS;
}

bool ExternalBusXcdDriver::IsExternalConnecterDeviceConnected() NN_NOEXCEPT
{
    return nn::xcd::IsAttachmentDeviceAttached(m_Handle);
}

nn::Result ExternalBusXcdDriver::EnableExternalConnecterDevice(bool isEnabled, uint64_t version) NN_NOEXCEPT
{
    nn::xcd::AttachmentFunctionRequiredVersion requiredVersion;
    requiredVersion.btVersion.major = static_cast<uint8_t>(version >> 40);
    requiredVersion.btVersion.minor = static_cast<uint8_t>(version >> 32);
    requiredVersion.mcuVersion.major = static_cast<uint16_t>(version >> 16);
    requiredVersion.mcuVersion.minor = static_cast<uint16_t>(version);

    auto result = nn::xcd::AttachmentEnable(isEnabled, requiredVersion, m_Handle);
    return ConvertXcdResultToHidResult(result);

}

nn::Result ExternalBusXcdDriver::GetExternalConnecterDeviceInfo(ExternalBusDeviceInfo* pOutInfo) NN_NOEXCEPT
{
    nn::xcd::ExtDevInfo info;
    auto result = nn::xcd::GetExtDevInfo(&info, m_Handle);
    pOutInfo->id = info.extDevType;

    return ConvertXcdResultToHidResult(result);
}

nn::Result ExternalBusXcdDriver::SendCommandAsyncToExternalBusDevice(const uint8_t* pInCommand, size_t inCommandSize) NN_NOEXCEPT
{
    auto result = nn::xcd::SendCommandToAttachmentDevice(pInCommand, inCommandSize, m_Handle);
    return ConvertXcdResultToHidResult(result);
}

nn::Result ExternalBusXcdDriver::GetSendCommandAsynceResultFromExternalBusDevice(size_t* pOutSize, uint8_t* pOutCommand, size_t outCommandSize) NN_NOEXCEPT
{
    auto result = nn::xcd::GetReceivedCommandFromAttachmentDevice(pOutSize, pOutCommand, outCommandSize, m_Handle);
    return ConvertXcdResultToHidResult(result);
}

nn::Result ExternalBusXcdDriver::SetEventForExternalBusDeviceSendCommandAsycResult(nn::os::SystemEventType* pReceiveEventHandle) NN_NOEXCEPT
{
    auto result = nn::xcd::SetAttachmentEvent(pReceiveEventHandle, m_Handle);
    return ConvertXcdResultToHidResult(result);
}

nn::Result ExternalBusXcdDriver::EnablePollingReceiveModeForAttachmentDevice(const uint8_t* pInCommand, size_t inCommandSize, ExternalBusJoyPollingMode mode) NN_NOEXCEPT
{
    auto result = nn::xcd::EnablePollingReceiveModeForAttachmentDevice(pInCommand, inCommandSize, m_Handle, static_cast<nn::xcd::AttachmentPollingMode>(mode));
    return ConvertXcdResultToHidResult(result);
}

nn::Result ExternalBusXcdDriver::DisablePollingReceiveModeForAttachmentDevice() NN_NOEXCEPT
{
    auto result = nn::xcd::DisablePollingReceiveModeForAttachmentDevice(m_Handle);
    return ConvertXcdResultToHidResult(result);
}

nn::Result ExternalBusXcdDriver::GetPollingDataForAttachmentDevice(size_t* pOutSize, uint8_t* pOutCommand, size_t outCommandSize) NN_NOEXCEPT
{
    auto result = nn::xcd::GetPollingDataForAttachmentDevice(pOutSize, pOutCommand, outCommandSize, m_Handle);
    return ConvertXcdResultToHidResult(result);
}

}}} // namespace nn::hid::detail
