﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_AnalogStickState.h>
#include <nn/hid/hid_ControlPadTypes.h>
#include <nn/hid/hid_Xpad.h>
#include <nn/hid/detail/hid_AbstractedPadTypes.h>
#include <nn/util/util_BitFlagSet.h>
#include <nn/xcd/xcd_Input.h>

//!< XInput のアナログスティックのカウント理論値
const int XInputAnalogStickTheoreticalCount = 0x7fff;

//!< XInput の左アナログスティックのデッドゾーン
const int XInputAnalogStickLDeadZoneCount = 0x1EA9;

//!< XInput の右アナログスティックのデッドゾーン
const int XInputAnalogStickRDeadZoneCount = 0x21f1;

//!< XInput のアナログスティックの最大値
const int XInputAnalogStickMaxCount = 0x7fff;

//!< アナログスティックの入力変化閾値
const int32_t AnalogStickThreshold = 1;

namespace nn { namespace hid { namespace detail {

//!< アナログスティックの十字キーエミュレーション入力のデッドゾーン半径の 2乗(最大値の 1/2 を指定)
const int32_t CrossStickEmulationDeadZoneRadiusSquaredCount =
    static_cast<int32_t>(::nn::hid::AnalogStickMax * ::nn::hid::AnalogStickMax * 0.25f);


//!< 平方根を返します。
inline int64_t GetSquareRoot(int64_t x) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(0 <= x);

    if (x == 0)
    {
        return 0;
    }

    int64_t tmp = 1;
    int64_t root = x;

    while (tmp < root)
    {
        tmp <<= 1;
        root >>= 1;
    }

    do
    {
        root = tmp;
        tmp = (x / tmp + tmp) >> 1;
    } while (tmp < root);

    return root;
}

//!< 内積計算結果を返します。
inline float Dot(const ::nn::util::Float2& vec1,
          const ::nn::util::Float2& vec2)
{
    return (vec1.x * vec2.x + vec1.y * vec2.y);
}

//!< 十字キーを含むデジタルボタンの入力状態をマージします。
template<int N, typename Tag>
inline ::nn::util::BitFlagSet<N, Tag> MergeButtonsIncludingDpad(
    const ::nn::util::BitFlagSet<N, Tag>& lhs,
    const ::nn::util::BitFlagSet<N, Tag>& rhs) NN_NOEXCEPT
{
    ::nn::util::BitFlagSet<N, Tag> value = rhs;

    // 十字キーについては第 1 引数の入力を優先

    if (lhs.Test(Tag::Up::Index) || lhs.Test(Tag::Down::Index))
    {
        value.Set(Tag::Up::Index, false);
        value.Set(Tag::Down::Index, false);
    }

    if (lhs.Test(Tag::Left::Index) || lhs.Test(Tag::Right::Index))
    {
        value.Set(Tag::Left::Index, false);
        value.Set(Tag::Right::Index, false);
    }

    value |= lhs;

    return value;
}

//!< 抽象化されたコントローラーのデジタルボタンの入力状態をマージします。
inline AbstractedPadButtonSet MergeNpadButtons(const AbstractedPadButtonSet& lhs,
                                           const AbstractedPadButtonSet& rhs
                                           ) NN_NOEXCEPT
{
    return MergeButtonsIncludingDpad(lhs, rhs);
}

//!< 十字キーを含むデジタルボタンの入力状態に禁則処理を適応します。
template<int N, typename Tag>
inline ::nn::util::BitFlagSet<N, Tag> RestrictButtonsIncludingDpad(
    const ::nn::util::BitFlagSet<N, Tag>& buttons,
    ControlPadRestrictionType mode,
    bool enabled) NN_NOEXCEPT
{
    if (enabled == false)
    {
        return buttons;
    }

    ::nn::util::BitFlagSet<N, Tag> value = buttons;

    if (mode == ControlPadRestrictionType_KeepUpAndLeft)
    {
        // 十字キー上を優先
        if (value.Test(Tag::Up::Index))
        {
            value.Set(Tag::Down::Index, false);
        }

        // 十字キー左を優先
        if (value.Test(Tag::Left::Index))
        {
            value.Set(Tag::Right::Index, false);
        }
    }

    return value;
}

//!< BasicXpad のデジタルボタンの押下状態を抽象化 Pad のものに変換します。(自動操作用)
inline AbstractedPadButtonSet ConvertBasicXpadButtonSetToAbstracted(const BasicXpadButtonSet& xpadButton
                                         ) NN_NOEXCEPT
{
    AbstractedPadButtonSet buttons;
    buttons.Reset();
    buttons.Set<AbstractedPadButton::A>(xpadButton.Test<BasicXpadButton::A>());
    buttons.Set<AbstractedPadButton::B>(xpadButton.Test<BasicXpadButton::B>());
    buttons.Set<AbstractedPadButton::X>(xpadButton.Test<BasicXpadButton::X>());
    buttons.Set<AbstractedPadButton::Y>(xpadButton.Test<BasicXpadButton::Y>());
    buttons.Set<AbstractedPadButton::L>(xpadButton.Test<BasicXpadButton::L>());
    buttons.Set<AbstractedPadButton::R>(xpadButton.Test<BasicXpadButton::R>());
    buttons.Set<AbstractedPadButton::ZL>(xpadButton.Test<BasicXpadButton::ZL>());
    buttons.Set<AbstractedPadButton::ZR>(xpadButton.Test<BasicXpadButton::ZR>());
    buttons.Set<AbstractedPadButton::Start>(xpadButton.Test<BasicXpadButton::Start>());
    buttons.Set<AbstractedPadButton::Select>(xpadButton.Test<BasicXpadButton::Select>());
    buttons.Set<AbstractedPadButton::Left>(xpadButton.Test<BasicXpadButton::Left>());
    buttons.Set<AbstractedPadButton::Up>(xpadButton.Test<BasicXpadButton::Up>());
    buttons.Set<AbstractedPadButton::Right>(xpadButton.Test<BasicXpadButton::Right>());
    buttons.Set<AbstractedPadButton::Down>(xpadButton.Test<BasicXpadButton::Down>());
    buttons.Set<AbstractedPadButton::StickL>(xpadButton.Test<BasicXpadButton::StickL>());
    buttons.Set<AbstractedPadButton::StickR>(xpadButton.Test<BasicXpadButton::StickR>());

    return buttons;
}

//!< アナログスティックの入力状態をマージします。
inline AnalogStickState MergeAnalogStick(const AnalogStickState& lhs,
                                         const AnalogStickState& rhs
                                         ) NN_NOEXCEPT
{
    // 第 1 引数が無入力だった場合のみ第 2 引数を採用
    return (lhs.x != 0 || lhs.y != 0) ? lhs : rhs;
}

//!< アナログスティックの座標値を変域内に収めます。
inline AnalogStickState LimitAnalogStick(const AnalogStickState& state,
                                         const int& minCount,
                                         const int& maxCount,
                                         const int& theoricalCount
                                         ) NN_NOEXCEPT
{
    // 座標の極性を記録
    const bool isXPositive = state.x >= 0;
    const bool isYPositive = state.y >= 0;

    AnalogStickState value;

    // 座標値を正の値に変換
    value.x = isXPositive ? state.x : -state.x;
    value.y = isYPositive ? state.y : -state.y;

    // アナログスティックのデッドゾーンを計算
    const int deadZone = minCount * AnalogStickMax / theoricalCount;

    // 下限側を処理
    value.x = (value.x <= deadZone) ? 0 : value.x - deadZone;
    value.y = (value.y <= deadZone) ? 0 : value.y - deadZone;

    // 座標がデッドゾーン内だった場合は処理を完了
    if (value.x == 0 && value.y == 0)
    {
        return value;
    }

    // 精度保証用のスケール値
    const int scale = 7;

    // アナログスティックの半径を計算
    const auto r = static_cast<int64_t>(
        (maxCount - minCount) * AnalogStickMax / theoricalCount) << scale;

    // アナログスティックの半径の 2 乗
    const int64_t r2 = r * r;

    // 原点からの距離の 2 乗
    const auto d2 = static_cast<int64_t>(value.x * value.x +
                                         value.y * value.y) << (scale * 2);

    if (r2 < d2)
    {
        // 座標値が上限を超えていた場合は処理
        const int64_t d = GetSquareRoot(d2);
        value.x = static_cast<int32_t>(r * value.x / d);
        value.y = static_cast<int32_t>(r * value.y / d);
    }

    // 座標の極性を復元
    value.x = isXPositive ? value.x : -value.x;
    value.y = isYPositive ? value.y : -value.y;

    // アナログスティックの上限値
    const auto l = static_cast<int64_t>(AnalogStickMax) << scale;

    // 上下限値を復元
    value.x = static_cast<int32_t>(l * value.x / r);
    value.y = static_cast<int32_t>(l * value.y / r);

    return value;
}

//!< アナログスティックの入力状態をマージします。
inline AnalogStickState MergeAnalogSticks(const AnalogStickState& lhs,
                                          const AnalogStickState& rhs,
                                          bool usesFirstAnalogStickState
                                           ) NN_NOEXCEPT
{
    return (usesFirstAnalogStickState) ? lhs : rhs;
}

//!< アナログスティックの入力状態を回転します。
inline AnalogStickState RotateAnalogSticks(const AnalogStickState& state, bool clockwise)
{
    auto returnValue = state;

    if (clockwise == true)
    {
        returnValue.x = state.y;
        returnValue.y = -state.x;
    }
    else
    {
        returnValue.x = -state.y;
        returnValue.y = state.x;
    }

    return returnValue;
}

template<typename T>
inline void ConvertLeftJoyAccAxis(T* pState)
{
    T state = *pState;

    pState->x = state.y;
    pState->y = -state.x;
    pState->z = -state.z;
}

template<typename T>
inline void ConvertLeftJoyGyroAxis(T* pState)
{
    T state = *pState;

    pState->x = -state.y;
    pState->y = state.x;
    pState->z = state.z;
}

template<typename T>
inline void ConvertRightJoyAccAxis(T* pState)
{
    T state = *pState;

    pState->x = -state.y;
    pState->y = -state.x;
    pState->z =  state.z;
}

template<typename T>
inline void ConvertRightJoyGyroAxis(T* pState)
{
    T state = *pState;

    pState->x =  state.y;
    pState->y =  state.x;
    pState->z = -state.z;
}

template<typename T>
inline void ConvertFullKeyAccAxis(T* pState)
{
    T state = *pState;

    pState->x =  state.y;
    pState->y = -state.x;
    pState->z = -state.z;
}

template<typename T>
inline void ConvertFullKeyGyroAxis(T* pState)
{
    T state = *pState;

    pState->x = -state.y;
    pState->y =  state.x;
    pState->z =  state.z;
}

inline void ConvertLeftJoyAxis(nn::xcd::SixAxisSensorState* pState)
{
    ConvertLeftJoyAccAxis(&pState->accelerometer);
    ConvertLeftJoyGyroAxis(&pState->gyroscope);
}

inline void ConvertRightJoyAxis(nn::xcd::SixAxisSensorState* pState)
{
    ConvertRightJoyAccAxis(&pState->accelerometer);
    ConvertRightJoyGyroAxis(&pState->gyroscope);
}

inline void ConvertFullKeyAxis(nn::xcd::SixAxisSensorState* pState)
{
    ConvertFullKeyAccAxis(&pState->accelerometer);
    ConvertFullKeyGyroAxis(&pState->gyroscope);
}

//!< アナログスティックの値をクランプします
inline AnalogStickState ClampAnalogStick(const AnalogStickState& state,
                                         const int16_t& inside,
                                         const AnalogStickState& outsideMin,
                                         const AnalogStickState& outsideMax) NN_NOEXCEPT
{
    AnalogStickState value = {0,0};
    int64_t outsideX = (state.x < 0) ? outsideMin.x : outsideMax.x;
    int64_t outsideY = (state.y < 0) ? outsideMin.y : outsideMax.y;
    int64_t scaledX = state.x * outsideY;
    int64_t scaledY = state.y * outsideX;
    auto scaledOutside = static_cast<int64_t>(outsideY * outsideX);
    auto scaledInside = static_cast<int64_t>(inside * ((outsideY < outsideX) ? outsideX : outsideY));

    // クランプ領域の 2 乗
    const auto max2 = static_cast<int64_t>(scaledOutside * scaledOutside);
    const auto min2 = static_cast<int64_t>(scaledInside * scaledInside);

    // 原点からの距離の 2 乗
    const auto d2 = static_cast<int64_t>(scaledX * scaledX +
                                            scaledY * scaledY);

    if(d2 <= min2)
    {
        // 中心のクランプ
        value.x = 0;
        value.y = 0;
    }
    else if(d2 >= max2)
    {
        // 外周のクランプ
        auto d = GetSquareRoot(d2);
        value.x = static_cast<int32_t>(scaledX * AnalogStickMax / d);
        value.y = static_cast<int32_t>(scaledY * AnalogStickMax / d);
    }
    else
    {
        // 上下限値内のスケール
        auto d = GetSquareRoot(d2);
        auto actualLength = d - scaledInside;
        auto actualLengthMax = scaledOutside - scaledInside;

        value.x = static_cast<int32_t>(scaledX * actualLength * AnalogStickMax / (actualLengthMax * d));
        value.y = static_cast<int32_t>(scaledY * actualLength * AnalogStickMax / (actualLengthMax * d));
    }

    return value;
}

inline AnalogStickState ClampAnalogStickInnerCross(const AnalogStickState& state,
                                         const int16_t& inside,
                                         const AnalogStickState& outsideMin,
                                         const AnalogStickState& outsideMax) NN_NOEXCEPT
{
    AnalogStickState value = {0,0};

    // Remove vertical zone
    if (-inside < state.x && state.x < inside)
    {
        value.x = 0;
    }
    else if (0 < state.x)
    {
        value.x = state.x - inside;
    }
    else
    {
        value.x = state.x + inside;
    }

    // Remove horizontal zone
    if (-inside < state.y && state.y < inside)
    {
        value.y = 0;
    }
    else if (0 < state.y)
    {
        value.y = state.y - inside;
    }
    else
    {
        value.y = state.y + inside;
    }

    // 原点にある場合は終了
    if (value.x == 0 && value.y == 0)
    {
        return value;
    }

    int64_t outsideX = (value.x < 0) ? outsideMin.x : outsideMax.x;
    int64_t outsideY = (value.y < 0) ? outsideMin.y : outsideMax.y;
    int64_t scaledX = value.x * outsideY;
    int64_t scaledY = value.y * outsideX;
    auto scaledOutside = static_cast<int64_t>(outsideY * outsideX);

    // クランプ領域の 2 乗
    const auto max2 = static_cast<int64_t>(scaledOutside * scaledOutside);

    // 原点からの距離の 2 乗
    const auto d2 = static_cast<int64_t>(scaledX * scaledX +
                                            scaledY * scaledY);

    // 距離が 0 の場合は原点なので終了
    if (d2 == 0)
    {
        return value;
    }

    if(d2 >= max2)
    {
        // 外周のクランプ
        auto d = GetSquareRoot(d2);
        value.x = static_cast<int32_t>(scaledX * AnalogStickMax / d);
        value.y = static_cast<int32_t>(scaledY * AnalogStickMax / d);
    }
    else
    {
        // 上下限値内のスケール
        auto d = GetSquareRoot(d2);

        value.x = static_cast<int32_t>(scaledX * d * AnalogStickMax / (scaledOutside * d));
        value.y = static_cast<int32_t>(scaledY * d * AnalogStickMax / (scaledOutside * d));
    }

    return value;
}

//!< Xcd のアナログスティックを AnalogStickState のものに変換します。
inline AnalogStickState ConvertXcdAnalogStickState(const ::nn::xcd::AnalogStickState& value, const ::nn::xcd::AnalogStickValidRange range)
{
    AnalogStickState analogStick;
    AnalogStickState outsideMin = { range.circuitMin.x, range.circuitMin.y};
    AnalogStickState outsideMax = { range.circuitMax.x, range.circuitMax.y};

    analogStick.x = value.x - range.origin.x;
    analogStick.y = value.y - range.origin.y;

    return ClampAnalogStick(analogStick, range.originPlay, outsideMin, outsideMax);
}

//!< Gc コントローラーアダプターのアナログスティックを AnalogStickState のものに変換します。
inline AnalogStickState ConvertGcAnalogStickState(const AnalogStickState& value,
                                                  const AnalogStickState& center,
                                                  const int16_t& inside,
                                                  const int16_t& outside) NN_NOEXCEPT
{
    AnalogStickState analogStick;
    AnalogStickState outsideMin = { outside, outside };
    AnalogStickState outsideMax = { outside, outside };

    analogStick.x = value.x - center.x;
    analogStick.y = value.y - center.y;

    return ClampAnalogStickInnerCross(analogStick, inside, outsideMin, outsideMax);
}

//!< 十字キーエミュレーションボタンを有効にします。
inline void SetCrossStickEmulationButtons(AbstractedPadButtonSet* pOutButtons,
                                          const ::nn::hid::AnalogStickState& stick,
                                          bool isLeft)
{
    // TORIAEZU: 判定領域は[上][下][左][右]それぞれ xy_deg = 45度。
    //           余った領域は両隣の領域の論理和となります。
    //           判定領域の回転角は rot_deg = 0度。

    // [右][右上]境界ベクトル
    // ( cos((xy_deg / 2 + rot_deg) * (pi/180)), sin((xy_deg / 2 + rot_deg) * (pi/180)))
    const ::nn::util::Float2 BoundaryRightAndRightUp = {{{0.92387953251f, 0.38268343235f}}};
    // [右上][上]境界ベクトル
    // ( cos((90 - xy_deg / 2 + rot_deg) * (pi/180)), sin((90 - xy_deg / 2 + rot_deg) * (pi/180)))
    const ::nn::util::Float2 BoundaryRightUpAndUp = {{{0.38268343236f, 0.92387953251f}}};

    // クランプ済のスティック入力がデッドゾーン範囲内の場合は何もしない
    int32_t SquaredRadius = stick.x * stick.x + stick.y * stick.y;

    if(SquaredRadius < CrossStickEmulationDeadZoneRadiusSquaredCount)
    {
        return;
    }

    // [上][左上]境界ベクトル
    const ::nn::util::Float2 BoundaryUpAndLeftUp = {{{-BoundaryRightAndRightUp.y,
                                                      BoundaryRightAndRightUp.x}}};
    // [左上][左]境界ベクトル
    const ::nn::util::Float2 BoundaryLeftUpAndLeft = {{{-BoundaryRightUpAndUp.y,
                                                        BoundaryRightUpAndUp.x}}};
    // 入力ベクトル
    const ::nn::util::Float2 StickInput = {{{static_cast<float>(stick.x),
                                             static_cast<float>(stick.y)}}};

    // 境界ベクトルと入力ベクトルとの内積結果から領域判定を行います。

    // [右][右上]境界ベクトルとの内積
    if(Dot(StickInput, BoundaryRightAndRightUp) < 0.0f)
    {
        // [上][左上]境界ベクトルとの内積
        if(Dot(StickInput, BoundaryUpAndLeftUp) < 0.0f)
        {
            // [左上][左]境界ベクトルとの内積
            if(Dot(StickInput, BoundaryLeftUpAndLeft) < 0.0f)
            {
                isLeft ? pOutButtons->Set<AbstractedPadButton::StickLDown>()
                       : pOutButtons->Set<AbstractedPadButton::StickRDown>();
            }
            else
            {
                isLeft ? pOutButtons->Set<AbstractedPadButton::StickLLeft>()
                       : pOutButtons->Set<AbstractedPadButton::StickRLeft>();

                isLeft ? pOutButtons->Set<AbstractedPadButton::StickLDown>()
                       : pOutButtons->Set<AbstractedPadButton::StickRDown>();
            }
        }
        else
        {
            // [右上][上]境界ベクトルとの内積
            if (Dot(StickInput, BoundaryRightUpAndUp) < 0.0f)
            {
                isLeft ? pOutButtons->Set<AbstractedPadButton::StickLLeft>()
                       : pOutButtons->Set<AbstractedPadButton::StickRLeft>();
            }
            else
            {
                isLeft ? pOutButtons->Set<AbstractedPadButton::StickLLeft>()
                       : pOutButtons->Set<AbstractedPadButton::StickRLeft>();

                isLeft ? pOutButtons->Set<AbstractedPadButton::StickLUp>()
                       : pOutButtons->Set<AbstractedPadButton::StickRUp>();
            }
        }
    }
    else
    {
        // [上][左上]境界ベクトルとの内積
        if(Dot(StickInput, BoundaryUpAndLeftUp) < 0.0f)
        {
            // [右上][上]境界ベクトルとの内積
            if (Dot(StickInput, BoundaryRightUpAndUp) < 0.0f)
            {
                isLeft ? pOutButtons->Set<AbstractedPadButton::StickLRight>()
                       : pOutButtons->Set<AbstractedPadButton::StickRRight>();

                isLeft ? pOutButtons->Set<AbstractedPadButton::StickLDown>()
                       : pOutButtons->Set<AbstractedPadButton::StickRDown>();
            }
            else
            {
                isLeft ? pOutButtons->Set<AbstractedPadButton::StickLRight>()
                       : pOutButtons->Set<AbstractedPadButton::StickRRight>();
            }
        }
        else
        {
            // [左上][左]境界ベクトルとの内積
            if (Dot(StickInput, BoundaryLeftUpAndLeft) < 0.0f)
            {
                isLeft ? pOutButtons->Set<AbstractedPadButton::StickLRight>()
                       : pOutButtons->Set<AbstractedPadButton::StickRRight>();

                isLeft ? pOutButtons->Set<AbstractedPadButton::StickLUp>()
                       : pOutButtons->Set<AbstractedPadButton::StickRUp>();
            }
            else
            {
                isLeft ? pOutButtons->Set<AbstractedPadButton::StickLUp>()
                       : pOutButtons->Set<AbstractedPadButton::StickRUp>();
            }
        }
    }
}

//!< 十字キーエミュレーションボタンを有効にします。
inline void SetCrossStickEmulationButtonsOnAbstractedPad(AbstractedPadState* pOutState)
{
    SetCrossStickEmulationButtons(&pOutState->buttons, pOutState->analogStickL, true);
    SetCrossStickEmulationButtons(&pOutState->buttons, pOutState->analogStickR, false);
}

inline void ConvertPalmaAxis(AnalogStickState* pState)
{
    auto state = *pState;
    pState->y = - state.y;
}

}}} // namespace nn::hid::detail
