﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <type_traits>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_IHidDebugServer.sfdl.h>
#include <nn/hid/hid_IHidServer.sfdl.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/hid_Xpad.h>
#include <nn/hid/debug/hid_Xpad.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/util/util_TypedStorage.h>

#include "hid_AppletResourceUserId.h"
#include "hid_BasicXpadImpl.h"
#include "hid_HidDebugServer.h"
#include "hid_HidServer.h"
#include "hid_SharedMemoryAccessor.h"
#include "hid_SharedMemoryFormat.h"
#include "hid_StaticObject.h"
#include "hid_XpadId.h"
#include "hid_XpadLifo.h"
#include "hid_XpadSessionManager.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< 基本的な構成を持つ Xpad の共有メモリへのアクセスを扱うクラスです。
class BasicXpadSharedMemoryAccessor final : public SharedMemoryAccessor
{
public:
    //!< Xpad ID の型
    typedef BasicXpadId XpadIdType;

private:
    //!< 基本的な構成を持つ Xpad の ID
    BasicXpadId m_XpadId;

    //!< 基本的な構成を持つ Xpad の LIFO
    ::std::atomic<BasicXpadLifo*> m_pLifo;

public:
    explicit BasicXpadSharedMemoryAccessor(
        const BasicXpadId& xpadId) NN_NOEXCEPT
        : m_XpadId(xpadId)
        , m_pLifo(nullptr)
    {
        this->SetResultActivationUpperLimitOver(
            ::nn::hid::ResultBasicXpadActivationUpperLimitOver());
        this->SetResultDeactivationLowerLimitOver(
            ::nn::hid::ResultBasicXpadDeactivationLowerLimitOver());
    }

    virtual ~BasicXpadSharedMemoryAccessor() NN_NOEXCEPT NN_OVERRIDE
    {
        // 何もしない
    }

    //!< BasicXpad の LIFO を返します。
    const BasicXpadLifo* GetLifo() const NN_NOEXCEPT
    {
        return m_pLifo;
    }

protected:
    virtual ::nn::Result Attach(SharedMemoryFormat* ptr) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(ptr);
        ::nn::sf::SharedPointer<IHidServer> pHidServer;
        NN_RESULT_DO(CreateHidServerProxy(&pHidServer));
        NN_RESULT_DO(
            pHidServer->ActivateXpad(GetAppletResourceUserId(), m_XpadId));
        const auto index = static_cast<size_t>(GetXpadIdPlayerNumber(m_XpadId));
        NN_SDK_ASSERT_LESS(
            index,
            ::std::extent<decltype(ptr->basicXpad.entries)>::value);
        m_pLifo = &::nn::util::Get(ptr->basicXpad.entries[index].lifo);
        NN_RESULT_SUCCESS;
    }

    virtual ::nn::Result Detach() NN_NOEXCEPT NN_OVERRIDE
    {
        ::nn::sf::SharedPointer<IHidDebugServer> pHidDebugServer;
        NN_RESULT_DO(CreateHidDebugServerProxy(&pHidDebugServer));
        NN_RESULT_DO(pHidDebugServer->DeactivateXpad(m_XpadId));
        m_pLifo = nullptr;
        NN_RESULT_SUCCESS;
    }
};

//!< 基本的な構成を持つ Xpad の共有メモリアクセッサを返します。
BasicXpadSharedMemoryAccessor* GetBasicXpadSharedMemoryAccessor(
    const BasicXpadId& xpadId) NN_NOEXCEPT
{
    return StaticObject<
        XpadSessionManager<
            BasicXpadSharedMemoryAccessor>>::Get().GetSession(xpadId);
}

} // namespace

::nn::Result GetXpadIds(int* pOutCount,
                        BasicXpadId outXpadIds[],
                        int count) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pOutCount != nullptr,
                           ResultBasicXpadNullIdCountBuffer());
    NN_RESULT_THROW_UNLESS(count >= 0, ResultBasicXpadInvalidIdCount());
    ::nn::sf::SharedPointer<IHidServer> pHidServer;
    CreateHidServerProxy(&pHidServer);
    auto value = int64_t();
    NN_RESULT_DO(
        pHidServer->GetXpadIds(&value,
                               ::nn::sf::OutArray<BasicXpadId>(
                                   outXpadIds, static_cast<size_t>(count))));
    *pOutCount = static_cast<int>(value);
    NN_RESULT_SUCCESS;
}

::nn::Result GetXpadPlayerNumber(int* pOutValue,
                                 const BasicXpadId& xpadId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    *pOutValue = GetXpadIdPlayerNumber(xpadId);
    NN_RESULT_SUCCESS;
}

::nn::Result InitializeXpad(const BasicXpadId& xpadId) NN_NOEXCEPT
{
    BasicXpadSharedMemoryAccessor* const
        pAccessor = GetBasicXpadSharedMemoryAccessor(xpadId);
    NN_RESULT_THROW_UNLESS(pAccessor != nullptr,
                           ResultBasicXpadPlayerNumberOutOfRange());
    NN_RESULT_DO(pAccessor->SetAppletResourceUserId(GetAppletResourceUserId()));
    NN_RESULT_DO(pAccessor->Activate());
    NN_RESULT_SUCCESS;
}

::nn::Result FinalizeXpad(const BasicXpadId& xpadId) NN_NOEXCEPT
{
    BasicXpadSharedMemoryAccessor* const
        pAccessor = GetBasicXpadSharedMemoryAccessor(xpadId);
    NN_RESULT_THROW_UNLESS(pAccessor != nullptr,
                           ResultBasicXpadPlayerNumberOutOfRange());
    NN_RESULT_DO(pAccessor->Deactivate());
    NN_RESULT_SUCCESS;
}

::nn::Result GetXpadState(BasicXpadState* pOutValue,
                          const BasicXpadId& xpadId) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                           ResultBasicXpadNullStateBuffer());
    BasicXpadSharedMemoryAccessor* const
        pAccessor = GetBasicXpadSharedMemoryAccessor(xpadId);
    NN_RESULT_THROW_UNLESS(pAccessor != nullptr,
                           ResultBasicXpadPlayerNumberOutOfRange());
    const BasicXpadLifo* const pLifo = pAccessor->GetLifo();
    NN_RESULT_THROW_UNLESS(pLifo != nullptr, ResultBasicXpadNotInitialized());
    const int count = pLifo->Read(pOutValue, 1);
    NN_SDK_REQUIRES_EQUAL(count, 1);
    NN_UNUSED(count);
    NN_RESULT_SUCCESS;
}

::nn::Result GetXpadStates(int* pOutCount,
                           BasicXpadState outStates[],
                           int count,
                           const BasicXpadId& xpadId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_RESULT_THROW_UNLESS(outStates != nullptr,
                           ResultBasicXpadNullStateBuffer());
    NN_RESULT_THROW_UNLESS(count >= 0, ResultBasicXpadInvalidStateCount());
    BasicXpadSharedMemoryAccessor* const
        pAccessor = GetBasicXpadSharedMemoryAccessor(xpadId);
    NN_RESULT_THROW_UNLESS(pAccessor != nullptr,
                           ResultBasicXpadPlayerNumberOutOfRange());
    const BasicXpadLifo* const pLifo = pAccessor->GetLifo();
    NN_RESULT_THROW_UNLESS(pLifo != nullptr, ResultBasicXpadNotInitialized());
    NN_SDK_REQUIRES(!pLifo->IsEmpty());
    *pOutCount = pLifo->Read(outStates, count);
    NN_RESULT_SUCCESS;
}

::nn::Result SetXpadAutoPilotState(
    BasicXpadId xpadId,
    const ::nn::hid::debug::BasicXpadAutoPilotState& value) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidDebugServer> pHidDebugServer;
    NN_RESULT_DO(CreateHidDebugServerProxy(&pHidDebugServer));
    NN_RESULT_DO(pHidDebugServer->SetXpadAutoPilotState(xpadId, value));
    NN_RESULT_SUCCESS;
}

::nn::Result UnsetXpadAutoPilotState(BasicXpadId xpadId) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidDebugServer> pHidDebugServer;
    NN_RESULT_DO(CreateHidDebugServerProxy(&pHidDebugServer));
    NN_RESULT_DO(pHidDebugServer->UnsetXpadAutoPilotState(xpadId));
    NN_RESULT_SUCCESS;
}

::nn::Result AssertValidXpadId(const BasicXpadId& xpadId) NN_NOEXCEPT
{
    BasicXpadSharedMemoryAccessor* const
        pAccessor = GetBasicXpadSharedMemoryAccessor(xpadId);
    NN_RESULT_THROW_UNLESS(pAccessor != nullptr,
                           ResultBasicXpadPlayerNumberOutOfRange());
    const BasicXpadLifo* const pLifo = pAccessor->GetLifo();
    NN_RESULT_THROW_UNLESS(pLifo != nullptr, ResultBasicXpadNotInitialized());
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
