﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/hid/system/hid_AnalogStickManualCalibration.h>

namespace nn { namespace hid { namespace detail {

//!< 外周キャリブレーションの状態
struct CircuitCalibrationState
{
    bool isUpdated;               //!< 値を更新したかどうか
    int  count;                   //!< 値を更新した回数
};

//!< アナログスティックのキャリブレーション処理を行うクラスです
class AnalogStickCalibrationManager final
{
    NN_DISALLOW_COPY(AnalogStickCalibrationManager);
    NN_DISALLOW_MOVE(AnalogStickCalibrationManager);

private:
    //!< キャリブレーション中の状況
    system::AnalogStickManualCalibrationStage m_Stage;

    AnalogStickState m_OriginMin;
    AnalogStickState m_OriginMax;
    AnalogStickState m_MinimumStrokePositive;
    AnalogStickState m_MinimumStrokeNegative;
    int m_ReleasePositionThreshold;
    int m_MinimumLength;

    AnalogStickState m_Origin;
    AnalogStickState m_CircuitMin;
    AnalogStickState m_CircuitMax;

    //!< アナログスティックのノイズ成分
    static const int Noise = 10;

    //!< リリースポジションに入ったかどうか
    bool m_EnteredToReleasePosition;

    //!< 前回のアナログスティックの入力状態
    AnalogStickState m_PreviousState;

    //!< 現在のアナログスティックの倒れている方向
    int m_CurrentDirection;

    //!< 仮原点位置が補正済みかどうか
    bool m_IsTemporalOriginSet;

    //!< 原点静止カウント
    int m_OriginStableCount;

    //!< 静止判定時の最少値
    AnalogStickState m_StabilityMin;

    //!< 静止判定時の最大値
    AnalogStickState m_StabilityMax;

    //!< 原点の戻り座標バッファ
    AnalogStickState m_OriginBuffer[4];

    //!< 外周キャリブレーションを行う最大回数
    static const int CircuitCalibrationCount = 3;

    //!< 各方向の外周キャリブレーションの状態
    CircuitCalibrationState m_CircuitCalibrationStates[4];

    //!< 各方向のキャリブレーションの暫定値
    AnalogStickState m_CircuitMinTemp[CircuitCalibrationCount];
    AnalogStickState m_CircuitMaxTemp[CircuitCalibrationCount];

public:
    AnalogStickCalibrationManager() NN_NOEXCEPT;

    ~AnalogStickCalibrationManager() NN_NOEXCEPT;

    //!< AnalogStick 処理を初期化します。
    void Reset() NN_NOEXCEPT;

    ::nn::Result StartAnalogStickManualCalibration(AnalogStickState& originMin,
                                                   AnalogStickState& originMax,
                                                   AnalogStickState& minimumStrokePositive,
                                                   AnalogStickState& minimumStrokeNegative,
                                                   int releasePositionThreshold
                                                   ) NN_NOEXCEPT;

    ::nn::Result RetryCurrentAnalogStickManualCalibration() NN_NOEXCEPT;

    ::nn::Result CancelAnalogStickManualCalibration() NN_NOEXCEPT;

    //!< アナログスティックの値の処理を行います
    Result ProceedAnalogStickCalibration(system::AnalogStickManualCalibrationStage* pOutStage,
                                         AnalogStickState* pOutState,
                                         bool* pOutIsInReleasePosition,
                                         bool* pOutIsInCircumference,
                                         AnalogStickState& state) NN_NOEXCEPT;

    //!< アナログスティックのキャリブレーション処理結果を取得します
    Result GetAnalogStickCalibrationValue(AnalogStickState* pOutOrigin,
                                          AnalogStickState* pOutCircuitMin,
                                          AnalogStickState* pOutCircuitMax) NN_NOEXCEPT;

private:
    void ProceedSetTemporalOrigin(AnalogStickState* pOutState, AnalogStickState& state) NN_NOEXCEPT;
    void HandleCenterPositionCalibrate(bool* pOutIsInReleasePosition, AnalogStickState& state) NN_NOEXCEPT;
    void HandleCircuitCalibrate(bool* pOutIsInCircumference, AnalogStickState& state) NN_NOEXCEPT;
    bool IsStableInPosition(AnalogStickState& state, AnalogStickState& max, AnalogStickState& min) NN_NOEXCEPT;

    void ResetCircuitCalibrationParameters() NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
