﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "hid_AnalogStickCalibrationManager.h"
#include "hid_CommonStateUtility.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< 原点位置に静止されていることを判定するのに必要なサンプル数
const int OriginStableDetectCount = 100;

//!< 原点検査時に静止判定を行う領域のサイズ
const int OriginStableRangeSize = 400;

//!< アナログスティックの値が指定の範囲内に入っているかを評価する
bool IsAnalogStickInRange(AnalogStickState& value, AnalogStickState& max, AnalogStickState& min)
{
    return (value.x < max.x) &&
           (value.x > min.x) &&
           (value.y < max.y) &&
           (value.y > min.y);
}

//!< リリースポジションの範囲を生成する
void GetReleasePositionRange(AnalogStickState* pOutMax,
                             AnalogStickState* pOutMin,
                             AnalogStickState& center,
                             int32_t length,
                             bool isX,
                             bool isPositive,
                             int mergin)
{
    if (isX)
    {
        pOutMax->x = isPositive ? 4096 : (center.x - length);
        pOutMin->x = isPositive ? (center.x + length) : 0;
        pOutMax->y = center.y + mergin;
        pOutMin->y = center.y - mergin;
    }
    else
    {
        pOutMax->x = center.x + mergin;
        pOutMin->x = center.x - mergin;
        pOutMax->y = isPositive ? 4096 : (center.y - length);
        pOutMin->y = isPositive ? (center.y + length) : 0;
    }
}

//!< AnalogStickState の平均をとる
AnalogStickState AverageAnalogStickState(AnalogStickState values[], int count)
{
    int64_t x = 0;
    int64_t y = 0;

    for (int i = 0; i < count; i++)
    {
        x += values[i].x;
        y += values[i].y;
    }

    AnalogStickState state = { static_cast<int32_t>(x / count), static_cast<int32_t>(y / count) };
    return state;
}

//!< AnalogStick の各方向毎の外周キャリブレーションの処理
void TestAnalogStickValuePerDirection(int32_t length,
                                      int32_t minimumStroke,
                                      int32_t* pTemporalBuffer,
                                      CircuitCalibrationState* pState)
{
    NN_SDK_REQUIRES_NOT_NULL(pState);
    if (length > minimumStroke)
    {
        if (length > *pTemporalBuffer)
        {
            *pTemporalBuffer = length;
            pState->isUpdated = true;
        }
    }
    else
    {
        // 最大点に達した
        if (pState->isUpdated == true)
        {
            pState->count++;
            pState->isUpdated = false;
        }
    }
}

} // namespcae

AnalogStickCalibrationManager::AnalogStickCalibrationManager() NN_NOEXCEPT :
    m_Stage(system::AnalogStickManualCalibrationStage_Completed)
{
    // 何もしない
}

AnalogStickCalibrationManager::~AnalogStickCalibrationManager() NN_NOEXCEPT
{
    // 何もしない
}

void AnalogStickCalibrationManager::Reset() NN_NOEXCEPT
{
    m_Stage = system::AnalogStickManualCalibrationStage_Completed;
}

::nn::Result AnalogStickCalibrationManager::StartAnalogStickManualCalibration(AnalogStickState& originMin,
                                                                              AnalogStickState& originMax,
                                                                              AnalogStickState& minimumStrokePositive,
                                                                              AnalogStickState& minimumStrokeNegative,
                                                                              int releasePositionThreshold
                                                                              ) NN_NOEXCEPT
{
    if (m_Stage != system::AnalogStickManualCalibrationStage_Completed)
    {
        NN_RESULT_THROW(ResultAnalogStickCalibrationInProgress());
    }

    m_OriginMin = originMin;
    m_OriginMax = originMax;
    m_CircuitMax = AnalogStickState();
    m_CircuitMin = AnalogStickState();
    m_MinimumStrokePositive = minimumStrokePositive;
    m_MinimumStrokeNegative = minimumStrokeNegative;
    m_ReleasePositionThreshold = releasePositionThreshold;
    m_MinimumLength = std::min(minimumStrokeNegative.x,
                      std::min(minimumStrokeNegative.y,
                      std::min(minimumStrokePositive.x,
                               minimumStrokePositive.y)));
    m_IsTemporalOriginSet = false;
    m_CurrentDirection = 0;
    m_EnteredToReleasePosition = false;
    m_OriginStableCount = 0;

    ResetCircuitCalibrationParameters();

    m_Stage = system::AnalogStickManualCalibrationStage_ReleaseFromRight;

    NN_RESULT_SUCCESS;
}

::nn::Result AnalogStickCalibrationManager::RetryCurrentAnalogStickManualCalibration() NN_NOEXCEPT
{
    switch (m_Stage)
    {
    case system::AnalogStickManualCalibrationStage_ReleaseFromRight:
    case system::AnalogStickManualCalibrationStage_ReleaseFromBottom:
    case system::AnalogStickManualCalibrationStage_ReleaseFromLeft:
    case system::AnalogStickManualCalibrationStage_ReleaseFromTop:
        m_EnteredToReleasePosition = false;
        break;
    case system::AnalogStickManualCalibrationStage_Rotate:
        for (auto& calibrationState : m_CircuitCalibrationStates)
        {
            calibrationState.isUpdated = false;
            calibrationState.count = 0;
        }
        m_MinimumStrokePositive = AnalogStickState();
        m_MinimumStrokeNegative = AnalogStickState();
        break;
    case system::AnalogStickManualCalibrationStage_Completed:
        // 何もしない
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result AnalogStickCalibrationManager::CancelAnalogStickManualCalibration() NN_NOEXCEPT
{
    // 進行中のキャリブレーション処理が存在しない
    if (m_Stage == system::AnalogStickManualCalibrationStage_Completed)
    {
        NN_RESULT_SUCCESS;
    }

    m_Stage = system::AnalogStickManualCalibrationStage_Completed;
    // 初期化処理など必要であれば追加する

    NN_RESULT_SUCCESS;
}

Result AnalogStickCalibrationManager::ProceedAnalogStickCalibration(system::AnalogStickManualCalibrationStage* pOutStage,
                                                                    AnalogStickState* pOutState,
                                                                    bool* pOutIsInReleasePosition,
                                                                    bool* pOutIsInCircumference,
                                                                    AnalogStickState& state) NN_NOEXCEPT
{
    if (m_IsTemporalOriginSet == false)
    {
        ProceedSetTemporalOrigin(pOutState, state);
        *pOutStage = m_Stage;
        NN_RESULT_SUCCESS;
    }

    switch (m_Stage)
    {
    case system::AnalogStickManualCalibrationStage_ReleaseFromRight:
    case system::AnalogStickManualCalibrationStage_ReleaseFromBottom:
    case system::AnalogStickManualCalibrationStage_ReleaseFromLeft:
    case system::AnalogStickManualCalibrationStage_ReleaseFromTop:
        HandleCenterPositionCalibrate(pOutIsInReleasePosition, state);
        break;
    case system::AnalogStickManualCalibrationStage_Rotate:
        HandleCircuitCalibrate(pOutIsInCircumference, state);
        break;
    case system::AnalogStickManualCalibrationStage_Completed:
        // 何もしない
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    *pOutStage = m_Stage;

    pOutState->x = state.x - m_Origin.x;
    pOutState->y = state.y - m_Origin.y;
    AnalogStickState range = { m_MinimumLength * 3 / 2, m_MinimumLength * 3 / 2 };
    *pOutState = ClampAnalogStick(*pOutState,
                                  0,
                                  range,
                                  range);

    NN_RESULT_SUCCESS;
}

Result AnalogStickCalibrationManager::GetAnalogStickCalibrationValue(AnalogStickState* pOutOrigin,
                                                                   AnalogStickState* pOutCircuitMin,
                                                                   AnalogStickState* pOutCircuitMax) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutOrigin);
    NN_SDK_REQUIRES_NOT_NULL(pOutCircuitMin);
    NN_SDK_REQUIRES_NOT_NULL(pOutCircuitMax);

    if (m_Stage != system::AnalogStickManualCalibrationStage_Completed)
    {
        NN_RESULT_THROW(ResultAnalogStickCalibartionNotCompleted());
    }

    *pOutOrigin = m_Origin;
    *pOutCircuitMin = m_CircuitMin;
    *pOutCircuitMax = m_CircuitMax;

    NN_RESULT_SUCCESS;
}

void AnalogStickCalibrationManager::ProceedSetTemporalOrigin(AnalogStickState* pOutState, AnalogStickState& state) NN_NOEXCEPT
{
    // 仮想原点が設定されるまでは、中心にいるように見せる
    *pOutState = AnalogStickState();

    // 仮想原点を取得する
    if (IsStableInPosition(state, m_OriginMax, m_OriginMin))
    {
        m_Origin = state;
        m_IsTemporalOriginSet = true;
        m_OriginStableCount = 0;
    }
}

void AnalogStickCalibrationManager::HandleCenterPositionCalibrate(bool* pOutIsInReleasePosition, AnalogStickState& state) NN_NOEXCEPT
{
    if (m_EnteredToReleasePosition == false)
    {
        // リリースポジションに達するのを待つ
        AnalogStickState releasePositionMax;
        AnalogStickState releasePositionMin;
        switch (m_Stage)
        {
        case system::AnalogStickManualCalibrationStage_ReleaseFromRight:
            GetReleasePositionRange(&releasePositionMax, &releasePositionMin,
                                    m_Origin, m_MinimumStrokePositive.x,
                                    true, true, m_ReleasePositionThreshold);
            break;
        case system::AnalogStickManualCalibrationStage_ReleaseFromBottom:
            GetReleasePositionRange(&releasePositionMax, &releasePositionMin,
                                    m_Origin, m_MinimumStrokeNegative.y,
                                    false, false, m_ReleasePositionThreshold);
            break;
        case system::AnalogStickManualCalibrationStage_ReleaseFromLeft:
            GetReleasePositionRange(&releasePositionMax, &releasePositionMin,
                                    m_Origin, m_MinimumStrokeNegative.x,
                                    true, false, m_ReleasePositionThreshold);
            break;
        case system::AnalogStickManualCalibrationStage_ReleaseFromTop:
            GetReleasePositionRange(&releasePositionMax, &releasePositionMin,
                                    m_Origin, m_MinimumStrokePositive.y,
                                    false, true, m_ReleasePositionThreshold);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        if (IsAnalogStickInRange(state, releasePositionMax, releasePositionMin))
        {
            m_EnteredToReleasePosition = true;
        }
    }
    else
    {
        AnalogStickState max = { m_Origin.x + OriginStableRangeSize / 2, m_Origin.y + OriginStableRangeSize / 2 };
        AnalogStickState min = { m_Origin.x - OriginStableRangeSize / 2, m_Origin.y - OriginStableRangeSize / 2 };

        // 原点に戻ったかどうかを評価する
        if (IsStableInPosition(state, max, min))
        {
            switch (m_Stage)
            {
            case system::AnalogStickManualCalibrationStage_ReleaseFromRight:
                m_OriginBuffer[0] = state;
                m_Stage = system::AnalogStickManualCalibrationStage_ReleaseFromBottom;
                break;
            case system::AnalogStickManualCalibrationStage_ReleaseFromBottom:
                m_OriginBuffer[1] = state;
                m_Stage = system::AnalogStickManualCalibrationStage_ReleaseFromLeft;
                break;
            case system::AnalogStickManualCalibrationStage_ReleaseFromLeft:
                m_OriginBuffer[2] = state;
                m_Stage = system::AnalogStickManualCalibrationStage_ReleaseFromTop;
                break;
            case system::AnalogStickManualCalibrationStage_ReleaseFromTop:
                m_OriginBuffer[3] = state;
                m_Origin = AverageAnalogStickState(m_OriginBuffer, sizeof(m_OriginBuffer) / sizeof(m_OriginBuffer[0]));
                m_Stage = system::AnalogStickManualCalibrationStage_Rotate;
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }

            m_EnteredToReleasePosition = false;
            m_OriginStableCount = 0;
        }
        // 一度 ReleasePosition に入ったら常に Flag は On
        *pOutIsInReleasePosition = true;
    }
}

void AnalogStickCalibrationManager::HandleCircuitCalibrate(bool* pOutIsInCircumference, AnalogStickState& state) NN_NOEXCEPT
{
    auto lengthX = static_cast<int64_t>(state.x - m_Origin.x);
    auto lengthY = static_cast<int64_t>(state.y - m_Origin.y);

    // 右
    if (m_CircuitCalibrationStates[0].count < CircuitCalibrationCount)
    {
        TestAnalogStickValuePerDirection(static_cast<int32_t>(lengthX),
                                         m_MinimumStrokePositive.x,
                                         &m_CircuitMaxTemp[m_CircuitCalibrationStates[0].count].x,
                                         &m_CircuitCalibrationStates[0]);
    }

    // 上
    if (m_CircuitCalibrationStates[1].count < CircuitCalibrationCount)
    {
        TestAnalogStickValuePerDirection(static_cast<int32_t>(lengthY),
                                         m_MinimumStrokePositive.y,
                                         &m_CircuitMaxTemp[m_CircuitCalibrationStates[1].count].y,
                                         &m_CircuitCalibrationStates[1]);
    }

    // 左
    if (m_CircuitCalibrationStates[2].count < CircuitCalibrationCount)
    {
        TestAnalogStickValuePerDirection(static_cast<int32_t>(-lengthX),
                                         m_MinimumStrokeNegative.x,
                                         &m_CircuitMinTemp[m_CircuitCalibrationStates[2].count].x,
                                         &m_CircuitCalibrationStates[2]);
    }

    // 下
    if (m_CircuitCalibrationStates[3].count < CircuitCalibrationCount)
    {
        TestAnalogStickValuePerDirection(static_cast<int32_t>(-lengthY),
                                         m_MinimumStrokeNegative.y,
                                         &m_CircuitMinTemp[m_CircuitCalibrationStates[3].count].y,
                                         &m_CircuitCalibrationStates[3]);
    }

    // 外周にそってまわっているかどうか
    int64_t length = lengthX * lengthX + lengthY * lengthY;
    if (length > static_cast<int64_t>(m_MinimumLength) * m_MinimumLength)
    {
        *pOutIsInCircumference = true;
    }
    else
    {
        // 外周からはずれたら、リセットする
        ResetCircuitCalibrationParameters();
    }

    // 4方向のキャリブレーションが完了したかどうか確認
    bool isCalibrated = true;
    for (auto& calibrationState : m_CircuitCalibrationStates)
    {
        if (calibrationState.count < CircuitCalibrationCount)
        {
            isCalibrated = false;
            break;
        }
    }
    if (isCalibrated)
    {
        for(int i = 0; i < CircuitCalibrationCount; i++)
        {
            m_CircuitMax.x = std::max(m_CircuitMax.x, m_CircuitMaxTemp[i].x);
            m_CircuitMax.y = std::max(m_CircuitMax.y, m_CircuitMaxTemp[i].y);
            m_CircuitMin.x = std::max(m_CircuitMin.x, m_CircuitMinTemp[i].x);
            m_CircuitMin.y = std::max(m_CircuitMin.y, m_CircuitMinTemp[i].y);
        }
        m_Stage = system::AnalogStickManualCalibrationStage_Completed;
    }
}

bool AnalogStickCalibrationManager::IsStableInPosition(AnalogStickState& state, AnalogStickState& max, AnalogStickState& min) NN_NOEXCEPT
{
    // 仮想原点を取得する
    if (IsAnalogStickInRange(state, max, min))
    {
        if (m_OriginStableCount == 0)
        {
            m_StabilityMin = state;
            m_StabilityMax = state;
            m_OriginStableCount++;
        }
        else
        {
            if (m_StabilityMin.x > state.x)
            {
                m_StabilityMin.x = state.x;
            }
            if (m_StabilityMin.y > state.y)
            {
                m_StabilityMin.y = state.y;
            }
            if (m_StabilityMax.x < state.x)
            {
                m_StabilityMin.x = state.x;
            }
            if (m_StabilityMax.y < state.y)
            {
                m_StabilityMin.y = state.y;
            }

            if ((m_StabilityMax.x - m_StabilityMin.x) < Noise &&
                (m_StabilityMax.y - m_StabilityMin.y) < Noise)
            {
                m_OriginStableCount++;
            }
            else
            {
                m_OriginStableCount = 0;
            }

            if (m_OriginStableCount > OriginStableDetectCount)
            {
                m_OriginStableCount = 0;
                return true;
            }
        }
    }
    else
    {
        m_OriginStableCount = 0;
    }

    return false;
}

void AnalogStickCalibrationManager::ResetCircuitCalibrationParameters() NN_NOEXCEPT
{
    for (auto& circuitMinTemp : m_CircuitMinTemp)
    {
        circuitMinTemp = AnalogStickState();
    }
    for (auto& circuitMaxTemp : m_CircuitMaxTemp)
    {
        circuitMaxTemp = AnalogStickState();
    }
    for (auto& calibrationState : m_CircuitCalibrationStates)
    {
        calibrationState.isUpdated = false;
        calibrationState.count = 0;
    }
}

}}} // namespace nn::hid::detail
