﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>

#include "hid_ActivationCount.h"
#include "hid_AbstractedPadUsbDriver.h"
#include "hid_AudioControlDriver.h"
#include "hid_GcAdapterDriver.h"
#include "hid_KeyboardDriver-os.horizon.h"
#include "hid_MouseDriver-os.horizon.h"
#include "hid_SynchronizedTimer.h"

namespace nn { namespace hid { namespace detail {

//!< ahid のポートの数
const size_t AhidPortCount = 8;

//!< ahid の解決を担うマネージャを扱うクラスです。
class AhidManager final
{
    NN_DISALLOW_COPY(AhidManager);
    NN_DISALLOW_MOVE(AhidManager);

public:
    //!< サンプリング間隔
    static const ::nn::TimeSpan SamplingInterval;

    //!< ポートの種類を表す列挙型です。
    enum PortType
    {
        PortType_Control, //!< 制御ポート
        PortType_In,      //!< 受信ポート
        PortType_Out,     //!< 送信ポート
    };

private:
    //!< アクティブ化された回数
    ActivationCount m_ActivationCount;

    //!< タイマーイベント
    SynchronizedTimer* m_pTimerEvent;

    //!< キーボードドライバ
    KeyboardDriver* m_pKeyboardDriver;

    //!< マウスドライバ
    MouseDriver* m_pMouseDriver;

    //!< UsbPadドライバ
    AbstractedPadUsbDriver* m_pAbstractedPadUsbDriver;

    //!< オーディオコントロールドライバ
    AudioControlDriver* m_pAudioControlDriver;

    //!< Gc コントローラーアダプタードライバ
    GcAdapterDriver* m_pGcAdapterDriver;

    //!< ポートの種類
    PortType m_PortType;

    //!< ポート
    size_t m_Port;

public:
    AhidManager() NN_NOEXCEPT;

    //!< タイマーイベントを設定します。
    void SetTimerEvent(SynchronizedTimer* pTimerEvent) NN_NOEXCEPT;

    //!< キーボードドライバを設定します。
    void SetKeyboardDriver(KeyboardDriver* pDriver) NN_NOEXCEPT;

    //!< マウスドライバを設定します。
    void SetMouseDriver(MouseDriver* pDriver) NN_NOEXCEPT;

    //!< UsbPadドライバを設定します。
    void SetAbstractedPadUsbDriver(AbstractedPadUsbDriver* pDriver) NN_NOEXCEPT;

    //!< オーディオコントロールドライバを設定します。
    void SetAudioControlDriver(AudioControlDriver* pDriver) NN_NOEXCEPT;

    //!< Gc コントローラーアダプタードライバーを設定する
    void SetGcAdapterDriver(GcAdapterDriver* pDriver) NN_NOEXCEPT;

    //!< ポートを設定します。
    void SetPort(PortType type, size_t port) NN_NOEXCEPT;

    //!< マネージャをアクティブ化します。
    ::nn::Result Activate() NN_NOEXCEPT;

    //!< マネージャを非アクティブ化します。
    ::nn::Result Deactivate() NN_NOEXCEPT;

    //!< 事前処理を実施します。
    ::nn::Result DoPreProcessing() NN_NOEXCEPT;

    //!< 事後処理を実施します。
    ::nn::Result DoPostProcessing() NN_NOEXCEPT;

    //!< 送受信を実施します。
    ::nn::Result Communicate() NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
